import React, { useState, useEffect } from 'react';
import { Link, useNavigate } from 'react-router-dom';
import { Helmet } from 'react-helmet';
import Header from '../../components/ui/Header';
import Button from '../../components/ui/Button';
import Icon from '../../components/AppIcon';
import SignUpForm from './components/SignUpForm';
import SocialSignUp from './components/SocialSignUp';
import OnboardingProgress from './components/OnboardingProgress';
import UserTypeSelection from './components/UserTypeSelection';
import PlatformSetup from './components/PlatformSetup';
import WelcomeStep from './components/WelcomeStep';

const SignUpPage = () => {
  const navigate = useNavigate();
  const [currentStep, setCurrentStep] = useState(1);
  const [loading, setLoading] = useState(false);
  const [userData, setUserData] = useState(null);
  const [selectedUserType, setSelectedUserType] = useState('');
  const [selectedPlatforms, setSelectedPlatforms] = useState([]);

  // Mock credentials for testing
  const mockCredentials = {
    email: "demo@socialcomposer.com",
    password: "Demo123!",
    firstName: "Demo",
    lastName: "User"
  };

  useEffect(() => {
    // Scroll to top when step changes
    window.scrollTo({ top: 0, behavior: 'smooth' });
  }, [currentStep]);

  const handleFormSubmit = async (formData) => {
    setLoading(true);
    
    try {
      // Simulate API call
      await new Promise(resolve => setTimeout(resolve, 2000));
      
      // Mock validation - check if credentials match
      if (formData?.email === mockCredentials?.email && 
          formData?.password === mockCredentials?.password) {
        setUserData(formData);
        setCurrentStep(2);
      } else {
        alert(`Please use demo credentials:\nEmail: ${mockCredentials?.email}\nPassword: ${mockCredentials?.password}`);
      }
    } catch (error) {
      console.error('Sign up error:', error);
      alert('An error occurred during sign up. Please try again.');
    } finally {
      setLoading(false);
    }
  };

  const handleSocialSignUp = async (provider) => {
    setLoading(true);
    
    try {
      // Simulate social login
      await new Promise(resolve => setTimeout(resolve, 1500));
      
      // Mock successful social login
      setUserData({
        firstName: mockCredentials?.firstName,
        lastName: mockCredentials?.lastName,
        email: mockCredentials?.email,
        provider: provider
      });
      setCurrentStep(2);
    } catch (error) {
      console.error('Social sign up error:', error);
      alert('Social sign up temporarily unavailable. Please use email sign up.');
    } finally {
      setLoading(false);
    }
  };

  const handleUserTypeSelect = (userType) => {
    setSelectedUserType(userType);
  };

  const handleUserTypeNext = () => {
    if (selectedUserType) {
      setCurrentStep(3);
    }
  };

  const handlePlatformSetupComplete = (platforms) => {
    setSelectedPlatforms(platforms);
    setCurrentStep(4);
  };

  const handleStartTour = () => {
    // Navigate to dashboard with tour parameter
    navigate('/dashboard?tour=true');
  };

  const renderStepContent = () => {
    switch (currentStep) {
      case 1:
        return (
          <div className="space-y-8">
            <div className="text-center">
              <h1 className="text-3xl font-bold text-text-primary mb-2">
                Create Your Account
              </h1>
              <p className="text-lg text-text-secondary">
                Join thousands of creators managing their social media with confidence
              </p>
            </div>

            <div className="max-w-md mx-auto">
              <SocialSignUp 
                onSocialSignUp={handleSocialSignUp}
                loading={loading}
              />
              
              <SignUpForm 
                onSubmit={handleFormSubmit}
                loading={loading}
              />
            </div>

            <div className="text-center">
              <p className="text-sm text-text-secondary">
                Already have an account?{' '}
                <Link to="/sign-in" className="text-primary hover:underline font-medium">
                  Sign in here
                </Link>
              </p>
            </div>
          </div>
        );

      case 2:
        return (
          <div className="space-y-8">
            <UserTypeSelection 
              onSelect={handleUserTypeSelect}
              selectedType={selectedUserType}
            />
            
            <div className="flex justify-center">
              <Button
                variant="default"
                size="lg"
                onClick={handleUserTypeNext}
                disabled={!selectedUserType}
                iconName="ArrowRight"
                iconPosition="right"
              >
                Continue
              </Button>
            </div>
          </div>
        );

      case 3:
        return (
          <PlatformSetup 
            onComplete={handlePlatformSetupComplete}
            selectedPlatforms={selectedPlatforms}
            onPlatformChange={setSelectedPlatforms}
          />
        );

      case 4:
        return (
          <WelcomeStep 
            userData={userData}
            onStartTour={handleStartTour}
          />
        );

      default:
        return null;
    }
  };

  return (
    <>
      <Helmet>
        <title>Sign Up - Social Composer Pro</title>
        <meta name="description" content="Create your Social Composer Pro account and start managing your social media with confidence. Join thousands of creators and marketers." />
        <meta name="keywords" content="social media management, content creation, sign up, social composer pro" />
      </Helmet>
      <div className="min-h-screen bg-background">
        <Header />
        
        <main className="pt-16">
          <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
            {currentStep > 1 && (
              <OnboardingProgress 
                currentStep={currentStep} 
                totalSteps={4} 
              />
            )}

            <div className="bg-card rounded-xl creative-confidence-shadow p-8 lg:p-12">
              {renderStepContent()}
            </div>

            {/* Trust Indicators */}
            <div className="mt-12 text-center">
              <div className="flex flex-wrap justify-center items-center gap-8 opacity-60">
                <div className="flex items-center space-x-2">
                  <Icon name="Shield" size={16} color="var(--color-success)" />
                  <span className="text-sm text-text-secondary">SSL Secured</span>
                </div>
                <div className="flex items-center space-x-2">
                  <Icon name="Lock" size={16} color="var(--color-success)" />
                  <span className="text-sm text-text-secondary">GDPR Compliant</span>
                </div>
                <div className="flex items-center space-x-2">
                  <Icon name="CheckCircle" size={16} color="var(--color-success)" />
                  <span className="text-sm text-text-secondary">SOC 2 Certified</span>
                </div>
                <div className="flex items-center space-x-2">
                  <Icon name="Users" size={16} color="var(--color-success)" />
                  <span className="text-sm text-text-secondary">50K+ Users</span>
                </div>
              </div>
            </div>
          </div>
        </main>

        {/* Footer */}
        <footer className="bg-muted py-8 mt-16">
          <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="text-center">
              <p className="text-sm text-text-secondary">
                © {new Date()?.getFullYear()} Social Composer Pro. All rights reserved.
              </p>
              <div className="flex justify-center space-x-6 mt-4">
                <Link to="/privacy" className="text-sm text-text-secondary hover:text-primary">
                  Privacy Policy
                </Link>
                <Link to="/terms" className="text-sm text-text-secondary hover:text-primary">
                  Terms of Service
                </Link>
                <Link to="/support" className="text-sm text-text-secondary hover:text-primary">
                  Support
                </Link>
              </div>
            </div>
          </div>
        </footer>
      </div>
    </>
  );
};

export default SignUpPage;