import React, { useState } from 'react';
import Button from '../../../components/ui/Button';
import { Checkbox } from '../../../components/ui/Checkbox';
import Icon from '../../../components/AppIcon';

const PlatformSetup = ({ onComplete, selectedPlatforms, onPlatformChange }) => {
  const [isConnecting, setIsConnecting] = useState(false);

  const platforms = [
    {
      id: 'facebook',
      name: 'Facebook',
      icon: 'Facebook',
      color: 'text-blue-600',
      description: 'Connect your Facebook pages and profiles',
      connected: false
    },
    {
      id: 'instagram',
      name: 'Instagram',
      icon: 'Instagram',
      color: 'text-pink-600',
      description: 'Manage your Instagram business account',
      connected: false
    },
    {
      id: 'twitter',
      name: 'Twitter',
      icon: 'Twitter',
      color: 'text-sky-500',
      description: 'Schedule tweets and manage engagement',
      connected: false
    },
    {
      id: 'linkedin',
      name: 'LinkedIn',
      icon: 'Linkedin',
      color: 'text-blue-700',
      description: 'Share professional content and updates',
      connected: false
    },
    {
      id: 'youtube',
      name: 'YouTube',
      icon: 'Youtube',
      color: 'text-red-600',
      description: 'Manage video content and community posts',
      connected: false
    },
    {
      id: 'tiktok',
      name: 'TikTok',
      icon: 'Music',
      color: 'text-black',
      description: 'Schedule and manage short-form videos',
      connected: false
    }
  ];

  const handlePlatformToggle = (platformId) => {
    const updatedPlatforms = selectedPlatforms?.includes(platformId)
      ? selectedPlatforms?.filter(id => id !== platformId)
      : [...selectedPlatforms, platformId];
    
    onPlatformChange(updatedPlatforms);
  };

  const handleConnect = async (platformId) => {
    setIsConnecting(true);
    // Simulate connection process
    await new Promise(resolve => setTimeout(resolve, 1500));
    setIsConnecting(false);
    
    // Mock successful connection
    alert(`Successfully connected to ${platforms?.find(p => p?.id === platformId)?.name}!`);
  };

  const handleSkipForNow = () => {
    onComplete(selectedPlatforms);
  };

  const handleContinueWithSelected = () => {
    onComplete(selectedPlatforms);
  };

  return (
    <div className="space-y-6">
      <div className="text-center">
        <h3 className="text-xl font-semibold text-text-primary mb-2">
          Connect Your Social Platforms
        </h3>
        <p className="text-text-secondary">
          Choose the platforms you want to manage with Social Composer Pro
        </p>
      </div>
      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
        {platforms?.map((platform) => (
          <div
            key={platform?.id}
            className={`p-4 rounded-lg border-2 transition-all duration-300 ${
              selectedPlatforms?.includes(platform?.id)
                ? 'border-primary bg-primary/5' :'border-border bg-background hover:border-primary/30'
            }`}
          >
            <div className="flex items-start space-x-4">
              <div className="flex-shrink-0">
                <Checkbox
                  checked={selectedPlatforms?.includes(platform?.id)}
                  onChange={() => handlePlatformToggle(platform?.id)}
                />
              </div>

              <div className="flex-1 min-w-0">
                <div className="flex items-center space-x-3 mb-2">
                  <Icon 
                    name={platform?.icon} 
                    size={24} 
                    className={platform?.color}
                  />
                  <h4 className="text-lg font-medium text-text-primary">
                    {platform?.name}
                  </h4>
                </div>
                
                <p className="text-sm text-text-secondary mb-3">
                  {platform?.description}
                </p>

                {selectedPlatforms?.includes(platform?.id) && (
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => handleConnect(platform?.id)}
                    loading={isConnecting}
                    iconName="Link"
                    iconPosition="left"
                  >
                    Connect Now
                  </Button>
                )}
              </div>
            </div>
          </div>
        ))}
      </div>
      <div className="flex flex-col sm:flex-row gap-4 pt-6">
        <Button
          variant="outline"
          size="lg"
          fullWidth
          onClick={handleSkipForNow}
        >
          Skip for Now
        </Button>
        
        <Button
          variant="default"
          size="lg"
          fullWidth
          onClick={handleContinueWithSelected}
          disabled={selectedPlatforms?.length === 0}
          iconName="ArrowRight"
          iconPosition="right"
        >
          Continue with Selected ({selectedPlatforms?.length})
        </Button>
      </div>
      <div className="text-center">
        <p className="text-xs text-text-secondary">
          You can always connect more platforms later from your dashboard
        </p>
      </div>
    </div>
  );
};

export default PlatformSetup;