import React, { useState, useEffect } from 'react';
import Header from '../../components/ui/Header';
import Icon from '../../components/AppIcon';
import Button from '../../components/ui/Button';
import ResourceCard from './components/ResourceCard';
import FilterSidebar from './components/FilterSidebar';
import SearchBar from './components/SearchBar';
import FeaturedSection from './components/FeaturedSection';
import NewsletterSignup from './components/NewsletterSignup';
import CategoryTabs from './components/CategoryTabs';

const Resources = () => {
  const [searchQuery, setSearchQuery] = useState('');
  const [activeCategory, setActiveCategory] = useState('all');
  const [activeFilters, setActiveFilters] = useState({});
  const [sortBy, setSortBy] = useState('newest');
  const [viewMode, setViewMode] = useState('grid');
  const [isFilterOpen, setIsFilterOpen] = useState(false);

  // Mock data for resources
  const mockResources = [
    {
      id: 1,
      title: "The Complete Guide to Social Media Content Strategy in 2024",
      description: "Master the art of content planning with our comprehensive guide covering platform-specific strategies, content calendars, and engagement optimization techniques.",
      type: "guide",
      category: "Content Strategy",
      image: "https://images.unsplash.com/photo-1611224923853-80b023f02d71?w=400&h=300&fit=crop",
      author: {
        name: "Sarah Chen",
        role: "Content Strategist",
        avatar: "https://randomuser.me/api/portraits/women/32.jpg"
      },
      readTime: "15 min read",
      publishedAt: "2024-10-10",
      downloadable: true,
      tags: ["strategy", "planning", "content", "2024"],
      featured: true
    },
    {
      id: 2,
      title: "50 Proven Social Media Templates for Maximum Engagement",
      description: "Ready-to-use templates for Instagram, LinkedIn, Twitter, and Facebook that have generated millions of impressions for our clients.",
      type: "template",
      category: "Templates",
      image: "https://images.pexels.com/photos/267350/pexels-photo-267350.jpeg?w=400&h=300&fit=crop",
      author: {
        name: "Marcus Rodriguez",
        role: "Creative Director",
        avatar: "https://randomuser.me/api/portraits/men/45.jpg"
      },
      readTime: "Download",
      publishedAt: "2024-10-08",
      downloadable: true,
      tags: ["templates", "engagement", "design", "social"],
      featured: true
    },
    {
      id: 3,
      title: "ROI Calculator: Measure Your Social Media Success",
      description: "Interactive calculator to track your social media return on investment with customizable metrics and automated reporting features.",
      type: "calculator",
      category: "Tools",
      image: "/assets/images/sample.png",
      author: {
        name: "Analytics Team",
        role: "Data Scientists",
        avatar: "https://randomuser.me/api/portraits/women/28.jpg"
      },
      readTime: "Interactive",
      publishedAt: "2024-10-05",
      downloadable: false,
      tags: ["analytics", "roi", "measurement", "tools"],
      featured: true
    },
    {
      id: 4,
      title: "Mastering Video Content: From Script to Viral Success",
      description: "Learn the secrets behind viral video content with insights from top creators and proven frameworks for video storytelling.",
      type: "webinar",
      category: "Video Marketing",
      image: "https://images.unsplash.com/photo-1574717024653-61fd2cf4d44d?w=400&h=300&fit=crop",
      author: {
        name: "David Kim",
        role: "Video Marketing Expert",
        avatar: "https://randomuser.me/api/portraits/men/33.jpg"
      },
      readTime: "45 min",
      duration: "45:32",
      publishedAt: "2024-10-03",
      downloadable: false,
      tags: ["video", "viral", "storytelling", "webinar"]
    },
    {
      id: 5,
      title: "Platform Algorithm Updates: What Changed in Q3 2024",
      description: "Stay ahead of the curve with our quarterly analysis of major algorithm changes across Instagram, TikTok, LinkedIn, and Twitter.",
      type: "blog",
      category: "Industry News",
      image: "https://images.pexels.com/photos/590016/pexels-photo-590016.jpeg?w=400&h=300&fit=crop",
      author: {
        name: "Emma Thompson",
        role: "Social Media Analyst",
        avatar: "https://randomuser.me/api/portraits/women/41.jpg"
      },
      readTime: "8 min read",
      publishedAt: "2024-10-01",
      downloadable: false,
      tags: ["algorithms", "updates", "platforms", "analysis"]
    },
    {
      id: 6,
      title: "Crisis Management: Handling Social Media Disasters",
      description: "A step-by-step playbook for managing social media crises, including response templates and reputation recovery strategies.",
      type: "guide",
      category: "Crisis Management",
      image: "/assets/images/sample.png",
      author: {
        name: "Jennifer Walsh",
        role: "PR Specialist",
        avatar: "https://randomuser.me/api/portraits/women/37.jpg"
      },
      readTime: "12 min read",
      publishedAt: "2024-09-28",
      downloadable: true,
      tags: ["crisis", "management", "reputation", "response"]
    },
    {
      id: 7,
      title: "Behind the Scenes: Building a Million-Follower Brand",
      description: "Exclusive interview with successful brand builders sharing their journey from zero to millions of engaged followers.",
      type: "video",
      category: "Case Studies",
      image: "https://images.unsplash.com/photo-1611605698335-8b1569810432?w=400&h=300&fit=crop",
      author: {
        name: "Michael Brown",
        role: "Brand Strategist",
        avatar: "https://randomuser.me/api/portraits/men/29.jpg"
      },
      readTime: "25 min",
      duration: "25:18",
      publishedAt: "2024-09-25",
      downloadable: false,
      tags: ["branding", "growth", "case-study", "interview"]
    },
    {
      id: 8,
      title: "Hashtag Research Tool: Find Your Perfect Tags",
      description: "Discover trending hashtags, analyze competition, and optimize your reach with our comprehensive hashtag research calculator.",
      type: "calculator",
      category: "Tools",
      image: "https://images.pexels.com/photos/267371/pexels-photo-267371.jpeg?w=400&h=300&fit=crop",
      author: {
        name: "Tech Team",
        role: "Developers",
        avatar: "https://randomuser.me/api/portraits/men/52.jpg"
      },
      readTime: "Interactive",
      publishedAt: "2024-09-22",
      downloadable: false,
      tags: ["hashtags", "research", "optimization", "tools"]
    }
  ];

  // Categories for filtering
  const categories = [
    { value: 'all', label: 'All Resources', count: mockResources?.length },
    { value: 'blog', label: 'Blog Posts', count: mockResources?.filter(r => r?.type === 'blog')?.length },
    { value: 'guide', label: 'Guides', count: mockResources?.filter(r => r?.type === 'guide')?.length },
    { value: 'template', label: 'Templates', count: mockResources?.filter(r => r?.type === 'template')?.length },
    { value: 'webinar', label: 'Webinars', count: mockResources?.filter(r => r?.type === 'webinar')?.length },
    { value: 'calculator', label: 'Tools', count: mockResources?.filter(r => r?.type === 'calculator')?.length },
    { value: 'video', label: 'Videos', count: mockResources?.filter(r => r?.type === 'video')?.length }
  ];

  // Filter options
  const filterOptions = {
    contentType: [
      { value: 'blog', label: 'Blog Posts', count: 15 },
      { value: 'guide', label: 'Guides', count: 12 },
      { value: 'template', label: 'Templates', count: 25 },
      { value: 'webinar', label: 'Webinars', count: 8 },
      { value: 'video', label: 'Videos', count: 18 },
      { value: 'calculator', label: 'Tools', count: 6 }
    ],
    audience: [
      { value: 'beginners', label: 'Beginners', count: 22 },
      { value: 'intermediate', label: 'Intermediate', count: 35 },
      { value: 'advanced', label: 'Advanced', count: 18 },
      { value: 'agencies', label: 'Agencies', count: 12 }
    ],
    topic: [
      { value: 'strategy', label: 'Content Strategy', count: 28 },
      { value: 'analytics', label: 'Analytics', count: 15 },
      { value: 'design', label: 'Design', count: 20 },
      { value: 'video', label: 'Video Marketing', count: 18 },
      { value: 'automation', label: 'Automation', count: 12 }
    ]
  };

  // Search suggestions
  const searchSuggestions = [
    "Content Strategy",
    "Social Media Templates",
    "Analytics Guide",
    "Video Marketing",
    "Instagram Tips",
    "LinkedIn Strategy",
    "Hashtag Research",
    "Crisis Management"
  ];

  // Featured resources (first 3)
  const featuredResources = mockResources?.filter(r => r?.featured)?.slice(0, 3);

  // Filter and search logic
  const filteredResources = mockResources?.filter(resource => {
    // Category filter
    if (activeCategory !== 'all' && resource?.type !== activeCategory) {
      return false;
    }

    // Search filter
    if (searchQuery) {
      const query = searchQuery?.toLowerCase();
      return (resource?.title?.toLowerCase()?.includes(query) ||
      resource?.description?.toLowerCase()?.includes(query) ||
      resource?.category?.toLowerCase()?.includes(query) || resource?.tags?.some(tag => tag?.toLowerCase()?.includes(query)));
    }

    // Additional filters
    for (const [filterType, filterValues] of Object.entries(activeFilters)) {
      if (filterValues?.length > 0) {
        // This would be implemented based on resource properties
        // For now, we'll just return true to show all resources
      }
    }

    return true;
  });

  // Sort resources
  const sortedResources = [...filteredResources]?.sort((a, b) => {
    switch (sortBy) {
      case 'newest':
        return new Date(b.publishedAt) - new Date(a.publishedAt);
      case 'oldest':
        return new Date(a.publishedAt) - new Date(b.publishedAt);
      case 'popular':
        return b?.id - a?.id; // Mock popularity
      case 'title':
        return a?.title?.localeCompare(b?.title);
      default:
        return 0;
    }
  });

  const handleSearch = (query) => {
    setSearchQuery(query);
  };

  const handleCategoryChange = (category) => {
    setActiveCategory(category);
  };

  const handleFilterChange = (category, filters) => {
    setActiveFilters(prev => ({
      ...prev,
      [category]: filters
    }));
  };

  const handleClearFilters = () => {
    setActiveFilters({});
    setActiveCategory('all');
    setSearchQuery('');
  };

  const handleViewResource = (resource) => {
    console.log('Viewing resource:', resource?.title);
    // In a real app, this would navigate to the resource detail page
  };

  const handleDownloadResource = (resource) => {
    console.log('Downloading resource:', resource?.title);
    // In a real app, this would trigger the download
  };

  return (
    <div className="min-h-screen bg-background">
      <Header />
      <main className="pt-16">
        {/* Hero Section */}
        <section className="bg-gradient-to-br from-primary/5 to-secondary/5 py-16">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="text-center max-w-3xl mx-auto">
              <h1 className="text-4xl md:text-5xl font-bold text-text-primary mb-6">
                Resource Center
              </h1>
              <p className="text-xl text-text-secondary mb-8">
                Everything you need to master social media marketing. From beginner guides to advanced strategies, 
                we've got the resources to accelerate your success.
              </p>
              
              {/* Search Bar */}
              <div className="max-w-2xl mx-auto mb-8">
                <SearchBar
                  onSearch={handleSearch}
                  suggestions={searchSuggestions}
                  placeholder="Search guides, templates, tools, and more..."
                />
              </div>

              {/* Quick Stats */}
              <div className="flex flex-wrap justify-center gap-8 text-center">
                <div>
                  <div className="text-2xl font-bold text-primary">100+</div>
                  <div className="text-sm text-text-secondary">Resources</div>
                </div>
                <div>
                  <div className="text-2xl font-bold text-primary">50+</div>
                  <div className="text-sm text-text-secondary">Templates</div>
                </div>
                <div>
                  <div className="text-2xl font-bold text-primary">25+</div>
                  <div className="text-sm text-text-secondary">Video Guides</div>
                </div>
                <div>
                  <div className="text-2xl font-bold text-primary">12K+</div>
                  <div className="text-sm text-text-secondary">Downloads</div>
                </div>
              </div>
            </div>
          </div>
        </section>

        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
          {/* Featured Section */}
          <FeaturedSection
            featuredResources={featuredResources}
            onViewResource={handleViewResource}
          />

          {/* Category Tabs */}
          <CategoryTabs
            categories={categories}
            activeCategory={activeCategory}
            onCategoryChange={handleCategoryChange}
          />

          <div className="flex flex-col lg:flex-row gap-8">
            {/* Sidebar */}
            <aside className="lg:w-80">
              <FilterSidebar
                filters={filterOptions}
                activeFilters={activeFilters}
                onFilterChange={handleFilterChange}
                onClearFilters={handleClearFilters}
                isOpen={isFilterOpen}
                onToggle={() => setIsFilterOpen(!isFilterOpen)}
              />
            </aside>

            {/* Main Content */}
            <div className="flex-1">
              {/* Controls */}
              <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between mb-8 gap-4">
                <div className="flex items-center gap-4">
                  <span className="text-sm text-text-secondary">
                    {sortedResources?.length} resources found
                  </span>
                  {(searchQuery || activeCategory !== 'all' || Object.keys(activeFilters)?.length > 0) && (
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={handleClearFilters}
                      iconName="X"
                      iconPosition="left"
                      className="text-text-secondary hover:text-primary"
                    >
                      Clear filters
                    </Button>
                  )}
                </div>

                <div className="flex items-center gap-4">
                  {/* Sort Dropdown */}
                  <select
                    value={sortBy}
                    onChange={(e) => setSortBy(e?.target?.value)}
                    className="text-sm border border-border rounded-lg px-3 py-2 bg-white text-text-primary focus:ring-2 focus:ring-primary focus:border-primary"
                  >
                    <option value="newest">Newest First</option>
                    <option value="oldest">Oldest First</option>
                    <option value="popular">Most Popular</option>
                    <option value="title">Alphabetical</option>
                  </select>

                  {/* View Mode Toggle */}
                  <div className="flex items-center border border-border rounded-lg overflow-hidden">
                    <button
                      onClick={() => setViewMode('grid')}
                      className={`p-2 ${viewMode === 'grid' ? 'bg-primary text-white' : 'text-text-secondary hover:text-text-primary'}`}
                    >
                      <Icon name="Grid3X3" size={16} />
                    </button>
                    <button
                      onClick={() => setViewMode('list')}
                      className={`p-2 ${viewMode === 'list' ? 'bg-primary text-white' : 'text-text-secondary hover:text-text-primary'}`}
                    >
                      <Icon name="List" size={16} />
                    </button>
                  </div>
                </div>
              </div>

              {/* Resources Grid */}
              {sortedResources?.length > 0 ? (
                <div className={`grid gap-6 mb-12 ${
                  viewMode === 'grid' ?'grid-cols-1 md:grid-cols-2 xl:grid-cols-3' :'grid-cols-1'
                }`}>
                  {sortedResources?.map((resource) => (
                    <ResourceCard
                      key={resource?.id}
                      resource={resource}
                      onView={handleViewResource}
                      onDownload={handleDownloadResource}
                    />
                  ))}
                </div>
              ) : (
                <div className="text-center py-16">
                  <Icon name="Search" size={48} className="text-text-secondary mx-auto mb-4" />
                  <h3 className="text-lg font-semibold text-text-primary mb-2">
                    No resources found
                  </h3>
                  <p className="text-text-secondary mb-6">
                    Try adjusting your search terms or filters to find what you're looking for.
                  </p>
                  <Button variant="outline" onClick={handleClearFilters}>
                    Clear all filters
                  </Button>
                </div>
              )}

              {/* Newsletter Signup */}
              <NewsletterSignup />
            </div>
          </div>
        </div>
      </main>
    </div>
  );
};

export default Resources;