import React from 'react';
import Image from '../../../components/AppImage';
import Icon from '../../../components/AppIcon';
import Button from '../../../components/ui/Button';

const ResourceCard = ({ resource, onView, onDownload }) => {
  const getTypeIcon = (type) => {
    switch (type) {
      case 'blog': return 'FileText';
      case 'guide': return 'BookOpen';
      case 'template': return 'Layout';
      case 'webinar': return 'Video';
      case 'calculator': return 'Calculator';
      case 'video': return 'Play';
      default: return 'File';
    }
  };

  const getTypeColor = (type) => {
    switch (type) {
      case 'blog': return 'text-blue-600 bg-blue-50';
      case 'guide': return 'text-green-600 bg-green-50';
      case 'template': return 'text-purple-600 bg-purple-50';
      case 'webinar': return 'text-red-600 bg-red-50';
      case 'calculator': return 'text-orange-600 bg-orange-50';
      case 'video': return 'text-pink-600 bg-pink-50';
      default: return 'text-gray-600 bg-gray-50';
    }
  };

  return (
    <div className="bg-white rounded-xl creative-confidence-shadow hover:hero-dramatic-shadow smooth-transition border border-border overflow-hidden group">
      <div className="relative overflow-hidden">
        <Image
          src={resource?.image}
          alt={resource?.title}
          className="w-full h-48 object-cover group-hover:scale-105 smooth-transition"
        />
        <div className="absolute top-4 left-4">
          <span className={`inline-flex items-center px-3 py-1 rounded-full text-xs font-medium ${getTypeColor(resource?.type)}`}>
            <Icon name={getTypeIcon(resource?.type)} size={12} className="mr-1" />
            {resource?.type?.charAt(0)?.toUpperCase() + resource?.type?.slice(1)}
          </span>
        </div>
        {resource?.duration && (
          <div className="absolute bottom-4 right-4 bg-black/70 text-white px-2 py-1 rounded text-xs">
            {resource?.duration}
          </div>
        )}
      </div>
      <div className="p-6">
        <div className="flex items-center gap-2 mb-3">
          <span className="text-xs text-text-secondary">{resource?.category}</span>
          <span className="text-xs text-text-secondary">•</span>
          <span className="text-xs text-text-secondary">{resource?.readTime}</span>
        </div>

        <h3 className="text-lg font-semibold text-text-primary mb-2 line-clamp-2 group-hover:text-primary smooth-transition">
          {resource?.title}
        </h3>

        <p className="text-text-secondary text-sm mb-4 line-clamp-3">
          {resource?.description}
        </p>

        <div className="flex items-center justify-between">
          <div className="flex items-center gap-2">
            <Image
              src={resource?.author?.avatar}
              alt={resource?.author?.name}
              className="w-6 h-6 rounded-full"
            />
            <span className="text-xs text-text-secondary">{resource?.author?.name}</span>
          </div>

          <div className="flex items-center gap-2">
            {resource?.downloadable && (
              <Button
                variant="ghost"
                size="sm"
                iconName="Download"
                iconSize={16}
                onClick={() => onDownload(resource)}
                className="text-text-secondary hover:text-primary"
              />
            )}
            <Button
              variant="outline"
              size="sm"
              onClick={() => onView(resource)}
            >
              {resource?.type === 'webinar' || resource?.type === 'video' ? 'Watch' : 'Read'}
            </Button>
          </div>
        </div>

        {resource?.tags && (
          <div className="flex flex-wrap gap-1 mt-4 pt-4 border-t border-border">
            {resource?.tags?.slice(0, 3)?.map((tag, index) => (
              <span
                key={index}
                className="text-xs bg-muted text-text-secondary px-2 py-1 rounded"
              >
                #{tag}
              </span>
            ))}
            {resource?.tags?.length > 3 && (
              <span className="text-xs text-text-secondary">+{resource?.tags?.length - 3} more</span>
            )}
          </div>
        )}
      </div>
    </div>
  );
};

export default ResourceCard;