import React from 'react';
import Icon from '../../../components/AppIcon';
import Button from '../../../components/ui/Button';

const FilterSidebar = ({ 
  filters, 
  activeFilters, 
  onFilterChange, 
  onClearFilters,
  isOpen,
  onToggle 
}) => {
  const handleFilterToggle = (category, value) => {
    const currentFilters = activeFilters?.[category] || [];
    const newFilters = currentFilters?.includes(value)
      ? currentFilters?.filter(f => f !== value)
      : [...currentFilters, value];
    
    onFilterChange(category, newFilters);
  };

  const getActiveFilterCount = () => {
    return Object.values(activeFilters)?.reduce((total, filters) => total + filters?.length, 0);
  };

  return (
    <>
      {/* Mobile Filter Toggle */}
      <div className="lg:hidden mb-6">
        <Button
          variant="outline"
          onClick={onToggle}
          iconName="Filter"
          iconPosition="left"
          className="w-full justify-center"
        >
          Filters {getActiveFilterCount() > 0 && `(${getActiveFilterCount()})`}
        </Button>
      </div>
      {/* Filter Sidebar */}
      <div className={`lg:block ${isOpen ? 'block' : 'hidden'} lg:sticky lg:top-24`}>
        <div className="bg-white rounded-xl creative-confidence-shadow p-6 mb-6">
          <div className="flex items-center justify-between mb-6">
            <h3 className="text-lg font-semibold text-text-primary">Filters</h3>
            {getActiveFilterCount() > 0 && (
              <Button
                variant="ghost"
                size="sm"
                onClick={onClearFilters}
                className="text-text-secondary hover:text-primary"
              >
                Clear All
              </Button>
            )}
          </div>

          <div className="space-y-6">
            {Object.entries(filters)?.map(([category, options]) => (
              <div key={category}>
                <h4 className="text-sm font-medium text-text-primary mb-3 capitalize">
                  {category?.replace(/([A-Z])/g, ' $1')?.trim()}
                </h4>
                <div className="space-y-2">
                  {options?.map((option) => (
                    <label
                      key={option?.value}
                      className="flex items-center gap-3 cursor-pointer group"
                    >
                      <input
                        type="checkbox"
                        checked={(activeFilters?.[category] || [])?.includes(option?.value)}
                        onChange={() => handleFilterToggle(category, option?.value)}
                        className="w-4 h-4 text-primary border-border rounded focus:ring-primary focus:ring-2"
                      />
                      <span className="text-sm text-text-secondary group-hover:text-text-primary flex-1">
                        {option?.label}
                      </span>
                      <span className="text-xs text-text-secondary bg-muted px-2 py-1 rounded">
                        {option?.count}
                      </span>
                    </label>
                  ))}
                </div>
              </div>
            ))}
          </div>
        </div>

        {/* Quick Actions */}
        <div className="bg-gradient-to-br from-primary/5 to-secondary/5 rounded-xl p-6">
          <div className="text-center">
            <Icon name="Bell" size={32} className="text-primary mx-auto mb-3" />
            <h4 className="text-sm font-semibold text-text-primary mb-2">
              Stay Updated
            </h4>
            <p className="text-xs text-text-secondary mb-4">
              Get notified when new resources are published
            </p>
            <Button variant="default" size="sm" fullWidth>
              Subscribe to Updates
            </Button>
          </div>
        </div>
      </div>
    </>
  );
};

export default FilterSidebar;