import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import Image from '../../../components/AppImage';
import Button from '../../../components/ui/Button';

const TestimonialsSection = () => {
  const [activeTestimonial, setActiveTestimonial] = useState(0);

  const testimonials = [
    {
      id: 1,
      name: 'Alexandra Martinez',
      role: 'Social Media Manager',
      company: 'TechFlow Solutions',
      avatar: 'https://images.unsplash.com/photo-1494790108755-2616b612b786?w=150',
      content: `Social Composer Pro transformed our content workflow completely. What used to take our team 6 hours now takes just 2. The AI suggestions are incredibly accurate, and the collaboration features keep everyone aligned. We've seen a 340% increase in engagement since switching.`,
      metrics: {
        engagement: '+340%',timeSaved: '67%',teamEfficiency: '+180%'
      },
      beforeAfter: {
        before: 'Struggled with inconsistent posting and brand voice across 5 platforms',after: 'Unified brand presence with 3x higher engagement and streamlined workflows'
      }
    },
    {
      id: 2,
      name: 'Marcus Chen',role: 'Content Creator',company: 'Independent Creator',avatar: 'https://images.unsplash.com/photo-1507003211169-0a1dd7228f2d?w=150',
      content: `As a solo creator managing multiple clients, Social Composer Pro is a game-changer. The calendar view helps me visualize content gaps, and the analytics show exactly what resonates with each audience. My client retention rate has improved dramatically.`,
      metrics: {
        clientRetention: '+85%',contentOutput: '+250%',revenue: '+120%'
      },
      beforeAfter: {
        before: 'Manually scheduling posts, missing optimal times, inconsistent analytics',after: 'Automated workflows, perfect timing, comprehensive insights across all clients'
      }
    },
    {
      id: 3,
      name: 'Sarah Thompson',role: 'Marketing Director',company: 'GreenTech Innovations',avatar: 'https://images.unsplash.com/photo-1438761681033-6461ffad8d80?w=150',
      content: `The approval workflows in Social Composer Pro ensure brand consistency while speeding up our publishing process. Our team can collaborate seamlessly, and I have complete visibility into our content pipeline. ROI tracking has never been clearer.`,
      metrics: {
        approvalSpeed: '+200%',brandConsistency: '99%',roi: '+156%'
      },
      beforeAfter: {
        before: 'Lengthy approval chains, brand inconsistencies, unclear ROI measurement',after: 'Streamlined approvals, perfect brand alignment, crystal-clear ROI insights'
      }
    }
  ];

  const currentTestimonial = testimonials?.[activeTestimonial];

  const nextTestimonial = () => {
    setActiveTestimonial((prev) => (prev + 1) % testimonials?.length);
  };

  const prevTestimonial = () => {
    setActiveTestimonial((prev) => (prev - 1 + testimonials?.length) % testimonials?.length);
  };

  return (
    <section className="bg-gradient-to-br from-indigo-50 via-white to-purple-50 py-20">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Header */}
        <div className="text-center mb-16">
          <div className="flex items-center justify-center mb-6">
            <div className="bg-gradient-to-r from-primary to-secondary p-3 rounded-2xl">
              <Icon name="Quote" size={24} color="white" />
            </div>
          </div>
          
          <h2 className="text-4xl font-bold text-text-primary mb-6">
            Real Results from Real Users
          </h2>
          <p className="text-xl text-text-secondary max-w-3xl mx-auto">
            Don't just take our word for it. See how Social Composer Pro has transformed content workflows for creators, agencies, and enterprises worldwide.
          </p>
        </div>

        {/* Main Testimonial */}
        <div className="bg-white rounded-2xl creative-confidence-shadow p-8 mb-8">
          <div className="grid lg:grid-cols-2 gap-8 items-center">
            {/* Testimonial Content */}
            <div>
              <div className="flex items-center mb-6">
                <Image
                  src={currentTestimonial?.avatar}
                  alt={currentTestimonial?.name}
                  className="w-16 h-16 rounded-full mr-4"
                />
                <div>
                  <h3 className="text-xl font-bold text-text-primary">
                    {currentTestimonial?.name}
                  </h3>
                  <p className="text-text-secondary">
                    {currentTestimonial?.role} at {currentTestimonial?.company}
                  </p>
                </div>
              </div>
              
              <blockquote className="text-lg text-text-primary leading-relaxed mb-6 italic">
                "{currentTestimonial?.content}"
              </blockquote>
              
              <div className="flex items-center space-x-4">
                <div className="flex text-warning">
                  {[...Array(5)]?.map((_, i) => (
                    <Icon key={i} name="Star" size={16} fill="currentColor" />
                  ))}
                </div>
                <span className="text-sm text-text-secondary">
                  Verified Customer Review
                </span>
              </div>
            </div>
            
            {/* Metrics & Before/After */}
            <div className="space-y-6">
              {/* Key Metrics */}
              <div className="bg-gradient-to-br from-primary/5 to-secondary/5 rounded-xl p-6">
                <h4 className="font-semibold text-text-primary mb-4">Key Results</h4>
                <div className="grid grid-cols-1 gap-4">
                  {Object.entries(currentTestimonial?.metrics)?.map(([key, value]) => (
                    <div key={key} className="flex items-center justify-between">
                      <span className="text-text-secondary capitalize">
                        {key?.replace(/([A-Z])/g, ' $1')?.trim()}
                      </span>
                      <span className="text-2xl font-bold text-primary">{value}</span>
                    </div>
                  ))}
                </div>
              </div>
              
              {/* Before/After */}
              <div className="space-y-4">
                <div className="bg-red-50 border border-red-200 rounded-lg p-4">
                  <div className="flex items-center mb-2">
                    <Icon name="X" size={16} color="var(--color-error)" className="mr-2" />
                    <span className="font-semibold text-error">Before</span>
                  </div>
                  <p className="text-sm text-text-secondary">
                    {currentTestimonial?.beforeAfter?.before}
                  </p>
                </div>
                
                <div className="bg-green-50 border border-green-200 rounded-lg p-4">
                  <div className="flex items-center mb-2">
                    <Icon name="Check" size={16} color="var(--color-success)" className="mr-2" />
                    <span className="font-semibold text-success">After</span>
                  </div>
                  <p className="text-sm text-text-secondary">
                    {currentTestimonial?.beforeAfter?.after}
                  </p>
                </div>
              </div>
            </div>
          </div>
        </div>

        {/* Navigation */}
        <div className="flex items-center justify-between mb-12">
          <div className="flex items-center space-x-4">
            <Button
              variant="outline"
              size="sm"
              onClick={prevTestimonial}
              iconName="ChevronLeft"
              iconPosition="left"
            >
              Previous
            </Button>
            <Button
              variant="outline"
              size="sm"
              onClick={nextTestimonial}
              iconName="ChevronRight"
              iconPosition="right"
            >
              Next
            </Button>
          </div>
          
          <div className="flex space-x-2">
            {testimonials?.map((_, index) => (
              <button
                key={index}
                onClick={() => setActiveTestimonial(index)}
                className={`w-3 h-3 rounded-full micro-transition ${
                  index === activeTestimonial ? 'bg-primary' : 'bg-gray-300'
                }`}
              />
            ))}
          </div>
        </div>

        {/* All Testimonials Grid */}
        <div className="grid md:grid-cols-3 gap-6 mb-12">
          {testimonials?.map((testimonial, index) => (
            <button
              key={testimonial?.id}
              onClick={() => setActiveTestimonial(index)}
              className={`text-left p-6 rounded-xl micro-transition hover:scale-105 ${
                index === activeTestimonial
                  ? 'bg-gradient-to-br from-primary to-secondary text-white creative-confidence-shadow'
                  : 'bg-white hover:bg-muted creative-confidence-shadow'
              }`}
            >
              <div className="flex items-center mb-4">
                <Image
                  src={testimonial?.avatar}
                  alt={testimonial?.name}
                  className="w-12 h-12 rounded-full mr-3"
                />
                <div>
                  <h4 className={`font-semibold ${
                    index === activeTestimonial ? 'text-white' : 'text-text-primary'
                  }`}>
                    {testimonial?.name}
                  </h4>
                  <p className={`text-sm ${
                    index === activeTestimonial ? 'text-white/80' : 'text-text-secondary'
                  }`}>
                    {testimonial?.company}
                  </p>
                </div>
              </div>
              
              <p className={`text-sm line-clamp-3 ${
                index === activeTestimonial ? 'text-white/90' : 'text-text-secondary'
              }`}>
                "{testimonial?.content}"
              </p>
              
              <div className="flex items-center mt-4">
                <div className={`flex ${
                  index === activeTestimonial ? 'text-white/80' : 'text-warning'
                }`}>
                  {[...Array(5)]?.map((_, i) => (
                    <Icon key={i} name="Star" size={12} fill="currentColor" />
                  ))}
                </div>
              </div>
            </button>
          ))}
        </div>

        {/* CTA Section */}
        <div className="text-center bg-gradient-to-r from-primary/10 to-secondary/10 rounded-2xl p-8">
          <h3 className="text-2xl font-bold text-text-primary mb-4">
            Join Thousands of Satisfied Customers
          </h3>
          <p className="text-text-secondary mb-6 max-w-2xl mx-auto">
            Experience the same transformation these customers did. Start your free trial today and see why Social Composer Pro is the #1 choice for content creators worldwide.
          </p>
          
          <div className="flex flex-col sm:flex-row gap-4 justify-center items-center">
            <Button 
              variant="default" 
              size="lg"
              iconName="Rocket"
              iconPosition="left"
              className="px-8"
            >
              Start Your Success Story
            </Button>
            <Button 
              variant="outline" 
              size="lg"
              iconName="MessageCircle"
              iconPosition="left"
              className="px-8"
            >
              Read More Reviews
            </Button>
          </div>
          
          <div className="flex items-center justify-center mt-6 text-sm text-text-secondary">
            <Icon name="Shield" size={16} className="mr-2" />
            <span>Join 10,000+ happy customers • 4.9/5 average rating</span>
          </div>
        </div>
      </div>
    </section>
  );
};

export default TestimonialsSection;