import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import Button from '../../../components/ui/Button';

const ComparisonTable = () => {
  const [activeTab, setActiveTab] = useState('features');

  const competitors = [
    {
      name: 'Social Composer Pro',
      logo: '🚀',
      price: '$29/mo',
      highlight: true
    },
    {
      name: 'Competitor A',
      logo: '📱',
      price: '$39/mo',
      highlight: false
    },
    {
      name: 'Competitor B',
      logo: '📊',
      price: '$49/mo',
      highlight: false
    },
    {
      name: 'Competitor C',
      logo: '🎯',
      price: '$35/mo',
      highlight: false
    }
  ];

  const featureCategories = {
    features: {
      name: 'Core Features',
      items: [
        {
          name: 'AI-Powered Content Creation',
          description: 'Intelligent content suggestions and optimization',
          values: [true, false, true, false]
        },
        {
          name: 'Multi-Platform Publishing',
          description: 'Publish to all major social platforms',
          values: [true, true, true, true]
        },
        {
          name: 'Visual Content Calendar',
          description: 'Drag-and-drop scheduling interface',
          values: [true, true, false, true]
        },
        {
          name: 'Advanced Analytics',
          description: 'Comprehensive performance insights',
          values: [true, true, true, false]
        },
        {
          name: 'Team Collaboration',
          description: 'Real-time collaboration tools',
          values: [true, false, true, false]
        },
        {
          name: 'Custom Workflows',
          description: 'Approval processes and role management',
          values: [true, false, false, false]
        },
        {
          name: 'Brand Kit Integration',
          description: 'Centralized brand assets and guidelines',
          values: [true, false, true, false]
        },
        {
          name: 'API Access',
          description: 'Developer-friendly integrations',
          values: [true, true, false, true]
        }
      ]
    },
    platforms: {
      name: 'Platform Support',
      items: [
        {
          name: 'Instagram',
          description: 'Posts, Stories, Reels',
          values: [true, true, true, true]
        },
        {
          name: 'Twitter/X',
          description: 'Tweets, Threads, Spaces',
          values: [true, true, true, false]
        },
        {
          name: 'LinkedIn',
          description: 'Posts, Articles, Company Pages',
          values: [true, true, false, true]
        },
        {
          name: 'Facebook',
          description: 'Posts, Stories, Events',
          values: [true, true, true, true]
        },
        {
          name: 'TikTok',
          description: 'Video posts and scheduling',
          values: [true, false, true, false]
        },
        {
          name: 'YouTube',
          description: 'Video uploads and optimization',
          values: [true, false, false, false]
        },
        {
          name: 'Pinterest',
          description: 'Pins and board management',
          values: [true, true, false, true]
        }
      ]
    },
    support: {
      name: 'Support & Training',
      items: [
        {
          name: '24/7 Customer Support',
          description: 'Round-the-clock assistance',
          values: [true, false, true, false]
        },
        {
          name: 'Live Chat Support',
          description: 'Instant help when you need it',
          values: [true, true, false, true]
        },
        {
          name: 'Video Tutorials',
          description: 'Comprehensive learning resources',
          values: [true, true, true, false]
        },
        {
          name: 'Onboarding Program',
          description: 'Guided setup and training',
          values: [true, false, false, false]
        },
        {
          name: 'Community Forum',
          description: 'Peer-to-peer support network',
          values: [true, true, false, true]
        },
        {
          name: 'Dedicated Account Manager',
          description: 'Personal success partner',
          values: [true, false, true, false]
        }
      ]
    }
  };

  const tabs = [
    { id: 'features', name: 'Features', icon: 'Layers' },
    { id: 'platforms', name: 'Platforms', icon: 'Globe' },
    { id: 'support', name: 'Support', icon: 'HeadphonesIcon' }
  ];

  const currentCategory = featureCategories?.[activeTab];

  return (
    <section className="bg-gradient-to-br from-gray-50 to-white py-20">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Header */}
        <div className="text-center mb-12">
          <h2 className="text-4xl font-bold text-text-primary mb-6">
            See How We Compare
          </h2>
          <p className="text-xl text-text-secondary max-w-3xl mx-auto">
            Don't just take our word for it. See how Social Composer Pro stacks up against the competition across features, platform support, and customer service.
          </p>
        </div>

        {/* Tab Navigation */}
        <div className="flex justify-center mb-8">
          <div className="flex space-x-1 bg-muted rounded-lg p-1">
            {tabs?.map((tab) => (
              <button
                key={tab?.id}
                onClick={() => setActiveTab(tab?.id)}
                className={`flex items-center px-6 py-3 rounded-md text-sm font-medium micro-transition ${
                  activeTab === tab?.id
                    ? 'bg-white text-primary creative-confidence-shadow'
                    : 'text-text-secondary hover:text-text-primary'
                }`}
              >
                <Icon name={tab?.icon} size={16} className="mr-2" />
                {tab?.name}
              </button>
            ))}
          </div>
        </div>

        {/* Comparison Table */}
        <div className="bg-white rounded-2xl creative-confidence-shadow overflow-hidden">
          {/* Table Header */}
          <div className="bg-gradient-to-r from-primary to-secondary p-6">
            <div className="grid grid-cols-5 gap-4">
              <div className="text-white font-semibold">
                {currentCategory?.name}
              </div>
              {competitors?.map((competitor, index) => (
                <div key={index} className={`text-center ${
                  competitor?.highlight ? 'bg-white/20 rounded-lg p-3' : 'p-3'
                }`}>
                  <div className="text-2xl mb-2">{competitor?.logo}</div>
                  <div className="text-white font-semibold text-sm mb-1">
                    {competitor?.name}
                  </div>
                  <div className="text-white/80 text-xs">
                    {competitor?.price}
                  </div>
                  {competitor?.highlight && (
                    <div className="mt-2">
                      <span className="bg-white text-primary text-xs px-2 py-1 rounded-full font-medium">
                        Recommended
                      </span>
                    </div>
                  )}
                </div>
              ))}
            </div>
          </div>

          {/* Table Body */}
          <div className="divide-y divide-border">
            {currentCategory?.items?.map((item, itemIndex) => (
              <div key={itemIndex} className="grid grid-cols-5 gap-4 p-4 hover:bg-muted/30 micro-transition">
                <div className="flex flex-col justify-center">
                  <div className="font-medium text-text-primary mb-1">
                    {item?.name}
                  </div>
                  <div className="text-sm text-text-secondary">
                    {item?.description}
                  </div>
                </div>
                
                {item?.values?.map((value, valueIndex) => (
                  <div key={valueIndex} className="flex items-center justify-center">
                    {value ? (
                      <div className={`p-2 rounded-full ${
                        competitors?.[valueIndex]?.highlight 
                          ? 'bg-success/10' :'bg-success/5'
                      }`}>
                        <Icon 
                          name="Check" 
                          size={16} 
                          color="var(--color-success)" 
                        />
                      </div>
                    ) : (
                      <div className="p-2 rounded-full bg-gray-100">
                        <Icon 
                          name="X" 
                          size={16} 
                          color="var(--color-text-secondary)" 
                        />
                      </div>
                    )}
                  </div>
                ))}
              </div>
            ))}
          </div>

          {/* Table Footer */}
          <div className="bg-muted p-6">
            <div className="grid grid-cols-5 gap-4">
              <div className="flex items-center">
                <Icon name="Star" size={20} color="var(--color-warning)" className="mr-2" />
                <span className="font-semibold text-text-primary">
                  Start Your Free Trial
                </span>
              </div>
              
              {competitors?.map((competitor, index) => (
                <div key={index} className="text-center">
                  {competitor?.highlight ? (
                    <Button variant="default" size="sm" fullWidth>
                      Try Free
                    </Button>
                  ) : (
                    <Button variant="outline" size="sm" fullWidth disabled>
                      Visit Site
                    </Button>
                  )}
                </div>
              ))}
            </div>
          </div>
        </div>

        {/* Bottom CTA */}
        <div className="text-center mt-12">
          <div className="bg-gradient-to-r from-primary/10 to-secondary/10 rounded-2xl p-8">
            <h3 className="text-2xl font-bold text-text-primary mb-4">
              Ready to Experience the Difference?
            </h3>
            <p className="text-text-secondary mb-6 max-w-2xl mx-auto">
              Join thousands of content creators who've made the switch to Social Composer Pro. Start your free trial today and see why we're the #1 choice for professional social media management.
            </p>
            
            <div className="flex flex-col sm:flex-row gap-4 justify-center items-center">
              <Button 
                variant="default" 
                size="lg"
                iconName="Rocket"
                iconPosition="left"
                className="px-8"
              >
                Start Free Trial
              </Button>
              <Button 
                variant="outline" 
                size="lg"
                iconName="Calendar"
                iconPosition="left"
                className="px-8"
              >
                Schedule Demo
              </Button>
            </div>
            
            <div className="flex items-center justify-center mt-6 text-sm text-text-secondary">
              <Icon name="Shield" size={16} className="mr-2" />
              <span>No credit card required • 14-day free trial • Cancel anytime</span>
            </div>
          </div>
        </div>
      </div>
    </section>
  );
};

export default ComparisonTable;