import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import Image from '../../../components/AppImage';
import Button from '../../../components/ui/Button';

const CollaborationFeature = () => {
  const [activeWorkflow, setActiveWorkflow] = useState('review');

  const teamMembers = [
    {
      id: 1,
      name: 'Sarah Chen',
      role: 'Content Manager',
      avatar: 'https://images.unsplash.com/photo-1494790108755-2616b612b786?w=150',
      status: 'online',
      lastActive: 'now'
    },
    {
      id: 2,
      name: 'Marcus Rodriguez',
      role: 'Designer',
      avatar: 'https://images.unsplash.com/photo-1507003211169-0a1dd7228f2d?w=150',
      status: 'away',
      lastActive: '5m ago'
    },
    {
      id: 3,
      name: 'Emily Johnson',
      role: 'Social Media Specialist',
      avatar: 'https://images.unsplash.com/photo-1438761681033-6461ffad8d80?w=150',
      status: 'online',
      lastActive: 'now'
    },
    {
      id: 4,
      name: 'David Kim',
      role: 'Brand Manager',
      avatar: 'https://images.unsplash.com/photo-1472099645785-5658abf4ff4e?w=150',
      status: 'offline',
      lastActive: '2h ago'
    }
  ];

  const workflowSteps = [
    {
      id: 'draft',
      name: 'Draft',
      icon: 'Edit3',
      status: 'completed',
      assignee: 'Sarah Chen',
      description: 'Initial content creation'
    },
    {
      id: 'review',
      name: 'Review',
      icon: 'Eye',
      status: 'active',
      assignee: 'David Kim',
      description: 'Brand compliance check'
    },
    {
      id: 'approve',
      name: 'Approve',
      icon: 'CheckCircle',
      status: 'pending',
      assignee: 'Emily Johnson',
      description: 'Final approval'
    },
    {
      id: 'schedule',
      name: 'Schedule',
      icon: 'Calendar',
      status: 'pending',
      assignee: 'Marcus Rodriguez',
      description: 'Schedule for publishing'
    }
  ];

  const comments = [
    {
      id: 1,
      author: 'David Kim',
      avatar: 'https://images.unsplash.com/photo-1472099645785-5658abf4ff4e?w=150',
      content: 'The messaging looks great! Can we adjust the CTA to be more action-oriented?',
      timestamp: '2 minutes ago',
      type: 'suggestion'
    },
    {
      id: 2,
      author: 'Sarah Chen',
      avatar: 'https://images.unsplash.com/photo-1494790108755-2616b612b786?w=150',
      content: 'Updated the CTA as requested. How does "Transform Your Content Today" sound?',
      timestamp: '1 minute ago',
      type: 'update'
    },
    {
      id: 3,
      author: 'Emily Johnson',
      avatar: 'https://images.unsplash.com/photo-1438761681033-6461ffad8d80?w=150',
      content: 'Perfect! Ready for final approval. 🚀',
      timestamp: 'Just now',
      type: 'approval'
    }
  ];

  const getStatusColor = (status) => {
    const colors = {
      completed: 'bg-success text-white',
      active: 'bg-primary text-white',
      pending: 'bg-gray-200 text-text-secondary'
    };
    return colors?.[status] || 'bg-gray-200 text-text-secondary';
  };

  const getStatusIcon = (status) => {
    const icons = {
      completed: 'Check',
      active: 'Clock',
      pending: 'Circle'
    };
    return icons?.[status] || 'Circle';
  };

  return (
    <section className="bg-gradient-to-br from-purple-50 via-white to-pink-50 py-20">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="grid lg:grid-cols-2 gap-12 items-center">
          {/* Content */}
          <div>
            <div className="flex items-center mb-6">
              <div className="bg-purple-500/10 p-3 rounded-xl mr-4">
                <Icon name="Users" size={24} color="var(--color-secondary)" />
              </div>
              <span className="text-secondary font-semibold">Team Collaboration</span>
            </div>
            
            <h2 className="text-4xl font-bold text-text-primary mb-6">
              Streamline Your Team's Creative Process
            </h2>
            
            <p className="text-xl text-text-secondary mb-8 leading-relaxed">
              Bring your entire team together in one collaborative workspace. From content creation to approval workflows, manage every step of your social media process with transparency and efficiency.
            </p>
            
            <div className="space-y-6 mb-8">
              <div className="flex items-start">
                <div className="bg-success/10 p-2 rounded-lg mr-4 mt-1">
                  <Icon name="Check" size={16} color="var(--color-success)" />
                </div>
                <div>
                  <h4 className="font-semibold text-text-primary mb-1">
                    Real-time Collaboration
                  </h4>
                  <p className="text-text-secondary">
                    Work together simultaneously with live editing and instant feedback
                  </p>
                </div>
              </div>
              
              <div className="flex items-start">
                <div className="bg-success/10 p-2 rounded-lg mr-4 mt-1">
                  <Icon name="Check" size={16} color="var(--color-success)" />
                </div>
                <div>
                  <h4 className="font-semibold text-text-primary mb-1">
                    Custom Approval Workflows
                  </h4>
                  <p className="text-text-secondary">
                    Set up approval processes that match your team's structure and requirements
                  </p>
                </div>
              </div>
              
              <div className="flex items-start">
                <div className="bg-success/10 p-2 rounded-lg mr-4 mt-1">
                  <Icon name="Check" size={16} color="var(--color-success)" />
                </div>
                <div>
                  <h4 className="font-semibold text-text-primary mb-1">
                    Role-based Permissions
                  </h4>
                  <p className="text-text-secondary">
                    Control who can create, edit, approve, and publish content
                  </p>
                </div>
              </div>
            </div>
            
            <div className="bg-gradient-to-r from-purple-50 to-pink-50 rounded-xl p-6 mb-8">
              <div className="flex items-start">
                <div className="bg-secondary/10 p-2 rounded-lg mr-4">
                  <Icon name="Shield" size={20} color="var(--color-secondary)" />
                </div>
                <div>
                  <h4 className="font-semibold text-text-primary mb-2">
                    Brand Consistency Guaranteed
                  </h4>
                  <p className="text-text-secondary text-sm">
                    Built-in brand guidelines and approval workflows ensure every piece of content maintains your brand standards before going live.
                  </p>
                </div>
              </div>
            </div>
            
            <div className="flex flex-col sm:flex-row gap-4">
              <Button 
                variant="default" 
                size="lg"
                iconName="Users"
                iconPosition="left"
              >
                Invite Your Team
              </Button>
              <Button 
                variant="outline" 
                size="lg"
                iconName="Settings"
                iconPosition="right"
              >
                Setup Workflows
              </Button>
            </div>
          </div>
          
          {/* Interactive Collaboration Demo */}
          <div className="relative">
            <div className="bg-white rounded-2xl creative-confidence-shadow p-6">
              {/* Team Members */}
              <div className="flex items-center justify-between mb-6">
                <h3 className="text-xl font-bold text-text-primary">Team Workspace</h3>
                <div className="flex -space-x-2">
                  {teamMembers?.slice(0, 3)?.map((member) => (
                    <div key={member?.id} className="relative">
                      <Image
                        src={member?.avatar}
                        alt={member?.name}
                        className="w-8 h-8 rounded-full border-2 border-white"
                      />
                      <div className={`absolute -bottom-1 -right-1 w-3 h-3 rounded-full border-2 border-white ${
                        member?.status === 'online' ? 'bg-success' : 
                        member?.status === 'away' ? 'bg-warning' : 'bg-gray-400'
                      }`} />
                    </div>
                  ))}
                  <div className="w-8 h-8 rounded-full bg-muted border-2 border-white flex items-center justify-center">
                    <span className="text-xs font-medium text-text-secondary">+2</span>
                  </div>
                </div>
              </div>
              
              {/* Workflow Steps */}
              <div className="mb-6">
                <h4 className="font-semibold text-text-primary mb-4">Content Approval Workflow</h4>
                <div className="space-y-3">
                  {workflowSteps?.map((step, index) => (
                    <div key={step?.id} className="flex items-center">
                      <div className={`w-8 h-8 rounded-full flex items-center justify-center mr-3 ${getStatusColor(step?.status)}`}>
                        <Icon name={getStatusIcon(step?.status)} size={16} />
                      </div>
                      
                      <div className="flex-1">
                        <div className="flex items-center justify-between">
                          <div>
                            <div className="font-medium text-text-primary">{step?.name}</div>
                            <div className="text-sm text-text-secondary">{step?.description}</div>
                          </div>
                          <div className="text-right">
                            <div className="text-sm font-medium text-text-primary">{step?.assignee}</div>
                            <div className={`text-xs px-2 py-1 rounded-full ${
                              step?.status === 'completed' ? 'bg-success/10 text-success' :
                              step?.status === 'active'? 'bg-primary/10 text-primary' : 'bg-gray-100 text-text-secondary'
                            }`}>
                              {step?.status}
                            </div>
                          </div>
                        </div>
                      </div>
                      
                      {index < workflowSteps?.length - 1 && (
                        <div className="absolute left-4 mt-8 w-0.5 h-6 bg-border" />
                      )}
                    </div>
                  ))}
                </div>
              </div>
              
              {/* Comments Section */}
              <div className="border-t border-border pt-4">
                <div className="flex items-center justify-between mb-4">
                  <h4 className="font-semibold text-text-primary">Recent Activity</h4>
                  <Button variant="ghost" size="sm">
                    <Icon name="MessageCircle" size={16} className="mr-2" />
                    Add Comment
                  </Button>
                </div>
                
                <div className="space-y-4 max-h-48 overflow-y-auto">
                  {comments?.map((comment) => (
                    <div key={comment?.id} className="flex items-start space-x-3">
                      <Image
                        src={comment?.avatar}
                        alt={comment?.author}
                        className="w-8 h-8 rounded-full flex-shrink-0"
                      />
                      <div className="flex-1 min-w-0">
                        <div className="flex items-center space-x-2 mb-1">
                          <span className="font-medium text-text-primary text-sm">
                            {comment?.author}
                          </span>
                          <span className="text-xs text-text-secondary">
                            {comment?.timestamp}
                          </span>
                          <div className={`px-2 py-0.5 text-xs rounded-full ${
                            comment?.type === 'suggestion' ? 'bg-warning/10 text-warning' :
                            comment?.type === 'approval'? 'bg-success/10 text-success' : 'bg-primary/10 text-primary'
                          }`}>
                            {comment?.type}
                          </div>
                        </div>
                        <p className="text-sm text-text-secondary">{comment?.content}</p>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            </div>
            
            {/* Floating Notification */}
            <div className="absolute -top-4 -right-4 bg-white rounded-xl creative-confidence-shadow p-4 border border-primary/20 animate-pulse">
              <div className="flex items-center">
                <div className="bg-primary/10 p-2 rounded-lg mr-3">
                  <Icon name="Bell" size={16} color="var(--color-primary)" />
                </div>
                <div>
                  <div className="text-sm font-semibold text-text-primary">New Comment</div>
                  <div className="text-xs text-text-secondary">Emily just approved</div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </section>
  );
};

export default CollaborationFeature;