import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import Button from '../../../components/ui/Button';

const CalendarFeature = () => {
  const [selectedDate, setSelectedDate] = useState(15);
  const [viewMode, setViewMode] = useState('month');

  const scheduledPosts = [
    {
      id: 1,
      date: 12,
      time: '09:00',
      platform: 'Instagram',
      type: 'image',
      title: 'Morning motivation post',
      status: 'scheduled',
      engagement: '2.4K'
    },
    {
      id: 2,
      date: 15,
      time: '14:30',
      platform: 'Twitter',
      type: 'text',
      title: 'Industry insights thread',
      status: 'draft',
      engagement: 'Pending'
    },
    {
      id: 3,
      date: 18,
      time: '18:00',
      platform: 'LinkedIn',
      type: 'article',
      title: 'Weekly newsletter',
      status: 'scheduled',
      engagement: '1.8K'
    }
  ];

  const calendarDays = Array.from({ length: 31 }, (_, i) => i + 1);

  const getPostsForDate = (date) => {
    return scheduledPosts?.filter(post => post?.date === date);
  };

  const getPlatformColor = (platform) => {
    const colors = {
      Instagram: 'bg-pink-500',
      Twitter: 'bg-blue-500',
      LinkedIn: 'bg-blue-700',
      Facebook: 'bg-blue-600'
    };
    return colors?.[platform] || 'bg-gray-500';
  };

  return (
    <section className="bg-gradient-to-br from-blue-50 via-white to-indigo-50 py-20">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="grid lg:grid-cols-2 gap-12 items-center">
          {/* Interactive Calendar Demo */}
          <div className="order-2 lg:order-1">
            <div className="bg-white rounded-2xl creative-confidence-shadow p-6">
              {/* Calendar Header */}
              <div className="flex items-center justify-between mb-6">
                <div>
                  <h3 className="text-xl font-bold text-text-primary">October 2025</h3>
                  <p className="text-text-secondary">Content Calendar</p>
                </div>
                
                <div className="flex items-center space-x-2">
                  <Button
                    variant={viewMode === 'month' ? 'default' : 'outline'}
                    size="sm"
                    onClick={() => setViewMode('month')}
                  >
                    Month
                  </Button>
                  <Button
                    variant={viewMode === 'week' ? 'default' : 'outline'}
                    size="sm"
                    onClick={() => setViewMode('week')}
                  >
                    Week
                  </Button>
                </div>
              </div>
              
              {/* Calendar Grid */}
              <div className="grid grid-cols-7 gap-1 mb-4">
                {['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat']?.map((day) => (
                  <div key={day} className="p-2 text-center text-sm font-medium text-text-secondary">
                    {day}
                  </div>
                ))}
                
                {calendarDays?.slice(0, 21)?.map((date) => {
                  const posts = getPostsForDate(date);
                  const isSelected = selectedDate === date;
                  
                  return (
                    <button
                      key={date}
                      onClick={() => setSelectedDate(date)}
                      className={`relative p-2 h-12 text-sm rounded-lg micro-transition hover:bg-muted ${
                        isSelected ? 'bg-primary text-white' : 'text-text-primary'
                      }`}
                    >
                      {date}
                      {posts?.length > 0 && (
                        <div className="absolute bottom-1 left-1/2 transform -translate-x-1/2 flex space-x-1">
                          {posts?.slice(0, 3)?.map((post, index) => (
                            <div
                              key={index}
                              className={`w-1.5 h-1.5 rounded-full ${getPlatformColor(post?.platform)}`}
                            />
                          ))}
                        </div>
                      )}
                    </button>
                  );
                })}
              </div>
              
              {/* Selected Date Details */}
              <div className="border-t border-border pt-4">
                <h4 className="font-semibold text-text-primary mb-3">
                  October {selectedDate}, 2025
                </h4>
                
                {getPostsForDate(selectedDate)?.length > 0 ? (
                  <div className="space-y-3">
                    {getPostsForDate(selectedDate)?.map((post) => (
                      <div key={post?.id} className="flex items-center justify-between p-3 bg-muted rounded-lg">
                        <div className="flex items-center">
                          <div className={`w-3 h-3 rounded-full ${getPlatformColor(post?.platform)} mr-3`} />
                          <div>
                            <div className="text-sm font-medium text-text-primary">
                              {post?.title}
                            </div>
                            <div className="text-xs text-text-secondary">
                              {post?.platform} • {post?.time}
                            </div>
                          </div>
                        </div>
                        
                        <div className="flex items-center space-x-2">
                          <span className={`px-2 py-1 text-xs rounded-full ${
                            post?.status === 'scheduled' ?'bg-success/10 text-success' :'bg-warning/10 text-warning'
                          }`}>
                            {post?.status}
                          </span>
                          <Icon name="MoreHorizontal" size={16} color="var(--color-text-secondary)" />
                        </div>
                      </div>
                    ))}
                  </div>
                ) : (
                  <div className="text-center py-8 text-text-secondary">
                    <Icon name="Calendar" size={32} color="var(--color-text-secondary)" className="mx-auto mb-2" />
                    <p>No posts scheduled for this date</p>
                    <Button variant="outline" size="sm" className="mt-2">
                      Add Post
                    </Button>
                  </div>
                )}
              </div>
            </div>
            
            {/* Analytics Preview */}
            <div className="mt-6 bg-white rounded-xl creative-confidence-shadow p-4">
              <div className="flex items-center justify-between mb-3">
                <h4 className="font-semibold text-text-primary">This Week's Performance</h4>
                <Icon name="TrendingUp" size={16} color="var(--color-success)" />
              </div>
              
              <div className="grid grid-cols-3 gap-4">
                <div className="text-center">
                  <div className="text-2xl font-bold text-primary">24</div>
                  <div className="text-xs text-text-secondary">Posts Published</div>
                </div>
                <div className="text-center">
                  <div className="text-2xl font-bold text-success">+18%</div>
                  <div className="text-xs text-text-secondary">Engagement</div>
                </div>
                <div className="text-center">
                  <div className="text-2xl font-bold text-secondary">4.2K</div>
                  <div className="text-xs text-text-secondary">Total Reach</div>
                </div>
              </div>
            </div>
          </div>
          
          {/* Content */}
          <div className="order-1 lg:order-2">
            <div className="flex items-center mb-6">
              <div className="bg-blue-500/10 p-3 rounded-xl mr-4">
                <Icon name="Calendar" size={24} color="var(--color-secondary)" />
              </div>
              <span className="text-secondary font-semibold">Visual Calendar</span>
            </div>
            
            <h2 className="text-4xl font-bold text-text-primary mb-6">
              Plan Your Content Strategy with Precision
            </h2>
            
            <p className="text-xl text-text-secondary mb-8 leading-relaxed">
              Visualize your entire content pipeline with our intuitive calendar interface. Drag, drop, and schedule posts across all platforms while maintaining perfect timing and consistency.
            </p>
            
            <div className="space-y-6 mb-8">
              <div className="flex items-start">
                <div className="bg-success/10 p-2 rounded-lg mr-4 mt-1">
                  <Icon name="Check" size={16} color="var(--color-success)" />
                </div>
                <div>
                  <h4 className="font-semibold text-text-primary mb-1">
                    Drag & Drop Scheduling
                  </h4>
                  <p className="text-text-secondary">
                    Effortlessly move posts between dates and times with visual feedback
                  </p>
                </div>
              </div>
              
              <div className="flex items-start">
                <div className="bg-success/10 p-2 rounded-lg mr-4 mt-1">
                  <Icon name="Check" size={16} color="var(--color-success)" />
                </div>
                <div>
                  <h4 className="font-semibold text-text-primary mb-1">
                    Multi-Platform View
                  </h4>
                  <p className="text-text-secondary">
                    See all your scheduled content across platforms in one unified view
                  </p>
                </div>
              </div>
              
              <div className="flex items-start">
                <div className="bg-success/10 p-2 rounded-lg mr-4 mt-1">
                  <Icon name="Check" size={16} color="var(--color-success)" />
                </div>
                <div>
                  <h4 className="font-semibold text-text-primary mb-1">
                    Smart Recommendations
                  </h4>
                  <p className="text-text-secondary">
                    AI suggests optimal posting times based on your audience engagement
                  </p>
                </div>
              </div>
            </div>
            
            <div className="bg-gradient-to-r from-blue-50 to-indigo-50 rounded-xl p-6 mb-8">
              <div className="flex items-start">
                <div className="bg-blue-500/10 p-2 rounded-lg mr-4">
                  <Icon name="Lightbulb" size={20} color="var(--color-secondary)" />
                </div>
                <div>
                  <h4 className="font-semibold text-text-primary mb-2">
                    Pro Tip: Batch Content Creation
                  </h4>
                  <p className="text-text-secondary text-sm">
                    Use our calendar to identify content gaps and batch create posts for maximum efficiency. Most successful creators plan 2-4 weeks ahead.
                  </p>
                </div>
              </div>
            </div>
            
            <div className="flex flex-col sm:flex-row gap-4">
              <Button 
                variant="default" 
                size="lg"
                iconName="Calendar"
                iconPosition="left"
              >
                Explore Calendar
              </Button>
              <Button 
                variant="outline" 
                size="lg"
                iconName="Download"
                iconPosition="right"
              >
                Download Template
              </Button>
            </div>
          </div>
        </div>
      </div>
    </section>
  );
};

export default CalendarFeature;