import React, { useState } from 'react';
import { BarChart, Bar, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer, PieChart, Pie, Cell } from 'recharts';
import Icon from '../../../components/AppIcon';
import Button from '../../../components/ui/Button';

const AnalyticsFeature = () => {
  const [activeMetric, setActiveMetric] = useState('engagement');
  const [timeRange, setTimeRange] = useState('7d');

  const engagementData = [
    { name: 'Mon', value: 2400, posts: 3 },
    { name: 'Tue', value: 1398, posts: 2 },
    { name: 'Wed', value: 9800, posts: 5 },
    { name: 'Thu', value: 3908, posts: 4 },
    { name: 'Fri', value: 4800, posts: 3 },
    { name: 'Sat', value: 3800, posts: 2 },
    { name: 'Sun', value: 4300, posts: 4 }
  ];

  const platformData = [
    { name: 'Instagram', value: 45, color: '#E4405F' },
    { name: 'Twitter', value: 25, color: '#1DA1F2' },
    { name: 'LinkedIn', value: 20, color: '#0077B5' },
    { name: 'Facebook', value: 10, color: '#1877F2' }
  ];

  const performanceMetrics = [
    {
      id: 'engagement',
      name: 'Engagement Rate',
      value: '4.8%',
      change: '+12%',
      trend: 'up',
      icon: 'Heart'
    },
    {
      id: 'reach',
      name: 'Total Reach',
      value: '24.5K',
      change: '+8%',
      trend: 'up',
      icon: 'Eye'
    },
    {
      id: 'clicks',
      name: 'Link Clicks',
      value: '1.2K',
      change: '-3%',
      trend: 'down',
      icon: 'MousePointer'
    },
    {
      id: 'followers',
      name: 'New Followers',
      value: '342',
      change: '+15%',
      trend: 'up',
      icon: 'UserPlus'
    }
  ];

  const topPosts = [
    {
      id: 1,
      platform: 'Instagram',
      content: 'Behind the scenes of our latest photoshoot ✨',
      engagement: '2.4K',
      reach: '12.8K',
      date: '2 days ago'
    },
    {
      id: 2,
      platform: 'Twitter',
      content: 'Just launched our new feature! What do you think? 🚀',
      engagement: '1.8K',
      reach: '8.5K',
      date: '3 days ago'
    },
    {
      id: 3,
      platform: 'LinkedIn',
      content: 'The future of content creation: 5 trends to watch',
      engagement: '956',
      reach: '5.2K',
      date: '5 days ago'
    }
  ];

  return (
    <section className="bg-white py-20">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="grid lg:grid-cols-2 gap-12 items-center">
          {/* Content */}
          <div>
            <div className="flex items-center mb-6">
              <div className="bg-green-500/10 p-3 rounded-xl mr-4">
                <Icon name="BarChart3" size={24} color="var(--color-success)" />
              </div>
              <span className="text-success font-semibold">Advanced Analytics</span>
            </div>
            
            <h2 className="text-4xl font-bold text-text-primary mb-6">
              Turn Data Into Actionable Insights
            </h2>
            
            <p className="text-xl text-text-secondary mb-8 leading-relaxed">
              Understand what resonates with your audience through comprehensive analytics. Track performance, identify trends, and optimize your content strategy with data-driven decisions.
            </p>
            
            <div className="space-y-6 mb-8">
              <div className="flex items-start">
                <div className="bg-success/10 p-2 rounded-lg mr-4 mt-1">
                  <Icon name="Check" size={16} color="var(--color-success)" />
                </div>
                <div>
                  <h4 className="font-semibold text-text-primary mb-1">
                    Real-time Performance Tracking
                  </h4>
                  <p className="text-text-secondary">
                    Monitor engagement, reach, and conversions as they happen
                  </p>
                </div>
              </div>
              
              <div className="flex items-start">
                <div className="bg-success/10 p-2 rounded-lg mr-4 mt-1">
                  <Icon name="Check" size={16} color="var(--color-success)" />
                </div>
                <div>
                  <h4 className="font-semibold text-text-primary mb-1">
                    Cross-Platform Comparison
                  </h4>
                  <p className="text-text-secondary">
                    Compare performance across all social platforms in one dashboard
                  </p>
                </div>
              </div>
              
              <div className="flex items-start">
                <div className="bg-success/10 p-2 rounded-lg mr-4 mt-1">
                  <Icon name="Check" size={16} color="var(--color-success)" />
                </div>
                <div>
                  <h4 className="font-semibold text-text-primary mb-1">
                    Automated Reports
                  </h4>
                  <p className="text-text-secondary">
                    Weekly and monthly reports delivered to your inbox automatically
                  </p>
                </div>
              </div>
            </div>
            
            <div className="bg-gradient-to-r from-green-50 to-emerald-50 rounded-xl p-6 mb-8">
              <div className="flex items-start">
                <div className="bg-success/10 p-2 rounded-lg mr-4">
                  <Icon name="Target" size={20} color="var(--color-success)" />
                </div>
                <div>
                  <h4 className="font-semibold text-text-primary mb-2">
                    Smart Recommendations
                  </h4>
                  <p className="text-text-secondary text-sm">
                    Our AI analyzes your data to suggest optimal posting times, content types, and hashtags that drive the highest engagement for your specific audience.
                  </p>
                </div>
              </div>
            </div>
            
            <div className="flex flex-col sm:flex-row gap-4">
              <Button 
                variant="default" 
                size="lg"
                iconName="BarChart3"
                iconPosition="left"
              >
                View Full Analytics
              </Button>
              <Button 
                variant="outline" 
                size="lg"
                iconName="Download"
                iconPosition="right"
              >
                Export Report
              </Button>
            </div>
          </div>
          
          {/* Interactive Analytics Dashboard */}
          <div className="relative">
            <div className="bg-gradient-to-br from-gray-50 to-white rounded-2xl creative-confidence-shadow p-6">
              {/* Dashboard Header */}
              <div className="flex items-center justify-between mb-6">
                <div>
                  <h3 className="text-xl font-bold text-text-primary">Analytics Dashboard</h3>
                  <p className="text-text-secondary">Last 7 days</p>
                </div>
                
                <div className="flex items-center space-x-2">
                  <Button
                    variant={timeRange === '7d' ? 'default' : 'outline'}
                    size="sm"
                    onClick={() => setTimeRange('7d')}
                  >
                    7D
                  </Button>
                  <Button
                    variant={timeRange === '30d' ? 'default' : 'outline'}
                    size="sm"
                    onClick={() => setTimeRange('30d')}
                  >
                    30D
                  </Button>
                </div>
              </div>
              
              {/* Key Metrics */}
              <div className="grid grid-cols-2 gap-4 mb-6">
                {performanceMetrics?.map((metric) => (
                  <button
                    key={metric?.id}
                    onClick={() => setActiveMetric(metric?.id)}
                    className={`p-4 rounded-xl text-left micro-transition hover:scale-105 ${
                      activeMetric === metric?.id
                        ? 'bg-gradient-to-br from-primary to-secondary text-white creative-confidence-shadow'
                        : 'bg-white border border-border hover:border-primary/20'
                    }`}
                  >
                    <div className="flex items-center justify-between mb-2">
                      <Icon 
                        name={metric?.icon} 
                        size={20} 
                        color={activeMetric === metric?.id ? 'white' : 'var(--color-primary)'} 
                      />
                      <div className={`flex items-center text-sm ${
                        activeMetric === metric?.id ? 'text-white/80' : 
                        metric?.trend === 'up' ? 'text-success' : 'text-error'
                      }`}>
                        <Icon 
                          name={metric?.trend === 'up' ? 'TrendingUp' : 'TrendingDown'} 
                          size={14} 
                          className="mr-1" 
                        />
                        {metric?.change}
                      </div>
                    </div>
                    <div className={`text-2xl font-bold mb-1 ${
                      activeMetric === metric?.id ? 'text-white' : 'text-text-primary'
                    }`}>
                      {metric?.value}
                    </div>
                    <div className={`text-sm ${
                      activeMetric === metric?.id ? 'text-white/80' : 'text-text-secondary'
                    }`}>
                      {metric?.name}
                    </div>
                  </button>
                ))}
              </div>
              
              {/* Chart */}
              <div className="bg-white rounded-lg border border-border p-4 mb-6">
                <div className="h-48">
                  <ResponsiveContainer width="100%" height="100%">
                    <BarChart data={engagementData}>
                      <CartesianGrid strokeDasharray="3 3" stroke="#f0f0f0" />
                      <XAxis dataKey="name" stroke="#6b7280" fontSize={12} />
                      <YAxis stroke="#6b7280" fontSize={12} />
                      <Tooltip 
                        contentStyle={{ 
                          backgroundColor: 'white', 
                          border: '1px solid #e5e7eb',
                          borderRadius: '8px',
                          boxShadow: '0 4px 6px -1px rgba(0, 0, 0, 0.1)'
                        }}
                      />
                      <Bar dataKey="value" fill="var(--color-primary)" radius={[4, 4, 0, 0]} />
                    </BarChart>
                  </ResponsiveContainer>
                </div>
              </div>
              
              {/* Platform Distribution */}
              <div className="grid grid-cols-2 gap-4">
                <div className="bg-white rounded-lg border border-border p-4">
                  <h4 className="font-semibold text-text-primary mb-3">Platform Distribution</h4>
                  <div className="h-24">
                    <ResponsiveContainer width="100%" height="100%">
                      <PieChart>
                        <Pie
                          data={platformData}
                          cx="50%"
                          cy="50%"
                          innerRadius={20}
                          outerRadius={40}
                          dataKey="value"
                        >
                          {platformData?.map((entry, index) => (
                            <Cell key={`cell-${index}`} fill={entry?.color} />
                          ))}
                        </Pie>
                        <Tooltip />
                      </PieChart>
                    </ResponsiveContainer>
                  </div>
                </div>
                
                <div className="bg-white rounded-lg border border-border p-4">
                  <h4 className="font-semibold text-text-primary mb-3">Top Performing</h4>
                  <div className="space-y-2">
                    {topPosts?.slice(0, 2)?.map((post) => (
                      <div key={post?.id} className="flex items-center">
                        <div className="w-2 h-2 bg-primary rounded-full mr-2" />
                        <div className="flex-1 min-w-0">
                          <div className="text-xs font-medium text-text-primary truncate">
                            {post?.platform}
                          </div>
                          <div className="text-xs text-text-secondary">
                            {post?.engagement} engagements
                          </div>
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
              </div>
            </div>
            
            {/* Floating Insight Card */}
            <div className="absolute -bottom-4 -left-4 bg-white rounded-xl creative-confidence-shadow p-4 border border-success/20">
              <div className="flex items-center">
                <div className="bg-success/10 p-2 rounded-lg mr-3">
                  <Icon name="Zap" size={16} color="var(--color-success)" />
                </div>
                <div>
                  <div className="text-sm font-semibold text-text-primary">Best Time</div>
                  <div className="text-xs text-text-secondary">Wed 2-4 PM</div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </section>
  );
};

export default AnalyticsFeature;