import React, { useState } from 'react';
import { Link } from 'react-router-dom';
import Header from '../../components/ui/Header';
import Button from '../../components/ui/Button';
import Icon from '../../components/AppIcon';
import PricingCard from './components/PricingCard';
import ROICalculator from './components/ROICalculator';
import FeatureComparison from './components/FeatureComparison';
import TestimonialCard from './components/TestimonialCard';
import FAQ from './components/FAQ';

const PricingPage = () => {
  const [billingCycle, setBillingCycle] = useState('monthly');

  const pricingPlans = [
    {
      name: 'Starter',
      description: 'Perfect for individual creators and small businesses getting started with social media',
      monthlyPrice: 29,
      annualPrice: 290,
      savings: 17,
      ctaText: 'Start Free Trial',
      features: [
        'Connect up to 3 social platforms',
        'Schedule up to 100 posts per month',
        'Basic analytics and reporting',
        'Visual content composer',
        '50 AI-generated captions per month',
        '5 brand templates',
        'Email support',
        'Mobile app access'
      ],
      limits: [
        { feature: 'Social Accounts', value: '3' },
        { feature: 'Monthly Posts', value: '100' },
        { feature: 'Team Members', value: '1' },
        { feature: 'AI Captions', value: '50/month' }
      ]
    },
    {
      name: 'Professional',
      description: 'Ideal for growing businesses and marketing teams who need advanced features',
      monthlyPrice: 99,
      annualPrice: 990,
      savings: 17,
      ctaText: 'Start Free Trial',
      features: [
        'Connect up to 8 social platforms',
        'Unlimited post scheduling',
        'Advanced analytics & insights',
        'Team collaboration (5 members)',
        '500 AI-generated captions per month',
        '50 brand templates',
        'Approval workflows',
        'Auto-reposting for evergreen content',
        'Stock photo library access',
        'Basic video editing tools',
        'Priority email support',
        'Custom posting times optimization'
      ],
      limits: [
        { feature: 'Social Accounts', value: '8' },
        { feature: 'Monthly Posts', value: 'Unlimited' },
        { feature: 'Team Members', value: '5' },
        { feature: 'AI Captions', value: '500/month' }
      ]
    },
    {
      name: 'Enterprise',
      description: 'Comprehensive solution for agencies and large organizations with advanced needs',
      monthlyPrice: 199,
      annualPrice: 1990,
      savings: 17,
      ctaText: 'Start Free Trial',
      features: [
        'Connect to all social platforms',
        'Unlimited everything',
        'Advanced analytics & competitor analysis',
        'Unlimited team members',
        'Unlimited AI-generated captions',
        'Unlimited brand templates',
        'Advanced approval workflows',
        'Role-based permissions',
        'Client access portal',
        'White-label reporting',
        'Advanced video editing suite',
        'Custom integrations via API',
        'Dedicated account manager',
        '24/7 phone & chat support'
      ],
      limits: [
        { feature: 'Social Accounts', value: 'Unlimited' },
        { feature: 'Monthly Posts', value: 'Unlimited' },
        { feature: 'Team Members', value: 'Unlimited' },
        { feature: 'AI Captions', value: 'Unlimited' }
      ]
    }
  ];

  const testimonials = [
    {
      quote: "Social Composer Pro transformed our content workflow completely. We're now publishing 3x more content in half the time, and our engagement rates have increased by 150%. The ROI calculator was spot on!",
      rating: 5,
      author: {
        name: "Sarah Chen",
        title: "Marketing Director",
        company: "TechFlow Solutions",
        avatar: "https://images.unsplash.com/photo-1494790108755-2616b612b786?w=150&h=150&fit=crop&crop=face"
      },
      plan: "Professional",
      metrics: [
        { label: "Time Saved", value: "15hrs/week" },
        { label: "Engagement Up", value: "150%" }
      ]
    },
    {
      quote: "As an agency managing 50+ client accounts, the Enterprise plan's white-label reporting and client portals have been game-changers. Our clients love the professional reports, and we've reduced our reporting time by 80%.",
      rating: 5,
      author: {
        name: "Marcus Rodriguez",
        title: "Agency Owner",
        company: "Digital Growth Co.",
        avatar: "https://images.unsplash.com/photo-1472099645785-5658abf4ff4e?w=150&h=150&fit=crop&crop=face"
      },
      plan: "Enterprise",
      metrics: [
        { label: "Clients Managed", value: "50+" },
        { label: "Reporting Time", value: "-80%" }
      ]
    },
    {
      quote: "The AI caption feature alone has saved me hours every week. As a solo creator managing multiple brands, the Starter plan gives me everything I need without breaking the bank. Highly recommended!",
      rating: 5,
      author: {
        name: "Emma Thompson",
        title: "Content Creator",
        company: "Lifestyle Blogger",
        avatar: "https://images.unsplash.com/photo-1438761681033-6461ffad8d80?w=150&h=150&fit=crop&crop=face"
      },
      plan: "Starter",
      metrics: [
        { label: "Time Saved", value: "8hrs/week" },
        { label: "Content Output", value: "+200%" }
      ]
    }
  ];

  const handleSelectPlan = (plan) => {
    // Navigate to sign-up with selected plan
    console.log('Selected plan:', plan?.name);
  };

  return (
    <div className="min-h-screen bg-gray-50">
      <Header />
      {/* Hero Section */}
      <section className="pt-24 pb-16 bg-gradient-to-br from-blue-50 via-white to-purple-50">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-16">
            <h1 className="text-4xl md:text-6xl font-bold text-gray-900 mb-6">
              Simple, Transparent
              <span className="block text-primary">Pricing</span>
            </h1>
            <p className="text-xl text-gray-600 mb-8 max-w-3xl mx-auto">
              Choose the perfect plan for your social media needs. All plans include a 14-day free trial 
              with no credit card required. Upgrade, downgrade, or cancel anytime.
            </p>

            {/* Billing Toggle */}
            <div className="inline-flex items-center bg-white rounded-full p-1 border border-gray-200 mb-12">
              <button
                onClick={() => setBillingCycle('monthly')}
                className={`px-6 py-2 rounded-full text-sm font-medium transition-all ${
                  billingCycle === 'monthly' ?'bg-primary text-white shadow-sm' :'text-gray-600 hover:text-gray-900'
                }`}
              >
                Monthly
              </button>
              <button
                onClick={() => setBillingCycle('annual')}
                className={`px-6 py-2 rounded-full text-sm font-medium transition-all relative ${
                  billingCycle === 'annual' ?'bg-primary text-white shadow-sm' :'text-gray-600 hover:text-gray-900'
                }`}
              >
                Annual
                <span className="absolute -top-2 -right-2 bg-green-500 text-white text-xs px-2 py-1 rounded-full">
                  Save 17%
                </span>
              </button>
            </div>
          </div>

          {/* Pricing Cards */}
          <div className="grid lg:grid-cols-3 gap-8 mb-16">
            {pricingPlans?.map((plan, index) => (
              <PricingCard
                key={index}
                plan={plan}
                isPopular={plan?.name === 'Professional'}
                billingCycle={billingCycle}
                onSelectPlan={handleSelectPlan}
              />
            ))}
          </div>

          {/* Trust Indicators */}
          <div className="text-center">
            <div className="flex flex-wrap justify-center items-center gap-8 text-sm text-gray-600 mb-8">
              <div className="flex items-center">
                <Icon name="Shield" size={16} className="text-green-500 mr-2" />
                14-day free trial
              </div>
              <div className="flex items-center">
                <Icon name="CreditCard" size={16} className="text-green-500 mr-2" />
                No credit card required
              </div>
              <div className="flex items-center">
                <Icon name="RefreshCw" size={16} className="text-green-500 mr-2" />
                Cancel anytime
              </div>
              <div className="flex items-center">
                <Icon name="DollarSign" size={16} className="text-green-500 mr-2" />
                30-day money back guarantee
              </div>
            </div>

            <Button
              variant="default"
              size="lg"
              asChild
              iconName="ArrowRight"
              iconPosition="right"
            >
              <Link to="/sign-up">Start Your Free Trial</Link>
            </Button>
          </div>
        </div>
      </section>
      {/* ROI Calculator Section */}
      <section className="py-16 bg-white">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <ROICalculator />
        </div>
      </section>
      {/* Feature Comparison Section */}
      <section className="py-16 bg-gray-50">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Compare All Features
            </h2>
            <p className="text-xl text-gray-600 max-w-3xl mx-auto">
              See exactly what's included in each plan to make the best choice for your needs
            </p>
          </div>
          <FeatureComparison />
        </div>
      </section>
      {/* Testimonials Section */}
      <section className="py-16 bg-white">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Trusted by Thousands of Creators
            </h2>
            <p className="text-xl text-gray-600 max-w-3xl mx-auto">
              See how Social Composer Pro is helping businesses and creators achieve their goals
            </p>
          </div>

          <div className="grid lg:grid-cols-3 gap-8 mb-12">
            {testimonials?.map((testimonial, index) => (
              <TestimonialCard key={index} testimonial={testimonial} />
            ))}
          </div>

          {/* Customer Logos */}
          <div className="text-center">
            <p className="text-sm text-gray-500 mb-8">Trusted by leading brands and agencies</p>
            <div className="flex flex-wrap justify-center items-center gap-8 opacity-60">
              <div className="text-2xl font-bold text-gray-400">TechFlow</div>
              <div className="text-2xl font-bold text-gray-400">Digital Growth Co.</div>
              <div className="text-2xl font-bold text-gray-400">Creative Studio</div>
              <div className="text-2xl font-bold text-gray-400">Brand Builders</div>
              <div className="text-2xl font-bold text-gray-400">Social Masters</div>
            </div>
          </div>
        </div>
      </section>
      {/* FAQ Section */}
      <section className="py-16 bg-gray-50">
        <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Frequently Asked Questions
            </h2>
            <p className="text-xl text-gray-600">
              Everything you need to know about our pricing and plans
            </p>
          </div>
          <FAQ />
        </div>
      </section>
      {/* Final CTA Section */}
      <section className="py-16 bg-gradient-to-r from-primary to-secondary">
        <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
          <h2 className="text-3xl font-bold text-white mb-4">
            Ready to Transform Your Social Media Strategy?
          </h2>
          <p className="text-xl text-blue-100 mb-8">
            Join thousands of creators and businesses who are already saving time and growing their audience
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Button
              variant="secondary"
              size="lg"
              asChild
              iconName="Play"
              iconPosition="left"
            >
              <Link to="/product-features">Watch Demo</Link>
            </Button>
            <Button
              variant="outline"
              size="lg"
              asChild
              className="bg-white text-primary border-white hover:bg-gray-50"
              iconName="ArrowRight"
              iconPosition="right"
            >
              <Link to="/sign-up">Start Free Trial</Link>
            </Button>
          </div>
          <p className="text-sm text-blue-100 mt-4">
            No credit card required • 14-day free trial • Cancel anytime
          </p>
        </div>
      </section>
      {/* Footer */}
      <footer className="bg-gray-900 text-white py-12">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="grid md:grid-cols-4 gap-8">
            <div>
              <div className="flex items-center space-x-2 mb-4">
                <div className="w-8 h-8 bg-gradient-to-br from-primary to-secondary rounded-lg flex items-center justify-center">
                  <Icon name="Zap" size={20} color="white" strokeWidth={2.5} />
                </div>
                <span className="text-xl font-bold">Social Composer Pro</span>
              </div>
              <p className="text-gray-400 text-sm">
                The complete social media management platform for creators and businesses.
              </p>
            </div>
            
            <div>
              <h4 className="font-semibold mb-4">Product</h4>
              <ul className="space-y-2 text-sm text-gray-400">
                <li><Link to="/product-features" className="hover:text-white transition-colors">Features</Link></li>
                <li><Link to="/pricing" className="hover:text-white transition-colors">Pricing</Link></li>
                <li><Link to="/sign-up" className="hover:text-white transition-colors">Free Trial</Link></li>
              </ul>
            </div>
            
            <div>
              <h4 className="font-semibold mb-4">Resources</h4>
              <ul className="space-y-2 text-sm text-gray-400">
                <li><Link to="/resources" className="hover:text-white transition-colors">Blog</Link></li>
                <li><Link to="/customer-stories" className="hover:text-white transition-colors">Case Studies</Link></li>
                <li><a href="#" className="hover:text-white transition-colors">Help Center</a></li>
              </ul>
            </div>
            
            <div>
              <h4 className="font-semibold mb-4">Company</h4>
              <ul className="space-y-2 text-sm text-gray-400">
                <li><a href="#" className="hover:text-white transition-colors">About</a></li>
                <li><a href="#" className="hover:text-white transition-colors">Contact</a></li>
                <li><a href="#" className="hover:text-white transition-colors">Privacy</a></li>
              </ul>
            </div>
          </div>
          
          <div className="border-t border-gray-800 mt-8 pt-8 text-center text-sm text-gray-400">
            <p>&copy; {new Date()?.getFullYear()} Social Composer Pro. All rights reserved.</p>
          </div>
        </div>
      </footer>
    </div>
  );
};

export default PricingPage;