import React, { useState, useEffect } from 'react';
import Button from '../../../components/ui/Button';
import Input from '../../../components/ui/Input';
import Select from '../../../components/ui/Select';
import Icon from '../../../components/AppIcon';

const ROICalculator = () => {
  const [inputs, setInputs] = useState({
    teamSize: 1,
    postsPerWeek: 10,
    platformCount: 3,
    currentTool: 'manual',
    hourlyRate: 25
  });

  const [results, setResults] = useState({
    timeSaved: 0,
    costSaved: 0,
    roiPercentage: 0,
    paybackPeriod: 0
  });

  const [isCalculated, setIsCalculated] = useState(false);

  const toolOptions = [
    { value: 'manual', label: 'Manual posting (no tools)' },
    { value: 'basic', label: 'Basic scheduling tool' },
    { value: 'hootsuite', label: 'Hootsuite' },
    { value: 'buffer', label: 'Buffer' },
    { value: 'sprout', label: 'Sprout Social' },
    { value: 'other', label: 'Other tool' }
  ];

  const calculateROI = () => {
    // Time calculation based on current workflow
    const timePerPost = inputs?.currentTool === 'manual' ? 15 : 8; // minutes
    const currentTimePerWeek = (inputs?.postsPerWeek * timePerPost * inputs?.platformCount) / 60; // hours
    
    // Time with Social Composer Pro (estimated 70% time savings)
    const newTimePerWeek = currentTimePerWeek * 0.3;
    const timeSavedPerWeek = currentTimePerWeek - newTimePerWeek;
    const timeSavedPerMonth = timeSavedPerWeek * 4.33;
    
    // Cost calculations
    const costSavedPerMonth = timeSavedPerMonth * inputs?.hourlyRate;
    const toolCostPerMonth = inputs?.teamSize <= 1 ? 29 : inputs?.teamSize <= 5 ? 99 : 199;
    const netSavingsPerMonth = costSavedPerMonth - toolCostPerMonth;
    
    // ROI calculations
    const roiPercentage = ((netSavingsPerMonth * 12) / (toolCostPerMonth * 12)) * 100;
    const paybackPeriod = toolCostPerMonth / costSavedPerMonth;

    setResults({
      timeSaved: Math.round(timeSavedPerMonth),
      costSaved: Math.round(costSavedPerMonth),
      roiPercentage: Math.round(roiPercentage),
      paybackPeriod: Math.max(0.1, paybackPeriod)
    });

    setIsCalculated(true);
  };

  const handleInputChange = (field, value) => {
    setInputs(prev => ({
      ...prev,
      [field]: value
    }));
    setIsCalculated(false);
  };

  return (
    <div className="bg-gradient-to-br from-blue-50 to-indigo-50 rounded-2xl p-8">
      <div className="text-center mb-8">
        <Icon name="Calculator" size={48} className="text-primary mx-auto mb-4" />
        <h3 className="text-2xl font-bold text-gray-900 mb-2">ROI Calculator</h3>
        <p className="text-gray-600">
          See how much time and money you could save with Social Composer Pro
        </p>
      </div>
      <div className="grid md:grid-cols-2 gap-8">
        {/* Input Section */}
        <div className="space-y-6">
          <h4 className="font-semibold text-gray-900 mb-4">Tell us about your workflow:</h4>
          
          <Input
            label="Team size"
            type="number"
            min="1"
            max="50"
            value={inputs?.teamSize}
            onChange={(e) => handleInputChange('teamSize', parseInt(e?.target?.value) || 1)}
            description="How many people manage social media?"
          />

          <Input
            label="Posts per week"
            type="number"
            min="1"
            max="100"
            value={inputs?.postsPerWeek}
            onChange={(e) => handleInputChange('postsPerWeek', parseInt(e?.target?.value) || 1)}
            description="Across all platforms"
          />

          <Input
            label="Number of platforms"
            type="number"
            min="1"
            max="10"
            value={inputs?.platformCount}
            onChange={(e) => handleInputChange('platformCount', parseInt(e?.target?.value) || 1)}
            description="Instagram, Facebook, Twitter, etc."
          />

          <Select
            label="Current workflow"
            options={toolOptions}
            value={inputs?.currentTool}
            onChange={(value) => handleInputChange('currentTool', value)}
            description="How do you currently manage social media?"
          />

          <Input
            label="Hourly rate ($)"
            type="number"
            min="10"
            max="200"
            value={inputs?.hourlyRate}
            onChange={(e) => handleInputChange('hourlyRate', parseInt(e?.target?.value) || 25)}
            description="Average hourly cost for social media work"
          />

          <Button
            variant="default"
            size="lg"
            fullWidth
            onClick={calculateROI}
            iconName="Calculator"
            iconPosition="left"
          >
            Calculate My ROI
          </Button>
        </div>

        {/* Results Section */}
        <div className="bg-white rounded-xl p-6">
          <h4 className="font-semibold text-gray-900 mb-6">Your Potential Savings:</h4>
          
          {isCalculated ? (
            <div className="space-y-6">
              <div className="text-center p-4 bg-green-50 rounded-lg">
                <div className="text-3xl font-bold text-green-600 mb-1">
                  {results?.timeSaved} hours
                </div>
                <div className="text-sm text-green-700">saved per month</div>
              </div>

              <div className="grid grid-cols-2 gap-4">
                <div className="text-center p-4 bg-blue-50 rounded-lg">
                  <div className="text-2xl font-bold text-blue-600 mb-1">
                    ${results?.costSaved}
                  </div>
                  <div className="text-xs text-blue-700">monthly savings</div>
                </div>

                <div className="text-center p-4 bg-purple-50 rounded-lg">
                  <div className="text-2xl font-bold text-purple-600 mb-1">
                    {results?.roiPercentage}%
                  </div>
                  <div className="text-xs text-purple-700">annual ROI</div>
                </div>
              </div>

              <div className="text-center p-4 bg-amber-50 rounded-lg">
                <div className="text-lg font-bold text-amber-600 mb-1">
                  {results?.paybackPeriod?.toFixed(1)} months
                </div>
                <div className="text-sm text-amber-700">payback period</div>
              </div>

              <div className="pt-4 border-t border-gray-200">
                <Button
                  variant="default"
                  size="lg"
                  fullWidth
                  iconName="ArrowRight"
                  iconPosition="right"
                >
                  Start Free Trial
                </Button>
                <p className="text-xs text-gray-500 text-center mt-2">
                  No credit card required • 14-day free trial
                </p>
              </div>
            </div>
          ) : (
            <div className="text-center py-12">
              <Icon name="BarChart3" size={64} className="text-gray-300 mx-auto mb-4" />
              <p className="text-gray-500">
                Fill in your details and click "Calculate My ROI" to see your potential savings
              </p>
            </div>
          )}
        </div>
      </div>
    </div>
  );
};

export default ROICalculator;