import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import Button from '../../../components/ui/Button';

const FeatureComparison = () => {
  const [expandedCategory, setExpandedCategory] = useState(null);

  const plans = [
    {
      name: 'Starter',
      price: '$29',
      popular: false
    },
    {
      name: 'Professional',
      price: '$99',
      popular: true
    },
    {
      name: 'Enterprise',
      price: '$199',
      popular: false
    }
  ];

  const featureCategories = [
    {
      name: 'Content Creation',
      features: [
        {
          name: 'Visual Content Composer',
          starter: true,
          professional: true,
          enterprise: true,
          description: 'Drag-and-drop interface for creating engaging posts'
        },
        {
          name: 'AI-Powered Captions',
          starter: '50/month',
          professional: '500/month',
          enterprise: 'Unlimited',
          description: 'Generate compelling captions using AI technology'
        },
        {
          name: 'Brand Templates',
          starter: '5 templates',
          professional: '50 templates',
          enterprise: 'Unlimited',
          description: 'Pre-designed templates matching your brand'
        },
        {
          name: 'Stock Photo Library',
          starter: false,
          professional: true,
          enterprise: true,
          description: 'Access to millions of high-quality stock photos'
        },
        {
          name: 'Video Editing Tools',
          starter: false,
          professional: 'Basic',
          enterprise: 'Advanced',
          description: 'Built-in video editing and enhancement tools'
        }
      ]
    },
    {
      name: 'Publishing & Scheduling',
      features: [
        {
          name: 'Multi-Platform Publishing',
          starter: '3 platforms',
          professional: '8 platforms',
          enterprise: 'All platforms',
          description: 'Publish to Instagram, Facebook, Twitter, LinkedIn, and more'
        },
        {
          name: 'Advanced Scheduling',
          starter: true,
          professional: true,
          enterprise: true,
          description: 'Schedule posts for optimal engagement times'
        },
        {
          name: 'Bulk Upload',
          starter: '10 posts',
          professional: '100 posts',
          enterprise: 'Unlimited',
          description: 'Upload and schedule multiple posts at once'
        },
        {
          name: 'Auto-Reposting',
          starter: false,
          professional: true,
          enterprise: true,
          description: 'Automatically repost evergreen content'
        },
        {
          name: 'Time Zone Optimization',
          starter: false,
          professional: true,
          enterprise: true,
          description: 'Optimize posting times for different time zones'
        }
      ]
    },
    {
      name: 'Analytics & Insights',
      features: [
        {
          name: 'Basic Analytics',
          starter: true,
          professional: true,
          enterprise: true,
          description: 'Track likes, comments, shares, and reach'
        },
        {
          name: 'Advanced Reporting',
          starter: false,
          professional: true,
          enterprise: true,
          description: 'Detailed performance reports and insights'
        },
        {
          name: 'Competitor Analysis',
          starter: false,
          professional: false,
          enterprise: true,
          description: 'Monitor and analyze competitor performance'
        },
        {
          name: 'Custom Dashboards',
          starter: false,
          professional: false,
          enterprise: true,
          description: 'Create personalized analytics dashboards'
        },
        {
          name: 'White-Label Reports',
          starter: false,
          professional: false,
          enterprise: true,
          description: 'Branded reports for client presentations'
        }
      ]
    },
    {
      name: 'Team Collaboration',
      features: [
        {
          name: 'Team Members',
          starter: '1 user',
          professional: '5 users',
          enterprise: 'Unlimited',
          description: 'Collaborate with team members on content creation'
        },
        {
          name: 'Approval Workflows',
          starter: false,
          professional: true,
          enterprise: true,
          description: 'Set up approval processes for content review'
        },
        {
          name: 'Role-Based Permissions',
          starter: false,
          professional: 'Basic',
          enterprise: 'Advanced',
          description: 'Control what team members can access and edit'
        },
        {
          name: 'Client Access Portal',
          starter: false,
          professional: false,
          enterprise: true,
          description: 'Give clients view-only access to their campaigns'
        },
        {
          name: 'Team Activity Logs',
          starter: false,
          professional: false,
          enterprise: true,
          description: 'Track all team member actions and changes'
        }
      ]
    }
  ];

  const toggleCategory = (categoryIndex) => {
    setExpandedCategory(expandedCategory === categoryIndex ? null : categoryIndex);
  };

  const renderFeatureValue = (value) => {
    if (value === true) {
      return <Icon name="Check" size={20} className="text-green-500" />;
    }
    if (value === false) {
      return <Icon name="X" size={20} className="text-gray-300" />;
    }
    return <span className="text-sm font-medium text-gray-700">{value}</span>;
  };

  return (
    <div className="bg-white rounded-2xl border border-gray-200 overflow-hidden">
      <div className="bg-gray-50 px-8 py-6 border-b border-gray-200">
        <h3 className="text-2xl font-bold text-gray-900 mb-2">Feature Comparison</h3>
        <p className="text-gray-600">
          Compare all features across our plans to find the perfect fit for your needs
        </p>
      </div>
      {/* Plan Headers */}
      <div className="grid grid-cols-4 gap-4 p-6 border-b border-gray-200 bg-gray-50">
        <div className="font-semibold text-gray-900">Features</div>
        {plans?.map((plan, index) => (
          <div key={index} className="text-center">
            <div className={`font-bold text-lg ${plan?.popular ? 'text-primary' : 'text-gray-900'}`}>
              {plan?.name}
            </div>
            <div className="text-2xl font-bold text-gray-900 mt-1">{plan?.price}</div>
            {plan?.popular && (
              <div className="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-primary text-white mt-2">
                Most Popular
              </div>
            )}
          </div>
        ))}
      </div>
      {/* Feature Categories */}
      <div className="divide-y divide-gray-200">
        {featureCategories?.map((category, categoryIndex) => (
          <div key={categoryIndex}>
            {/* Category Header */}
            <button
              onClick={() => toggleCategory(categoryIndex)}
              className="w-full px-6 py-4 flex items-center justify-between hover:bg-gray-50 transition-colors"
            >
              <h4 className="font-semibold text-gray-900 text-left">{category?.name}</h4>
              <Icon 
                name={expandedCategory === categoryIndex ? "ChevronUp" : "ChevronDown"} 
                size={20} 
                className="text-gray-500" 
              />
            </button>

            {/* Category Features */}
            {expandedCategory === categoryIndex && (
              <div className="bg-gray-50">
                {category?.features?.map((feature, featureIndex) => (
                  <div key={featureIndex} className="grid grid-cols-4 gap-4 px-6 py-4 border-t border-gray-200">
                    <div className="flex flex-col">
                      <span className="font-medium text-gray-900">{feature?.name}</span>
                      <span className="text-sm text-gray-500 mt-1">{feature?.description}</span>
                    </div>
                    <div className="flex justify-center items-center">
                      {renderFeatureValue(feature?.starter)}
                    </div>
                    <div className="flex justify-center items-center">
                      {renderFeatureValue(feature?.professional)}
                    </div>
                    <div className="flex justify-center items-center">
                      {renderFeatureValue(feature?.enterprise)}
                    </div>
                  </div>
                ))}
              </div>
            )}
          </div>
        ))}
      </div>
      {/* CTA Section */}
      <div className="bg-gray-50 px-6 py-8 border-t border-gray-200">
        <div className="grid grid-cols-4 gap-4">
          <div className="flex items-center">
            <span className="font-semibold text-gray-900">Ready to get started?</span>
          </div>
          {plans?.map((plan, index) => (
            <div key={index} className="text-center">
              <Button
                variant={plan?.popular ? "default" : "outline"}
                size="sm"
                fullWidth
              >
                Choose {plan?.name}
              </Button>
            </div>
          ))}
        </div>
      </div>
    </div>
  );
};

export default FeatureComparison;