import React, { useState, useEffect } from 'react';
import Icon from '../../../components/AppIcon';
import Image from '../../../components/AppImage';

const SocialProofCarousel = () => {
  const [currentSlide, setCurrentSlide] = useState(0);
  const [isAutoPlaying, setIsAutoPlaying] = useState(true);

  const testimonials = [
    {
      id: 1,
      name: "Sarah Chen",
      role: "Content Creator",
      company: "@sarahcreates",
      avatar: "https://images.unsplash.com/photo-1494790108755-2616b612b786?w=100&h=100&fit=crop&crop=face&auto=format",
      content: `Social Composer Pro transformed my workflow completely. I went from spending 4 hours daily on social media management to just 30 minutes. The AI-powered scheduling is incredible!`,
      metrics: {
        engagement: "+340%",
        timeSaved: "3.5hrs/day",
        platforms: "8 platforms"
      },
      rating: 5
    },
    {
      id: 2,
      name: "Marcus Rodriguez",
      role: "Marketing Director",
      company: "TechFlow Agency",
      avatar: "https://images.unsplash.com/photo-1472099645785-5658abf4ff4e?w=100&h=100&fit=crop&crop=face&auto=format",
      content: `Our agency manages 50+ client accounts effortlessly now. The team collaboration features and client approval workflows have revolutionized how we deliver social media services.`,
      metrics: {
        clients: "50+ clients",
        efficiency: "+250%",
        revenue: "+180%"
      },
      rating: 5
    },
    {
      id: 3,
      name: "Emily Thompson",
      role: "Small Business Owner",
      company: "Bloom Bakery",
      avatar: "https://images.unsplash.com/photo-1438761681033-6461ffad8d80?w=100&h=100&fit=crop&crop=face&auto=format",
      content: `As a busy bakery owner, I needed something simple yet powerful. Social Composer Pro's templates and auto-posting helped me grow from 500 to 15K followers in 6 months!`,
      metrics: {
        followers: "500 → 15K",
        sales: "+220%",
        posts: "1,200+ posts"
      },
      rating: 5
    },
    {
      id: 4,
      name: "David Park",
      role: "Influencer",
      company: "@davidtravels",
      avatar: "https://images.unsplash.com/photo-1507003211169-0a1dd7228f2d?w=100&h=100&fit=crop&crop=face&auto=format",
      content: `The analytics dashboard gives me insights I never had before. I can see exactly what content performs best and when to post for maximum engagement across all platforms.`,
      metrics: {
        reach: "2M+ reach",
        engagement: "+420%",
        partnerships: "25+ brands"
      },
      rating: 5
    }
  ];

  const companyLogos = [
    { name: "TechCorp", logo: "https://images.unsplash.com/photo-1611224923853-80b023f02d71?w=120&h=60&fit=crop&auto=format" },
    { name: "StartupHub", logo: "https://images.pexels.com/photos/590022/pexels-photo-590022.jpeg?w=120&h=60&fit=crop&auto=format" },
    { name: "CreativeAgency", logo: "/assets/images/sample.png" },
    { name: "DigitalFlow", logo: "https://images.unsplash.com/photo-1560472354-b33ff0c44a43?w=120&h=60&fit=crop&auto=format" },
    { name: "BrandStudio", logo: "https://images.pexels.com/photos/265087/pexels-photo-265087.jpeg?w=120&h=60&fit=crop&auto=format" },
    { name: "MediaWorks", logo: "/assets/images/sample.png" }
  ];

  const keyMetrics = [
    {
      value: "2.5M+",
      label: "Posts Published",
      icon: "Send"
    },
    {
      value: "50K+",
      label: "Active Users",
      icon: "Users"
    },
    {
      value: "4.9/5",
      label: "User Rating",
      icon: "Star"
    },
    {
      value: "99.9%",
      label: "Uptime",
      icon: "Zap"
    }
  ];

  useEffect(() => {
    if (!isAutoPlaying) return;

    const interval = setInterval(() => {
      setCurrentSlide((prev) => (prev + 1) % testimonials?.length);
    }, 5000);

    return () => clearInterval(interval);
  }, [isAutoPlaying, testimonials?.length]);

  const nextSlide = () => {
    setCurrentSlide((prev) => (prev + 1) % testimonials?.length);
    setIsAutoPlaying(false);
  };

  const prevSlide = () => {
    setCurrentSlide((prev) => (prev - 1 + testimonials?.length) % testimonials?.length);
    setIsAutoPlaying(false);
  };

  const goToSlide = (index) => {
    setCurrentSlide(index);
    setIsAutoPlaying(false);
  };

  return (
    <section className="py-16 lg:py-24 bg-muted/30">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Header */}
        <div className="text-center mb-16">
          <div className="inline-flex items-center px-4 py-2 bg-success/10 rounded-full text-success text-sm font-medium mb-6">
            <Icon name="Heart" size={16} className="mr-2" />
            Loved by Creators Worldwide
          </div>
          
          <h2 className="text-3xl sm:text-4xl lg:text-5xl font-bold text-text-primary mb-6">
            Join Thousands of
            <span className="block text-transparent bg-clip-text bg-gradient-to-r from-primary to-secondary">
              Happy Creators
            </span>
          </h2>
        </div>

        {/* Key Metrics */}
        <div className="grid grid-cols-2 lg:grid-cols-4 gap-8 mb-16">
          {keyMetrics?.map((metric, index) => (
            <div key={index} className="text-center">
              <div className="w-16 h-16 bg-gradient-to-br from-primary to-secondary rounded-xl flex items-center justify-center mx-auto mb-4">
                <Icon 
                  name={metric?.icon} 
                  size={24} 
                  color="white"
                />
              </div>
              <div className="text-3xl font-bold text-text-primary mb-2">
                {metric?.value}
              </div>
              <div className="text-text-secondary text-sm">
                {metric?.label}
              </div>
            </div>
          ))}
        </div>

        {/* Testimonial Carousel */}
        <div className="relative bg-card rounded-2xl creative-confidence-shadow p-8 lg:p-12 mb-16">
          <div className="flex items-center justify-between mb-8">
            <div className="flex space-x-2">
              {testimonials?.map((_, index) => (
                <button
                  key={index}
                  onClick={() => goToSlide(index)}
                  className={`w-3 h-3 rounded-full transition-all duration-300 ${
                    index === currentSlide 
                      ? 'bg-primary scale-110' :'bg-border hover:bg-primary/50'
                  }`}
                  aria-label={`Testimonial ${index + 1}`}
                />
              ))}
            </div>
            
            <div className="flex space-x-2">
              <button
                onClick={prevSlide}
                className="w-10 h-10 bg-muted rounded-full flex items-center justify-center hover:bg-primary hover:text-white micro-transition"
                aria-label="Previous testimonial"
              >
                <Icon name="ChevronLeft" size={20} />
              </button>
              <button
                onClick={nextSlide}
                className="w-10 h-10 bg-muted rounded-full flex items-center justify-center hover:bg-primary hover:text-white micro-transition"
                aria-label="Next testimonial"
              >
                <Icon name="ChevronRight" size={20} />
              </button>
            </div>
          </div>

          <div className="grid lg:grid-cols-2 gap-8 items-center">
            {/* Testimonial Content */}
            <div>
              <div className="flex items-center mb-6">
                <Image
                  src={testimonials?.[currentSlide]?.avatar}
                  alt={testimonials?.[currentSlide]?.name}
                  className="w-16 h-16 rounded-full object-cover mr-4"
                />
                <div>
                  <h3 className="font-semibold text-text-primary">
                    {testimonials?.[currentSlide]?.name}
                  </h3>
                  <p className="text-text-secondary text-sm">
                    {testimonials?.[currentSlide]?.role}
                  </p>
                  <p className="text-primary text-sm font-medium">
                    {testimonials?.[currentSlide]?.company}
                  </p>
                </div>
              </div>

              <div className="flex items-center mb-4">
                {[...Array(testimonials?.[currentSlide]?.rating)]?.map((_, i) => (
                  <Icon key={i} name="Star" size={16} className="text-warning fill-current" />
                ))}
              </div>

              <blockquote className="text-lg text-text-primary mb-6 leading-relaxed">
                "{testimonials?.[currentSlide]?.content}"
              </blockquote>
            </div>

            {/* Metrics */}
            <div className="grid grid-cols-1 gap-4">
              {Object.entries(testimonials?.[currentSlide]?.metrics)?.map(([key, value], index) => (
                <div key={index} className="bg-muted/50 rounded-lg p-4 text-center">
                  <div className="text-2xl font-bold text-primary mb-1">
                    {value}
                  </div>
                  <div className="text-text-secondary text-sm capitalize">
                    {key?.replace(/([A-Z])/g, ' $1')?.trim()}
                  </div>
                </div>
              ))}
            </div>
          </div>
        </div>

        {/* Company Logos */}
        <div className="text-center">
          <p className="text-text-secondary text-sm mb-8">
            Trusted by leading companies and creators worldwide
          </p>
          
          <div className="flex flex-wrap items-center justify-center gap-8 opacity-60">
            {companyLogos?.map((company, index) => (
              <div key={index} className="grayscale hover:grayscale-0 micro-transition">
                <Image
                  src={company?.logo}
                  alt={company?.name}
                  className="h-8 w-auto object-contain"
                />
              </div>
            ))}
          </div>
        </div>
      </div>
    </section>
  );
};

export default SocialProofCarousel;