import React, { useState, useEffect } from 'react';
import Icon from '../../../components/AppIcon';
import Image from '../../../components/AppImage';
import Button from '../../../components/ui/Button';

const InteractiveComposerDemo = () => {
  const [activeTab, setActiveTab] = useState('compose');
  const [currentContent, setCurrentContent] = useState('');
  const [selectedPlatforms, setSelectedPlatforms] = useState(['facebook', 'twitter', 'instagram']);
  const [isTyping, setIsTyping] = useState(false);

  const demoContent = `🚀 Exciting news! Just launched our new product line and couldn't be more thrilled to share it with you all. 

The journey from concept to creation has been incredible, and seeing it come to life is truly rewarding. Thank you to everyone who supported us along the way! 

What's your favorite feature? Let me know in the comments! 👇

#ProductLaunch #Innovation #Grateful #NewBeginnings`;

  const platforms = [
    {
      id: 'facebook',
      name: 'Facebook',
      icon: 'Facebook',
      color: 'text-blue-600',
      bgColor: 'bg-blue-50',
      preview: {
        characterLimit: 63206,
        format: 'Full post with link preview',
        engagement: 'High reach potential'
      }
    },
    {
      id: 'twitter',
      name: 'Twitter',
      icon: 'Twitter',
      color: 'text-blue-400',
      bgColor: 'bg-blue-50',
      preview: {
        characterLimit: 280,
        format: 'Thread (3 tweets)',
        engagement: 'Optimal hashtag usage'
      }
    },
    {
      id: 'instagram',
      name: 'Instagram',
      icon: 'Instagram',
      color: 'text-pink-600',
      bgColor: 'bg-pink-50',
      preview: {
        characterLimit: 2200,
        format: 'Post + Story adaptation',
        engagement: 'Visual-first optimization'
      }
    },
    {
      id: 'linkedin',
      name: 'LinkedIn',
      icon: 'Linkedin',
      color: 'text-blue-700',
      bgColor: 'bg-blue-50',
      preview: {
        characterLimit: 3000,
        format: 'Professional tone',
        engagement: 'B2B audience focus'
      }
    }
  ];

  const tabs = [
    { id: 'compose', label: 'Compose', icon: 'Edit3' },
    { id: 'schedule', label: 'Schedule', icon: 'Calendar' },
    { id: 'preview', label: 'Preview', icon: 'Eye' }
  ];

  useEffect(() => {
    if (activeTab === 'compose') {
      setIsTyping(true);
      let index = 0;
      const interval = setInterval(() => {
        if (index < demoContent?.length) {
          setCurrentContent(demoContent?.slice(0, index + 1));
          index++;
        } else {
          setIsTyping(false);
          clearInterval(interval);
        }
      }, 30);

      return () => clearInterval(interval);
    }
  }, [activeTab, demoContent]);

  const togglePlatform = (platformId) => {
    setSelectedPlatforms(prev => 
      prev?.includes(platformId) 
        ? prev?.filter(id => id !== platformId)
        : [...prev, platformId]
    );
  };

  const renderComposeTab = () => (
    <div className="space-y-6">
      {/* Content Editor */}
      <div className="bg-background rounded-lg border border-border p-4">
        <div className="flex items-center justify-between mb-4">
          <h3 className="font-medium text-text-primary">Content Editor</h3>
          <div className="flex items-center space-x-2 text-sm text-text-secondary">
            <Icon name="Sparkles" size={16} className="text-primary" />
            <span>AI Assistant Active</span>
          </div>
        </div>
        
        <div className="relative">
          <textarea
            value={currentContent}
            readOnly
            className="w-full h-40 p-4 bg-muted/30 border border-border rounded-lg resize-none text-text-primary placeholder-text-secondary focus:outline-none focus:ring-2 focus:ring-primary/20"
            placeholder="What's on your mind?"
          />
          {isTyping && (
            <div className="absolute bottom-4 right-4">
              <div className="w-2 h-4 bg-primary animate-pulse rounded"></div>
            </div>
          )}
        </div>

        <div className="flex items-center justify-between mt-4">
          <div className="flex items-center space-x-4">
            <Button variant="ghost" size="sm" iconName="Image">
              Media
            </Button>
            <Button variant="ghost" size="sm" iconName="Hash">
              Hashtags
            </Button>
            <Button variant="ghost" size="sm" iconName="AtSign">
              Mention
            </Button>
          </div>
          <div className="text-sm text-text-secondary">
            {currentContent?.length} characters
          </div>
        </div>
      </div>

      {/* Platform Selection */}
      <div>
        <h3 className="font-medium text-text-primary mb-4">Select Platforms</h3>
        <div className="grid grid-cols-2 gap-3">
          {platforms?.map((platform) => (
            <button
              key={platform?.id}
              onClick={() => togglePlatform(platform?.id)}
              className={`flex items-center p-3 rounded-lg border transition-all duration-200 ${
                selectedPlatforms?.includes(platform?.id)
                  ? 'border-primary bg-primary/5' :'border-border hover:border-primary/50'
              }`}
            >
              <div className={`w-8 h-8 ${platform?.bgColor} rounded-lg flex items-center justify-center mr-3`}>
                <Icon name={platform?.icon} size={16} className={platform?.color} />
              </div>
              <div className="text-left">
                <div className="font-medium text-text-primary text-sm">
                  {platform?.name}
                </div>
                <div className="text-xs text-text-secondary">
                  {platform?.preview?.format}
                </div>
              </div>
              {selectedPlatforms?.includes(platform?.id) && (
                <Icon name="Check" size={16} className="text-primary ml-auto" />
              )}
            </button>
          ))}
        </div>
      </div>
    </div>
  );

  const renderScheduleTab = () => (
    <div className="space-y-6">
      <div className="bg-background rounded-lg border border-border p-4">
        <h3 className="font-medium text-text-primary mb-4">Optimal Posting Times</h3>
        
        <div className="space-y-3">
          {selectedPlatforms?.map((platformId) => {
            const platform = platforms?.find(p => p?.id === platformId);
            const times = ['9:00 AM', '1:00 PM', '6:00 PM'];
            
            return (
              <div key={platformId} className="flex items-center justify-between p-3 bg-muted/30 rounded-lg">
                <div className="flex items-center">
                  <div className={`w-8 h-8 ${platform?.bgColor} rounded-lg flex items-center justify-center mr-3`}>
                    <Icon name={platform?.icon} size={16} className={platform?.color} />
                  </div>
                  <span className="font-medium text-text-primary">{platform?.name}</span>
                </div>
                <div className="flex items-center space-x-2">
                  {times?.map((time, index) => (
                    <span key={index} className="px-2 py-1 bg-primary/10 text-primary text-xs rounded-full">
                      {time}
                    </span>
                  ))}
                </div>
              </div>
            );
          })}
        </div>

        <div className="mt-6 p-4 bg-success/10 rounded-lg">
          <div className="flex items-center mb-2">
            <Icon name="TrendingUp" size={16} className="text-success mr-2" />
            <span className="font-medium text-success">AI Recommendation</span>
          </div>
          <p className="text-sm text-text-secondary">
            Post at 1:00 PM today for 340% higher engagement based on your audience activity
          </p>
        </div>
      </div>
    </div>
  );

  const renderPreviewTab = () => (
    <div className="space-y-4">
      {selectedPlatforms?.map((platformId) => {
        const platform = platforms?.find(p => p?.id === platformId);
        
        return (
          <div key={platformId} className="bg-background rounded-lg border border-border p-4">
            <div className="flex items-center justify-between mb-3">
              <div className="flex items-center">
                <div className={`w-6 h-6 ${platform?.bgColor} rounded-lg flex items-center justify-center mr-2`}>
                  <Icon name={platform?.icon} size={12} className={platform?.color} />
                </div>
                <span className="font-medium text-text-primary text-sm">{platform?.name}</span>
              </div>
              <div className="flex items-center space-x-1">
                <Icon name="Eye" size={12} className="text-success" />
                <span className="text-xs text-success">Optimized</span>
              </div>
            </div>
            <div className="bg-muted/30 rounded-lg p-3">
              <p className="text-sm text-text-primary line-clamp-3">
                {currentContent?.slice(0, platform?.preview?.characterLimit)}
                {currentContent?.length > platform?.preview?.characterLimit && '...'}
              </p>
            </div>
            <div className="flex items-center justify-between mt-3 text-xs text-text-secondary">
              <span>{platform?.preview?.engagement}</span>
              <span>{Math.min(currentContent?.length, platform?.preview?.characterLimit)}/{platform?.preview?.characterLimit}</span>
            </div>
          </div>
        );
      })}
    </div>
  );

  return (
    <section className="py-16 lg:py-24 bg-background">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="grid lg:grid-cols-2 gap-12 items-start">
          {/* Left Content */}
          <div>
            <div className="inline-flex items-center px-4 py-2 bg-accent/10 rounded-full text-accent text-sm font-medium mb-6">
              <Icon name="Zap" size={16} className="mr-2" />
              Interactive Demo
            </div>
            
            <h2 className="text-3xl sm:text-4xl lg:text-5xl font-bold text-text-primary mb-6">
              Experience the
              <span className="block text-transparent bg-clip-text bg-gradient-to-r from-primary to-secondary">
                Composer Magic
              </span>
            </h2>
            
            <p className="text-xl text-text-secondary mb-8">
              Watch how our intelligent composer transforms your ideas into platform-perfect content. Create once, optimize everywhere, and publish with confidence.
            </p>

            <div className="space-y-6">
              <div className="flex items-center space-x-4">
                <div className="w-12 h-12 bg-primary/10 rounded-lg flex items-center justify-center">
                  <Icon name="Edit3" size={20} className="text-primary" />
                </div>
                <div>
                  <h3 className="font-semibold text-text-primary">Smart Content Creation</h3>
                  <p className="text-text-secondary text-sm">AI-powered suggestions and auto-formatting</p>
                </div>
              </div>
              
              <div className="flex items-center space-x-4">
                <div className="w-12 h-12 bg-secondary/10 rounded-lg flex items-center justify-center">
                  <Icon name="Target" size={20} className="text-secondary" />
                </div>
                <div>
                  <h3 className="font-semibold text-text-primary">Platform Optimization</h3>
                  <p className="text-text-secondary text-sm">Automatic adaptation for each social platform</p>
                </div>
              </div>
              
              <div className="flex items-center space-x-4">
                <div className="w-12 h-12 bg-accent/10 rounded-lg flex items-center justify-center">
                  <Icon name="Clock" size={20} className="text-accent" />
                </div>
                <div>
                  <h3 className="font-semibold text-text-primary">Perfect Timing</h3>
                  <p className="text-text-secondary text-sm">AI-recommended posting schedules</p>
                </div>
              </div>
            </div>
          </div>

          {/* Right Demo */}
          <div className="bg-card rounded-2xl creative-confidence-shadow p-6">
            {/* Tab Navigation */}
            <div className="flex space-x-1 bg-muted/50 rounded-lg p-1 mb-6">
              {tabs?.map((tab) => (
                <button
                  key={tab?.id}
                  onClick={() => setActiveTab(tab?.id)}
                  className={`flex items-center space-x-2 px-4 py-2 rounded-md text-sm font-medium transition-all duration-200 flex-1 justify-center ${
                    activeTab === tab?.id
                      ? 'bg-white text-primary shadow-sm'
                      : 'text-text-secondary hover:text-text-primary'
                  }`}
                >
                  <Icon name={tab?.icon} size={16} />
                  <span>{tab?.label}</span>
                </button>
              ))}
            </div>

            {/* Tab Content */}
            <div className="min-h-[400px]">
              {activeTab === 'compose' && renderComposeTab()}
              {activeTab === 'schedule' && renderScheduleTab()}
              {activeTab === 'preview' && renderPreviewTab()}
            </div>

            {/* Action Buttons */}
            <div className="flex space-x-3 mt-6">
              <Button variant="default" size="sm" iconName="Send" className="flex-1">
                Publish Now
              </Button>
              <Button variant="outline" size="sm" iconName="Clock" className="flex-1">
                Schedule
              </Button>
            </div>
          </div>
        </div>
      </div>
    </section>
  );
};

export default InteractiveComposerDemo;