import React, { useState, useEffect } from 'react';
import { Helmet } from 'react-helmet';
import Header from '../../components/ui/Header';
import StoryFilters from './components/StoryFilters';
import CaseStudyCard from './components/CaseStudyCard';
import VideoTestimonials from './components/VideoTestimonials';
import MetricsVisualization from './components/MetricsVisualization';
import CustomerQuotes from './components/CustomerQuotes';
import StatsOverview from './components/StatsOverview';
import Icon from '../../components/AppIcon';
import Button from '../../components/ui/Button';

const CustomerStories = () => {
  const [selectedIndustry, setSelectedIndustry] = useState('all');
  const [selectedCompanySize, setSelectedCompanySize] = useState('all');
  const [filteredStories, setFilteredStories] = useState([]);
  
  const caseStudies = [
    {
      id: 1,
      companyName: "TechFlow Solutions",
      location: "San Francisco, CA",
      companyLogo: "https://images.unsplash.com/photo-1560472354-b33ff0c44a43?w=100&h=100&fit=crop",
      industry: "saas",
      industryLabel: "SaaS",
      companySize: "medium",
      companySizeLabel: "Medium (11-50)",
      title: "How TechFlow Scaled Content Creation by 400% While Reducing Team Workload",
      summary: "TechFlow Solutions transformed their social media strategy from chaotic posting to systematic content orchestration, achieving unprecedented growth while empowering their team with creative confidence.",
      metrics: [
        { value: "400%", label: "Content Increase" },
        { value: "15hrs", label: "Weekly Time Saved" },
        { value: "250%", label: "Engagement Growth" }
      ],
      challenge: `TechFlow's marketing team was struggling with inconsistent posting across multiple platforms. With a growing SaaS product and limited resources, they were spending 25+ hours weekly on content creation and scheduling, yet their engagement rates were declining. The team felt overwhelmed by the complexity of managing LinkedIn, Twitter, Instagram, and Facebook simultaneously while maintaining brand consistency.`,
      solution: `Social Composer Pro's content-first approach revolutionized their workflow. The team implemented the unified composer for cross-platform content creation, utilized smart scheduling for optimal posting times, and leveraged the collaborative features for seamless team coordination. The analytics dashboard provided insights that guided their content strategy evolution.`,
      results: `Within three months, TechFlow achieved a 400% increase in content output while reducing weekly content creation time to just 10 hours. Their engagement rates improved by 250%, and the team reported feeling more creative and confident in their content strategy. The collaborative workflow enabled them to maintain consistent brand voice across all platforms.`,
      quote: {
        text: "Social Composer Pro didn\'t just improve our workflow—it completely transformed how we think about content creation. The creative confidence it gives our team is invaluable.",
        author: "Sarah Chen",
        role: "Head of Marketing",
        authorImage: "https://images.unsplash.com/photo-1494790108755-2616b612b786?w=100&h=100&fit=crop&crop=face"
      },
      videoUrl: "#"
    },
    {
      id: 2,
      companyName: "Bright Ideas Agency",
      location: "New York, NY",
      companyLogo: "https://images.unsplash.com/photo-1572021335469-31706a17aaef?w=100&h=100&fit=crop",
      industry: "marketing-agency",
      industryLabel: "Marketing Agency",
      companySize: "large",
      companySizeLabel: "Large (50+)",
      title: "Agency Scales Client Management with 60% Faster Project Delivery",
      summary: "Bright Ideas Agency leveraged Social Composer Pro's collaborative features to streamline client workflows, resulting in faster delivery times and improved client satisfaction across their portfolio.",
      metrics: [
        { value: "60%", label: "Faster Delivery" },
        { value: "25+", label: "Active Clients" },
        { value: "95%", label: "Client Satisfaction" }
      ],
      challenge: `Managing social media campaigns for 25+ clients was becoming increasingly complex. The agency struggled with maintaining brand consistency across different client accounts, coordinating team members on various projects, and delivering campaigns on time. Client feedback cycles were slow, and the team was working overtime to meet deadlines.`,
      solution: `The agency implemented Social Composer Pro's multi-account management system with role-based permissions. They utilized the collaborative workspace for client feedback integration, automated approval workflows, and centralized asset management. The platform's client portal feature enabled transparent communication and faster approval cycles.`,
      results: `Project delivery time decreased by 60%, allowing the agency to take on more clients without expanding their team. Client satisfaction scores improved to 95%, and the team reported better work-life balance. The streamlined workflow enabled them to focus more on creative strategy rather than administrative tasks.`,
      quote: {
        text: "The collaborative features revolutionized how our agency works with clients. We can now deliver premium social media strategies at scale without sacrificing quality.",
        author: "Marcus Rodriguez",
        role: "Creative Director",
        authorImage: "https://images.unsplash.com/photo-1472099645785-5658abf4ff4e?w=100&h=100&fit=crop&crop=face"
      },
      videoUrl: "#"
    },
    {
      id: 3,
      companyName: "Lifestyle by Emily",
      location: "Los Angeles, CA",
      companyLogo: "https://images.unsplash.com/photo-1611224923853-80b023f02d71?w=100&h=100&fit=crop",
      industry: "content-creator",
      industryLabel: "Content Creator",
      companySize: "solo",
      companySizeLabel: "Solo Creator",
      title: "Solo Creator Achieves 200% Follower Growth with Professional Tools",
      summary: "Emily transformed her lifestyle brand from inconsistent posting to professional content orchestration, achieving significant follower growth while maintaining authentic engagement.",
      metrics: [
        { value: "200%", label: "Follower Growth" },
        { value: "180%", label: "Engagement Rate" },
        { value: "12hrs", label: "Weekly Time Saved" }
      ],
      challenge: `As a solo lifestyle content creator, Emily was overwhelmed by the demands of creating content for Instagram, TikTok, YouTube, and Pinterest. She was spending 20+ hours weekly on content creation and posting, leaving little time for actual creative work. Her posting schedule was inconsistent, and she struggled to maintain engagement across platforms.`,
      solution: `Emily adopted Social Composer Pro's creator-focused features, including the visual content calendar, cross-platform optimization tools, and automated scheduling. She utilized the content templates and hashtag suggestions to maintain consistency while the analytics helped her understand what resonated with her audience.`,
      results: `Within six months, Emily's follower count grew by 200% across all platforms. Her engagement rate improved by 180%, and she reduced content management time to just 8 hours weekly. This freed up time for creating higher-quality content and building deeper connections with her audience.`,
      quote: {
        text: "As a solo creator, this platform gives me enterprise-level tools without the complexity. It's perfect for creators who want to grow professionally.",
        author: "Emily Watson",
        role: "Lifestyle Content Creator",
        authorImage: "https://images.unsplash.com/photo-1438761681033-6461ffad8d80?w=100&h=100&fit=crop&crop=face"
      },
      videoUrl: "#"
    },
    {
      id: 4,
      companyName: "EcoTech Solutions",
      location: "Portland, OR",
      companyLogo: "https://images.unsplash.com/photo-1441986300917-64674bd600d8?w=100&h=100&fit=crop",
      industry: "ecommerce",
      industryLabel: "E-commerce",
      companySize: "small",
      companySizeLabel: "Small Team (2-10)",
      title: "E-commerce Brand Increases Sales 180% Through Strategic Social Presence",
      summary: "EcoTech Solutions leveraged Social Composer Pro to build a consistent brand presence that drove significant sales growth while educating customers about sustainable technology.",
      metrics: [
        { value: "180%", label: "Sales Increase" },
        { value: "300%", label: "Website Traffic" },
        { value: "150%", label: "Brand Awareness" }
      ],
      challenge: `EcoTech's small team was struggling to maintain a professional social media presence while running their sustainable technology business. Their posting was sporadic, brand messaging was inconsistent, and they weren't effectively showcasing their products' environmental benefits. Sales from social media were minimal despite having great products.`,solution: `The team implemented Social Composer Pro's e-commerce focused features, including product showcase templates, educational content frameworks, and customer story amplification tools. They used the platform's analytics to identify the best times to post and which content types drove the most engagement and conversions.`,
      results: `Social media-driven sales increased by 180% within four months. Website traffic from social platforms grew by 300%, and brand awareness surveys showed a 150% improvement in recognition. The consistent, professional presence helped establish EcoTech as a thought leader in sustainable technology.`,
      quote: {
        text: "We went from posting sporadically to having a consistent, professional presence that actually drives sales. The analytics help us understand what resonates with our eco-conscious audience.",
        author: "Michael Chen",
        role: "Marketing Director",
        authorImage: "https://images.unsplash.com/photo-1472099645785-5658abf4ff4e?w=100&h=100&fit=crop&crop=face"
      }
    },
    {
      id: 5,
      companyName: "Wellness Warriors",
      location: "Austin, TX",
      companyLogo: "https://images.unsplash.com/photo-1559757148-5c350d0d3c56?w=100&h=100&fit=crop",
      industry: "healthcare",
      industryLabel: "Healthcare",
      companySize: "medium",
      companySizeLabel: "Medium (11-50)",
      title: "Healthcare Brand Builds Trust Through Consistent Educational Content",
      summary: "Wellness Warriors established themselves as a trusted health resource by leveraging Social Composer Pro's content orchestration to deliver consistent, valuable educational content across platforms.",
      metrics: [
        { value: "250%", label: "Community Growth" },
        { value: "15hrs", label: "Weekly Time Saved" },
        { value: "90%", label: "Content Consistency" }
      ],
      challenge: `Wellness Warriors needed to build trust and authority in the competitive healthcare space while ensuring all content met strict compliance requirements. Their small marketing team was overwhelmed trying to create educational content for multiple platforms while maintaining medical accuracy and brand consistency.`,
      solution: `They implemented Social Composer Pro's compliance-friendly workflow with approval processes, content templates for educational posts, and collaborative features that allowed medical professionals to review content before publication. The platform's scheduling ensured consistent posting without overwhelming the team.`,
      results: `Their online community grew by 250% as followers recognized them as a reliable health information source. The streamlined workflow saved 15 hours weekly while maintaining 90% content consistency across platforms. Patient inquiries increased significantly, and their reputation as health educators was firmly established.`,
      quote: {
        text: "The time savings alone justify the investment, but the quality improvement in our content is what really sets this platform apart. Our brand voice is now consistent everywhere.",author: "Rachel Adams",role: "Brand Manager",authorImage: "https://images.unsplash.com/photo-1544005313-94ddf0286df2?w=100&h=100&fit=crop&crop=face"
      }
    },
    {
      id: 6,
      companyName: "EduTech Academy",location: "Boston, MA",companyLogo: "https://images.unsplash.com/photo-1503676260728-1c00da094a0b?w=100&h=100&fit=crop",industry: "education",industryLabel: "Education",companySize: "large",companySizeLabel: "Large (50+)",title: "Educational Institution Engages Students with Modern Social Strategy",summary: "EduTech Academy modernized their student engagement approach using Social Composer Pro to create compelling educational content that resonates with digital-native learners.",
      metrics: [
        { value: "300%", label: "Student Engagement" },
        { value: "150%", label: "Course Inquiries" },
        { value: "200%", label: "Social Following" }
      ],
      challenge: `EduTech Academy's traditional marketing approach wasn't resonating with modern students. Their social media presence was outdated, engagement was low, and they were struggling to showcase their innovative programs effectively. The marketing team lacked the tools to create content that appealed to digital-native learners.`,
      solution: `The academy adopted Social Composer Pro's educational content framework, utilizing video content templates, student success story formats, and interactive post designs. They implemented a content calendar that aligned with academic seasons and used analytics to understand which educational content performed best.`,
      results: `Student engagement on social platforms increased by 300%, leading to a 150% increase in course inquiries. Their social media following doubled, and they became recognized as an innovative educational institution. The modern approach helped them attract younger demographics and improve their brand perception.`,
      quote: {
        text: "Social Composer Pro helped us speak the language of modern students. Our engagement rates prove that educational content can be both informative and compelling.",author: "Dr. Amanda Foster",role: "Director of Digital Marketing",authorImage: "https://images.unsplash.com/photo-1559757175-0eb30cd8c063?w=100&h=100&fit=crop&crop=face"
      }
    }
  ];

  useEffect(() => {
    let filtered = caseStudies;

    if (selectedIndustry !== 'all') {
      filtered = filtered?.filter(story => story?.industry === selectedIndustry);
    }

    if (selectedCompanySize !== 'all') {
      filtered = filtered?.filter(story => story?.companySize === selectedCompanySize);
    }

    setFilteredStories(filtered);
  }, [selectedIndustry, selectedCompanySize]);

  const handleClearFilters = () => {
    setSelectedIndustry('all');
    setSelectedCompanySize('all');
  };

  const beforeMetrics = {
    posts: 12,
    engagement: 2.3,
    timeSaved: 0,
    reach: 5000
  };

  const afterMetrics = {
    posts: 48,
    engagement: 8.1,
    timeSaved: 15,
    reach: 25000
  };

  return (
    <>
      <Helmet>
        <title>Customer Stories - Social Composer Pro</title>
        <meta name="description" content="Discover how creators, marketers, and teams transform their social media presence with Social Composer Pro. Read detailed case studies with real metrics and results." />
        <meta name="keywords" content="customer stories, case studies, social media success, content creation results, marketing transformation" />
      </Helmet>
      <div className="min-h-screen bg-background">
        <Header />
        
        {/* Hero Section */}
        <section className="pt-24 pb-16 bg-gradient-to-br from-primary/10 via-white to-secondary/10">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="text-center max-w-4xl mx-auto">
              <h1 className="text-4xl lg:text-6xl font-bold text-text-primary mb-6">
                Real Stories, Real Results
              </h1>
              <p className="text-xl lg:text-2xl text-text-secondary mb-8 leading-relaxed">
                Discover how creators, marketers, and teams worldwide have transformed their social media presence with Social Composer Pro. From solo creators to enterprise agencies, see the creative confidence our platform delivers.
              </p>
              <div className="flex flex-col sm:flex-row gap-4 justify-center">
                <Button
                  variant="default"
                  size="lg"
                  iconName="Zap"
                  iconPosition="left"
                >
                  Start Your Success Story
                </Button>
                <Button
                  variant="outline"
                  size="lg"
                  iconName="Play"
                  iconPosition="left"
                >
                  Watch Customer Videos
                </Button>
              </div>
            </div>
          </div>
        </section>

        {/* Stats Overview */}
        <StatsOverview />

        {/* Filters and Case Studies */}
        <section className="py-16 bg-white">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="text-center mb-12">
              <h2 className="text-3xl lg:text-4xl font-bold text-text-primary mb-4">
                Detailed Case Studies
              </h2>
              <p className="text-xl text-text-secondary max-w-3xl mx-auto">
                Explore in-depth transformation stories with metrics, challenges, solutions, and measurable results across different industries and company sizes.
              </p>
            </div>

            <StoryFilters
              selectedIndustry={selectedIndustry}
              selectedCompanySize={selectedCompanySize}
              onIndustryChange={setSelectedIndustry}
              onCompanySizeChange={setSelectedCompanySize}
              onClearFilters={handleClearFilters}
            />

            {/* Results Count */}
            <div className="mb-8">
              <p className="text-text-secondary">
                Showing {filteredStories?.length} of {caseStudies?.length} customer stories
              </p>
            </div>

            {/* Case Studies Grid */}
            <div className="grid lg:grid-cols-2 gap-8 mb-16">
              {filteredStories?.map((story) => (
                <CaseStudyCard key={story?.id} story={story} />
              ))}
            </div>

            {/* No Results */}
            {filteredStories?.length === 0 && (
              <div className="text-center py-16">
                <Icon name="Search" size={48} className="text-text-secondary mx-auto mb-4" />
                <h3 className="text-xl font-semibold text-text-primary mb-2">
                  No stories match your filters
                </h3>
                <p className="text-text-secondary mb-6">
                  Try adjusting your filters to see more customer stories.
                </p>
                <Button
                  variant="outline"
                  onClick={handleClearFilters}
                  iconName="RotateCcw"
                  iconPosition="left"
                >
                  Clear All Filters
                </Button>
              </div>
            )}
          </div>
        </section>

        {/* Metrics Visualization */}
        <section className="py-16 bg-muted/30">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="text-center mb-12">
              <h2 className="text-3xl lg:text-4xl font-bold text-text-primary mb-4">
                Average Customer Results
              </h2>
              <p className="text-xl text-text-secondary max-w-3xl mx-auto">
                See the typical transformation our customers experience within their first three months using Social Composer Pro.
              </p>
            </div>

            <MetricsVisualization
              beforeMetrics={beforeMetrics}
              afterMetrics={afterMetrics}
              title="Before vs. After: 3-Month Transformation"
            />
          </div>
        </section>

        {/* Video Testimonials */}
        <VideoTestimonials />

        {/* Customer Quotes */}
        <CustomerQuotes />

        {/* Final CTA */}
        <section className="py-16 bg-gradient-to-br from-primary to-secondary text-white">
          <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
            <h2 className="text-3xl lg:text-4xl font-bold mb-6">
              Ready to Write Your Success Story?
            </h2>
            <p className="text-xl mb-8 opacity-90">
              Join thousands of creators, marketers, and teams who've discovered the creative confidence that comes with professional-grade social media tools.
            </p>
            <div className="flex flex-col sm:flex-row gap-4 justify-center">
              <Button
                variant="secondary"
                size="lg"
                iconName="Zap"
                iconPosition="left"
                className="bg-white text-primary hover:bg-gray-50"
              >
                Start Free Trial
              </Button>
              <Button
                variant="outline"
                size="lg"
                iconName="Calendar"
                iconPosition="left"
                className="border-white text-white hover:bg-white/10"
              >
                Schedule Demo
              </Button>
            </div>
            <p className="text-sm opacity-75 mt-4">
              No credit card required • 30-day money-back guarantee
            </p>
          </div>
        </section>

        {/* Footer */}
        <footer className="bg-text-primary text-white py-12">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="grid md:grid-cols-4 gap-8">
              <div>
                <div className="flex items-center space-x-2 mb-4">
                  <div className="w-8 h-8 bg-gradient-to-br from-primary to-secondary rounded-lg flex items-center justify-center">
                    <Icon name="Zap" size={20} color="white" strokeWidth={2.5} />
                  </div>
                  <span className="text-xl font-bold">Social Composer Pro</span>
                </div>
                <p className="text-gray-400 text-sm">
                  Empowering creators with professional-grade social media tools.
                </p>
              </div>
              
              <div>
                <h4 className="font-semibold mb-4">Product</h4>
                <ul className="space-y-2 text-sm text-gray-400">
                  <li><a href="/product-features" className="hover:text-white smooth-transition">Features</a></li>
                  <li><a href="/pricing" className="hover:text-white smooth-transition">Pricing</a></li>
                  <li><a href="#" className="hover:text-white smooth-transition">Integrations</a></li>
                  <li><a href="#" className="hover:text-white smooth-transition">API</a></li>
                </ul>
              </div>
              
              <div>
                <h4 className="font-semibold mb-4">Resources</h4>
                <ul className="space-y-2 text-sm text-gray-400">
                  <li><a href="/resources" className="hover:text-white smooth-transition">Blog</a></li>
                  <li><a href="/customer-stories" className="hover:text-white smooth-transition">Customer Stories</a></li>
                  <li><a href="#" className="hover:text-white smooth-transition">Help Center</a></li>
                  <li><a href="#" className="hover:text-white smooth-transition">Community</a></li>
                </ul>
              </div>
              
              <div>
                <h4 className="font-semibold mb-4">Company</h4>
                <ul className="space-y-2 text-sm text-gray-400">
                  <li><a href="#" className="hover:text-white smooth-transition">About</a></li>
                  <li><a href="#" className="hover:text-white smooth-transition">Careers</a></li>
                  <li><a href="#" className="hover:text-white smooth-transition">Press</a></li>
                  <li><a href="#" className="hover:text-white smooth-transition">Contact</a></li>
                </ul>
              </div>
            </div>
            
            <div className="border-t border-gray-800 mt-8 pt-8 flex flex-col md:flex-row justify-between items-center">
              <p className="text-gray-400 text-sm">
                © {new Date()?.getFullYear()} Social Composer Pro. All rights reserved.
              </p>
              <div className="flex space-x-6 mt-4 md:mt-0">
                <a href="#" className="text-gray-400 hover:text-white smooth-transition">
                  <Icon name="Twitter" size={20} />
                </a>
                <a href="#" className="text-gray-400 hover:text-white smooth-transition">
                  <Icon name="Linkedin" size={20} />
                </a>
                <a href="#" className="text-gray-400 hover:text-white smooth-transition">
                  <Icon name="Instagram" size={20} />
                </a>
              </div>
            </div>
          </div>
        </footer>
      </div>
    </>
  );
};

export default CustomerStories;