import React from 'react';
import Button from '../../../components/ui/Button';
import Icon from '../../../components/AppIcon';

const StoryFilters = ({ 
  selectedIndustry, 
  selectedCompanySize, 
  onIndustryChange, 
  onCompanySizeChange,
  onClearFilters 
}) => {
  const industries = [
    { value: 'all', label: 'All Industries' },
    { value: 'marketing-agency', label: 'Marketing Agency' },
    { value: 'ecommerce', label: 'E-commerce' },
    { value: 'saas', label: 'SaaS' },
    { value: 'content-creator', label: 'Content Creator' },
    { value: 'nonprofit', label: 'Non-profit' },
    { value: 'healthcare', label: 'Healthcare' },
    { value: 'education', label: 'Education' }
  ];

  const companySizes = [
    { value: 'all', label: 'All Sizes' },
    { value: 'solo', label: 'Solo Creator' },
    { value: 'small', label: 'Small Team (2-10)' },
    { value: 'medium', label: 'Medium (11-50)' },
    { value: 'large', label: 'Large (50+)' }
  ];

  const hasActiveFilters = selectedIndustry !== 'all' || selectedCompanySize !== 'all';

  return (
    <div className="bg-white rounded-xl creative-confidence-shadow p-6 mb-8">
      <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between gap-6">
        <div className="flex flex-col sm:flex-row gap-4 flex-1">
          {/* Industry Filter */}
          <div className="flex-1">
            <label className="block text-sm font-medium text-text-secondary mb-2">
              Filter by Industry
            </label>
            <select
              value={selectedIndustry}
              onChange={(e) => onIndustryChange(e?.target?.value)}
              className="w-full px-4 py-2 border border-border rounded-lg bg-white text-text-primary focus:ring-2 focus:ring-primary focus:border-transparent micro-interaction"
            >
              {industries?.map((industry) => (
                <option key={industry?.value} value={industry?.value}>
                  {industry?.label}
                </option>
              ))}
            </select>
          </div>

          {/* Company Size Filter */}
          <div className="flex-1">
            <label className="block text-sm font-medium text-text-secondary mb-2">
              Filter by Company Size
            </label>
            <select
              value={selectedCompanySize}
              onChange={(e) => onCompanySizeChange(e?.target?.value)}
              className="w-full px-4 py-2 border border-border rounded-lg bg-white text-text-primary focus:ring-2 focus:ring-primary focus:border-transparent micro-interaction"
            >
              {companySizes?.map((size) => (
                <option key={size?.value} value={size?.value}>
                  {size?.label}
                </option>
              ))}
            </select>
          </div>
        </div>

        {/* Clear Filters */}
        {hasActiveFilters && (
          <Button
            variant="outline"
            onClick={onClearFilters}
            iconName="X"
            iconPosition="left"
            className="lg:ml-4"
          >
            Clear Filters
          </Button>
        )}
      </div>
      {/* Active Filters Display */}
      {hasActiveFilters && (
        <div className="flex flex-wrap gap-2 mt-4 pt-4 border-t border-border">
          <span className="text-sm text-text-secondary">Active filters:</span>
          {selectedIndustry !== 'all' && (
            <span className="inline-flex items-center gap-1 px-3 py-1 bg-primary/10 text-primary rounded-full text-sm">
              {industries?.find(i => i?.value === selectedIndustry)?.label}
              <button
                onClick={() => onIndustryChange('all')}
                className="hover:bg-primary/20 rounded-full p-0.5 micro-transition"
              >
                <Icon name="X" size={12} />
              </button>
            </span>
          )}
          {selectedCompanySize !== 'all' && (
            <span className="inline-flex items-center gap-1 px-3 py-1 bg-secondary/10 text-secondary rounded-full text-sm">
              {companySizes?.find(s => s?.value === selectedCompanySize)?.label}
              <button
                onClick={() => onCompanySizeChange('all')}
                className="hover:bg-secondary/20 rounded-full p-0.5 micro-transition"
              >
                <Icon name="X" size={12} />
              </button>
            </span>
          )}
        </div>
      )}
    </div>
  );
};

export default StoryFilters;