import React from 'react';
import Icon from '../../../components/AppIcon';

const MetricsVisualization = ({ beforeMetrics, afterMetrics, title }) => {
  const calculateImprovement = (before, after) => {
    const improvement = ((after - before) / before) * 100;
    return improvement > 0 ? `+${improvement?.toFixed(0)}%` : `${improvement?.toFixed(0)}%`;
  };

  const formatNumber = (num) => {
    if (num >= 1000000) {
      return `${(num / 1000000)?.toFixed(1)}M`;
    } else if (num >= 1000) {
      return `${(num / 1000)?.toFixed(1)}K`;
    }
    return num?.toString();
  };

  const metrics = [
    {
      label: 'Monthly Posts',
      before: beforeMetrics?.posts,
      after: afterMetrics?.posts,
      icon: 'FileText',
      color: 'text-blue-600'
    },
    {
      label: 'Engagement Rate',
      before: beforeMetrics?.engagement,
      after: afterMetrics?.engagement,
      icon: 'Heart',
      color: 'text-pink-600',
      suffix: '%'
    },
    {
      label: 'Time Saved',
      before: beforeMetrics?.timeSaved,
      after: afterMetrics?.timeSaved,
      icon: 'Clock',
      color: 'text-green-600',
      suffix: 'hrs'
    },
    {
      label: 'Reach',
      before: beforeMetrics?.reach,
      after: afterMetrics?.reach,
      icon: 'Users',
      color: 'text-purple-600'
    }
  ];

  return (
    <div className="bg-white rounded-xl creative-confidence-shadow p-6">
      <h4 className="text-lg font-semibold text-text-primary mb-6 text-center">
        {title}
      </h4>
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        {metrics?.map((metric, index) => (
          <div key={index} className="text-center">
            {/* Icon */}
            <div className={`w-12 h-12 mx-auto mb-4 rounded-full bg-gray-100 flex items-center justify-center ${metric?.color}`}>
              <Icon name={metric?.icon} size={24} />
            </div>

            {/* Label */}
            <div className="text-sm font-medium text-text-secondary mb-3">
              {metric?.label}
            </div>

            {/* Before/After Comparison */}
            <div className="space-y-2">
              {/* Before */}
              <div className="flex items-center justify-center gap-2">
                <span className="text-xs text-text-secondary">Before:</span>
                <span className="text-sm text-text-primary">
                  {formatNumber(metric?.before)}{metric?.suffix || ''}
                </span>
              </div>

              {/* Arrow */}
              <div className="flex justify-center">
                <Icon name="ArrowDown" size={16} className="text-text-secondary" />
              </div>

              {/* After */}
              <div className="flex items-center justify-center gap-2">
                <span className="text-xs text-text-secondary">After:</span>
                <span className="text-lg font-bold text-primary">
                  {formatNumber(metric?.after)}{metric?.suffix || ''}
                </span>
              </div>

              {/* Improvement */}
              <div className="mt-2">
                <span className={`inline-flex items-center gap-1 px-2 py-1 rounded-full text-xs font-medium ${
                  metric?.after > metric?.before 
                    ? 'bg-success/10 text-success' :'bg-error/10 text-error'
                }`}>
                  <Icon 
                    name={metric?.after > metric?.before ? "TrendingUp" : "TrendingDown"} 
                    size={12} 
                  />
                  {calculateImprovement(metric?.before, metric?.after)}
                </span>
              </div>
            </div>
          </div>
        ))}
      </div>
      {/* Summary */}
      <div className="mt-8 pt-6 border-t border-border text-center">
        <p className="text-text-secondary text-sm">
          Results achieved within 3 months of implementing Social Composer Pro
        </p>
      </div>
    </div>
  );
};

export default MetricsVisualization;