import React, { useState } from 'react';
import Image from '../../../components/AppImage';
import Icon from '../../../components/AppIcon';
import Button from '../../../components/ui/Button';

const CaseStudyCard = ({ story }) => {
  const [isExpanded, setIsExpanded] = useState(false);

  const getIndustryColor = (industry) => {
    const colors = {
      'marketing-agency': 'bg-blue-100 text-blue-700',
      'ecommerce': 'bg-green-100 text-green-700',
      'saas': 'bg-purple-100 text-purple-700',
      'content-creator': 'bg-pink-100 text-pink-700',
      'nonprofit': 'bg-orange-100 text-orange-700',
      'healthcare': 'bg-red-100 text-red-700',
      'education': 'bg-indigo-100 text-indigo-700'
    };
    return colors?.[industry] || 'bg-gray-100 text-gray-700';
  };

  const getSizeColor = (size) => {
    const colors = {
      'solo': 'bg-yellow-100 text-yellow-700',
      'small': 'bg-emerald-100 text-emerald-700',
      'medium': 'bg-cyan-100 text-cyan-700',
      'large': 'bg-violet-100 text-violet-700'
    };
    return colors?.[size] || 'bg-gray-100 text-gray-700';
  };

  return (
    <div className="bg-white rounded-xl creative-confidence-shadow hover:shadow-lg smooth-transition overflow-hidden">
      {/* Header with Company Info */}
      <div className="p-6 pb-4">
        <div className="flex items-start justify-between mb-4">
          <div className="flex items-center gap-4">
            <div className="w-16 h-16 rounded-lg overflow-hidden bg-muted flex-shrink-0">
              <Image
                src={story?.companyLogo}
                alt={`${story?.companyName} logo`}
                className="w-full h-full object-cover"
              />
            </div>
            <div>
              <h3 className="text-xl font-bold text-text-primary mb-1">
                {story?.companyName}
              </h3>
              <p className="text-text-secondary text-sm">
                {story?.location}
              </p>
            </div>
          </div>
          <div className="flex flex-col gap-2">
            <span className={`px-3 py-1 rounded-full text-xs font-medium ${getIndustryColor(story?.industry)}`}>
              {story?.industryLabel}
            </span>
            <span className={`px-3 py-1 rounded-full text-xs font-medium ${getSizeColor(story?.companySize)}`}>
              {story?.companySizeLabel}
            </span>
          </div>
        </div>

        <h4 className="text-lg font-semibold text-text-primary mb-3">
          {story?.title}
        </h4>

        <p className="text-text-secondary leading-relaxed mb-4">
          {story?.summary}
        </p>
      </div>
      {/* Key Metrics */}
      <div className="px-6 pb-4">
        <div className="grid grid-cols-3 gap-4">
          {story?.metrics?.map((metric, index) => (
            <div key={index} className="text-center">
              <div className="text-2xl font-bold text-primary mb-1">
                {metric?.value}
              </div>
              <div className="text-xs text-text-secondary">
                {metric?.label}
              </div>
            </div>
          ))}
        </div>
      </div>
      {/* Expandable Content */}
      <div className={`px-6 overflow-hidden smooth-transition ${isExpanded ? 'max-h-none pb-6' : 'max-h-0'}`}>
        <div className="border-t border-border pt-4">
          {/* Challenge */}
          <div className="mb-4">
            <h5 className="font-semibold text-text-primary mb-2 flex items-center gap-2">
              <Icon name="AlertCircle" size={16} className="text-warning" />
              Challenge
            </h5>
            <p className="text-text-secondary text-sm leading-relaxed">
              {story?.challenge}
            </p>
          </div>

          {/* Solution */}
          <div className="mb-4">
            <h5 className="font-semibold text-text-primary mb-2 flex items-center gap-2">
              <Icon name="Lightbulb" size={16} className="text-accent" />
              Solution
            </h5>
            <p className="text-text-secondary text-sm leading-relaxed">
              {story?.solution}
            </p>
          </div>

          {/* Results */}
          <div className="mb-4">
            <h5 className="font-semibold text-text-primary mb-2 flex items-center gap-2">
              <Icon name="TrendingUp" size={16} className="text-success" />
              Results
            </h5>
            <p className="text-text-secondary text-sm leading-relaxed">
              {story?.results}
            </p>
          </div>

          {/* Quote */}
          {story?.quote && (
            <div className="bg-muted rounded-lg p-4 mb-4">
              <blockquote className="text-text-primary italic mb-3">
                "{story?.quote?.text}"
              </blockquote>
              <div className="flex items-center gap-3">
                <div className="w-10 h-10 rounded-full overflow-hidden bg-border">
                  <Image
                    src={story?.quote?.authorImage}
                    alt={story?.quote?.author}
                    className="w-full h-full object-cover"
                  />
                </div>
                <div>
                  <div className="font-medium text-text-primary text-sm">
                    {story?.quote?.author}
                  </div>
                  <div className="text-text-secondary text-xs">
                    {story?.quote?.role}
                  </div>
                </div>
              </div>
            </div>
          )}
        </div>
      </div>
      {/* Footer Actions */}
      <div className="px-6 pb-6 flex items-center justify-between">
        <Button
          variant="ghost"
          onClick={() => setIsExpanded(!isExpanded)}
          iconName={isExpanded ? "ChevronUp" : "ChevronDown"}
          iconPosition="right"
          size="sm"
        >
          {isExpanded ? 'Show Less' : 'Read Full Story'}
        </Button>

        <div className="flex items-center gap-2">
          <Button
            variant="outline"
            size="sm"
            iconName="Download"
            iconPosition="left"
          >
            Download PDF
          </Button>
          {story?.videoUrl && (
            <Button
              variant="outline"
              size="sm"
              iconName="Play"
              iconPosition="left"
            >
              Watch Video
            </Button>
          )}
        </div>
      </div>
    </div>
  );
};

export default CaseStudyCard;