import React, { useState, useEffect, useRef } from 'react';
import { Bold, Italic, Link, AtSign, Hash, Image, Save, Eye, Settings, X, Plus } from 'lucide-react';
import MainEditor from './components/MainEditor';
import PlatformPreview from './components/PlatformPreview';
import PlatformControls from './components/PlatformControls';
import MediaAttachment from './components/MediaAttachment';
import SchedulingPicker from './components/SchedulingPicker';
import CharacterCounter from './components/CharacterCounter';

const ComposerInterface = () => {
  const [content, setContent] = useState('');
  const [selectedPlatforms, setSelectedPlatforms] = useState({
    twitter: true,
    instagram: false,
    linkedin: false
  });
  const [activePreviewTab, setActivePreviewTab] = useState('twitter');
  const [scheduledDate, setScheduledDate] = useState(null);
  const [attachedMedia, setAttachedMedia] = useState([]);
  const [isMobile, setIsMobile] = useState(false);
  const [showMobilePanel, setShowMobilePanel] = useState(false);
  const [lastSaved, setLastSaved] = useState(null);
  const editorRef = useRef(null);

  // Auto-save functionality
  useEffect(() => {
    if (content?.trim()) {
      const timer = setTimeout(() => {
        setLastSaved(new Date());
        // Here you would implement actual save logic
        console.log('Auto-saved:', content);
      }, 30000); // 30 seconds

      return () => clearTimeout(timer);
    }
  }, [content]);

  // Responsive detection
  useEffect(() => {
    const checkMobile = () => {
      setIsMobile(window.innerWidth < 768);
    };
    
    checkMobile();
    window.addEventListener('resize', checkMobile);
    return () => window.removeEventListener('resize', checkMobile);
  }, []);

  // Keyboard shortcuts
  useEffect(() => {
    const handleKeyboard = (e) => {
      if ((e?.ctrlKey || e?.metaKey)) {
        switch (e?.key) {
          case 'n':
            e?.preventDefault();
            setContent('');
            setAttachedMedia([]);
            break;
          case 's':
            e?.preventDefault();
            handleSave();
            break;
          case 'z':
            if (e?.shiftKey) {
              e?.preventDefault();
              // Redo logic would go here
            } else {
              e?.preventDefault();
              // Undo logic would go here
            }
            break;
        }
      }
    };

    document.addEventListener('keydown', handleKeyboard);
    return () => document.removeEventListener('keydown', handleKeyboard);
  }, []);

  const handleSave = () => {
    setLastSaved(new Date());
    // Implement save logic here
    console.log('Manual save:', { content, selectedPlatforms, scheduledDate, attachedMedia });
  };

  const handlePublish = () => {
    // Implement publish logic here
    console.log('Publishing to:', Object.keys(selectedPlatforms)?.filter(p => selectedPlatforms?.[p]));
  };

  const getCharacterCount = (platform) => {
    const limits = {
      twitter: 280,
      instagram: 2200,
      linkedin: 3000
    };
    return { count: content?.length, limit: limits?.[platform] };
  };

  const getOptimalPostingTime = () => {
    // Mock optimal time based on platform analytics
    const now = new Date();
    const optimal = new Date(now);
    optimal?.setHours(14, 30, 0, 0); // 2:30 PM
    return optimal;
  };

  if (isMobile) {
    return (
      <div className="min-h-screen bg-gray-50">
        {/* Mobile Header */}
        <div className="bg-white border-b border-gray-200 p-4 flex items-center justify-between">
          <div className="flex items-center space-x-3">
            <button 
              onClick={() => setShowMobilePanel(!showMobilePanel)}
              className="p-2 hover:bg-gray-100 rounded-lg"
            >
              <Settings className="w-5 h-5" />
            </button>
            <h1 className="text-lg font-semibold">Compose</h1>
          </div>
          <div className="flex items-center space-x-2">
            <button 
              onClick={handleSave}
              className="px-3 py-1.5 text-sm bg-blue-600 text-white rounded-lg"
            >
              Save
            </button>
            <button 
              onClick={handlePublish}
              className="px-3 py-1.5 text-sm bg-green-600 text-white rounded-lg"
            >
              Publish
            </button>
          </div>
        </div>
        {/* Mobile Editor */}
        <div className="p-4">
          <MainEditor 
            content={content}
            setContent={setContent}
            ref={editorRef}
            isMobile={true}
          />
          
          <div className="mt-4">
            <PlatformPreview 
              content={content}
              activeTab={activePreviewTab}
              setActiveTab={setActivePreviewTab}
              selectedPlatforms={selectedPlatforms}
            />
          </div>
        </div>
        {/* Mobile Controls Panel */}
        {showMobilePanel && (
          <div className="fixed inset-0 bg-black bg-opacity-50 z-50">
            <div className="absolute bottom-0 left-0 right-0 bg-white rounded-t-2xl max-h-[80vh] overflow-y-auto">
              <div className="p-4 border-b border-gray-200 flex items-center justify-between">
                <h2 className="text-lg font-semibold">Publishing Options</h2>
                <button 
                  onClick={() => setShowMobilePanel(false)}
                  className="p-2 hover:bg-gray-100 rounded-lg"
                >
                  <X className="w-5 h-5" />
                </button>
              </div>
              
              <div className="p-4 space-y-6">
                <PlatformControls 
                  selectedPlatforms={selectedPlatforms}
                  setSelectedPlatforms={setSelectedPlatforms}
                />
                
                <MediaAttachment 
                  attachedMedia={attachedMedia}
                  setAttachedMedia={setAttachedMedia}
                />
                
                <SchedulingPicker 
                  scheduledDate={scheduledDate}
                  setScheduledDate={setScheduledDate}
                  optimalTime={getOptimalPostingTime()}
                />
                
                <div className="space-y-3">
                  {Object.entries(selectedPlatforms)?.map(([platform, selected]) => 
                    selected && (
                      <CharacterCounter 
                        key={platform}
                        platform={platform}
                        {...getCharacterCount(platform)}
                      />
                    )
                  )}
                </div>
              </div>
            </div>
          </div>
        )}
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Desktop Header */}
      <div className="bg-white border-b border-gray-200 px-6 py-4">
        <div className="flex items-center justify-between">
          <div className="flex items-center space-x-4">
            <h1 className="text-2xl font-bold text-gray-900">Social Composer Pro</h1>
            {lastSaved && (
              <span className="text-sm text-gray-500">
                Last saved: {lastSaved?.toLocaleTimeString()}
              </span>
            )}
          </div>
          
          <div className="flex items-center space-x-3">
            <button 
              onClick={() => setContent('')}
              className="px-4 py-2 text-gray-600 hover:text-gray-900 flex items-center space-x-2"
            >
              <Plus className="w-4 h-4" />
              <span>New Post</span>
            </button>
            
            <button 
              onClick={handleSave}
              className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 flex items-center space-x-2"
            >
              <Save className="w-4 h-4" />
              <span>Save Draft</span>
            </button>
            
            <button 
              onClick={handlePublish}
              className="px-6 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 font-medium"
            >
              Publish Now
            </button>
          </div>
        </div>
      </div>
      {/* Desktop Main Layout */}
      <div className="flex h-[calc(100vh-80px)]">
        {/* Left Panel - Editor */}
        <div className="flex-1 flex flex-col bg-white mr-1">
          {/* Editor Toolbar */}
          <div className="border-b border-gray-200 p-4">
            <div className="flex items-center space-x-1">
              <button className="p-2 hover:bg-gray-100 rounded" title="Bold (Ctrl+B)">
                <Bold className="w-4 h-4" />
              </button>
              <button className="p-2 hover:bg-gray-100 rounded" title="Italic (Ctrl+I)">
                <Italic className="w-4 h-4" />
              </button>
              <div className="w-px h-6 bg-gray-300 mx-2" />
              <button className="p-2 hover:bg-gray-100 rounded" title="Add Link">
                <Link className="w-4 h-4" />
              </button>
              <button className="p-2 hover:bg-gray-100 rounded" title="Mention">
                <AtSign className="w-4 h-4" />
              </button>
              <button className="p-2 hover:bg-gray-100 rounded" title="Hashtag">
                <Hash className="w-4 h-4" />
              </button>
              <div className="w-px h-6 bg-gray-300 mx-2" />
              <button className="p-2 hover:bg-gray-100 rounded" title="Add Media">
                <Image className="w-4 h-4" />
              </button>
            </div>
          </div>

          {/* Main Editor Area */}
          <div className="flex-1 p-6">
            <MainEditor 
              content={content}
              setContent={setContent}
              ref={editorRef}
            />
          </div>

          {/* Platform Preview Tabs */}
          <div className="border-t border-gray-200 p-4">
            <PlatformPreview 
              content={content}
              activeTab={activePreviewTab}
              setActiveTab={setActivePreviewTab}
              selectedPlatforms={selectedPlatforms}
            />
          </div>
        </div>

        {/* Right Panel - Controls */}
        <div className="w-80 bg-white border-l border-gray-200 flex flex-col">
          <div className="p-4 border-b border-gray-200">
            <h2 className="text-lg font-semibold text-gray-900">Publishing Options</h2>
          </div>
          
          <div className="flex-1 overflow-y-auto p-4 space-y-6">
            {/* Platform Selection */}
            <PlatformControls 
              selectedPlatforms={selectedPlatforms}
              setSelectedPlatforms={setSelectedPlatforms}
            />

            {/* Character Counters */}
            <div className="space-y-3">
              <h3 className="text-sm font-medium text-gray-900">Character Count</h3>
              {Object.entries(selectedPlatforms)?.map(([platform, selected]) => 
                selected && (
                  <CharacterCounter 
                    key={platform}
                    platform={platform}
                    {...getCharacterCount(platform)}
                  />
                )
              )}
            </div>

            {/* Media Attachments */}
            <MediaAttachment 
              attachedMedia={attachedMedia}
              setAttachedMedia={setAttachedMedia}
            />

            {/* Scheduling */}
            <SchedulingPicker 
              scheduledDate={scheduledDate}
              setScheduledDate={setScheduledDate}
              optimalTime={getOptimalPostingTime()}
            />

            {/* Audience Targeting */}
            <div>
              <h3 className="text-sm font-medium text-gray-900 mb-3">Audience Targeting</h3>
              <div className="space-y-2">
                <label className="flex items-center">
                  <input type="checkbox" className="rounded border-gray-300" />
                  <span className="ml-2 text-sm text-gray-600">High-engagement followers</span>
                </label>
                <label className="flex items-center">
                  <input type="checkbox" className="rounded border-gray-300" />
                  <span className="ml-2 text-sm text-gray-600">New audience expansion</span>
                </label>
              </div>
            </div>
          </div>

          {/* Bottom Actions */}
          <div className="p-4 border-t border-gray-200 space-y-3">
            <button 
              onClick={handlePublish}
              className="w-full px-4 py-3 bg-green-600 text-white rounded-lg hover:bg-green-700 font-medium"
              disabled={!content?.trim()}
            >
              {scheduledDate ? 'Schedule Post' : 'Publish Now'}
            </button>
            
            <div className="flex space-x-2">
              <button className="flex-1 px-3 py-2 border border-gray-300 text-gray-600 rounded-lg hover:bg-gray-50 text-sm">
                <Eye className="w-4 h-4 mx-auto" />
              </button>
              <button className="flex-1 px-3 py-2 border border-gray-300 text-gray-600 rounded-lg hover:bg-gray-50 text-sm">
                Save Draft
              </button>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default ComposerInterface;