import React, { useState } from 'react';
import { Calendar, Clock, Zap, Globe } from 'lucide-react';

const SchedulingPicker = ({ scheduledDate, setScheduledDate, optimalTime }) => {
  const [selectedTimezone, setSelectedTimezone] = useState('America/New_York');
  const [showAdvanced, setShowAdvanced] = useState(false);

  const timezones = [
    { value: 'America/New_York', label: 'Eastern Time (ET)' },
    { value: 'America/Chicago', label: 'Central Time (CT)' },
    { value: 'America/Denver', label: 'Mountain Time (MT)' },
    { value: 'America/Los_Angeles', label: 'Pacific Time (PT)' },
    { value: 'UTC', label: 'UTC' },
    { value: 'Europe/London', label: 'London (GMT/BST)' },
    { value: 'Europe/Paris', label: 'Paris (CET/CEST)' },
    { value: 'Asia/Tokyo', label: 'Tokyo (JST)' }
  ];

  const handleDateChange = (e) => {
    if (e?.target?.value) {
      setScheduledDate(new Date(e.target.value));
    } else {
      setScheduledDate(null);
    }
  };

  const handleOptimalTimeSelect = () => {
    setScheduledDate(optimalTime);
  };

  const formatDateTime = (date) => {
    if (!date) return '';
    return date?.toISOString()?.slice(0, 16); // Format for datetime-local input
  };

  const formatDisplayTime = (date) => {
    if (!date) return '';
    return new Intl.DateTimeFormat('en-US', {
      weekday: 'short',
      month: 'short',
      day: 'numeric',
      hour: 'numeric',
      minute: '2-digit',
      timeZoneName: 'short'
    })?.format(date);
  };

  return (
    <div>
      <h3 className="text-sm font-medium text-gray-900 mb-3">Scheduling</h3>
      {/* Quick Options */}
      <div className="space-y-2 mb-4">
        <button
          onClick={() => setScheduledDate(null)}
          className={`
            w-full flex items-center space-x-2 p-3 rounded-lg border transition-colors
            ${!scheduledDate 
              ? 'border-blue-500 bg-blue-50 text-blue-700' :'border-gray-200 hover:border-gray-300 text-gray-700'
            }
          `}
        >
          <Zap className="w-4 h-4" />
          <span className="font-medium">Publish Now</span>
        </button>

        <button
          onClick={handleOptimalTimeSelect}
          className={`
            w-full flex items-center justify-between p-3 rounded-lg border transition-colors
            ${scheduledDate?.getTime() === optimalTime?.getTime()
              ? 'border-green-500 bg-green-50 text-green-700' :'border-gray-200 hover:border-gray-300 text-gray-700'
            }
          `}
        >
          <div className="flex items-center space-x-2">
            <Clock className="w-4 h-4" />
            <span className="font-medium">Optimal Time</span>
          </div>
          <span className="text-xs">
            {optimalTime ? formatDisplayTime(optimalTime) : 'Loading...'}
          </span>
        </button>
      </div>
      {/* Custom Schedule */}
      <div className="space-y-3">
        <button
          onClick={() => setShowAdvanced(!showAdvanced)}
          className="flex items-center space-x-2 text-sm text-blue-600 hover:text-blue-700"
        >
          <Calendar className="w-4 h-4" />
          <span>Custom Schedule</span>
        </button>

        {showAdvanced && (
          <div className="space-y-3 p-3 bg-gray-50 rounded-lg">
            {/* Date/Time Picker */}
            <div>
              <label className="block text-xs font-medium text-gray-700 mb-1">
                Date & Time
              </label>
              <input
                type="datetime-local"
                value={formatDateTime(scheduledDate)}
                onChange={handleDateChange}
                min={formatDateTime(new Date())}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
              />
            </div>

            {/* Timezone Selector */}
            <div>
              <label className="block text-xs font-medium text-gray-700 mb-1">
                <Globe className="w-3 h-3 inline mr-1" />
                Timezone
              </label>
              <select
                value={selectedTimezone}
                onChange={(e) => setSelectedTimezone(e?.target?.value)}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
              >
                {timezones?.map(tz => (
                  <option key={tz?.value} value={tz?.value}>
                    {tz?.label}
                  </option>
                ))}
              </select>
            </div>

            {/* Schedule Summary */}
            {scheduledDate && (
              <div className="pt-2 border-t border-gray-200">
                <p className="text-xs text-gray-600">
                  <strong>Scheduled for:</strong> {formatDisplayTime(scheduledDate)}
                </p>
              </div>
            )}
          </div>
        )}
      </div>
      {/* Analytics Insight */}
      <div className="mt-4 p-3 bg-blue-50 rounded-lg">
        <p className="text-xs text-blue-700">
          <strong>💡 Smart Tip:</strong> Posts scheduled for {optimalTime ? optimalTime?.toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'}) : '2:30 PM'} typically receive 40% more engagement based on your audience analytics.
        </p>
      </div>
    </div>
  );
};

export default SchedulingPicker;