import React, { useState, useCallback } from 'react';
import { Upload, Image, Video, File, X } from 'lucide-react';

const MediaAttachment = ({ attachedMedia, setAttachedMedia }) => {
  const [dragActive, setDragActive] = useState(false);

  const handleDrag = useCallback((e) => {
    e?.preventDefault();
    e?.stopPropagation();
    if (e?.type === "dragenter" || e?.type === "dragover") {
      setDragActive(true);
    } else if (e?.type === "dragleave") {
      setDragActive(false);
    }
  }, []);

  const handleDrop = useCallback((e) => {
    e?.preventDefault();
    e?.stopPropagation();
    setDragActive(false);
    
    if (e?.dataTransfer?.files && e?.dataTransfer?.files?.[0]) {
      handleFiles(Array.from(e?.dataTransfer?.files));
    }
  }, []);

  const handleChange = (e) => {
    e?.preventDefault();
    if (e?.target?.files && e?.target?.files?.[0]) {
      handleFiles(Array.from(e?.target?.files));
    }
  };

  const handleFiles = (files) => {
    const newMedia = files?.map(file => ({
      id: Date.now() + Math.random(),
      file,
      name: file?.name,
      size: file?.size,
      type: file?.type,
      preview: file?.type?.startsWith('image/') ? URL.createObjectURL(file) : null
    }));
    
    setAttachedMedia(prev => [...prev, ...newMedia]);
  };

  const removeMedia = (id) => {
    setAttachedMedia(prev => {
      const updated = prev?.filter(item => item?.id !== id);
      // Clean up preview URLs
      const removed = prev?.find(item => item?.id === id);
      if (removed?.preview) {
        URL.revokeObjectURL(removed?.preview);
      }
      return updated;
    });
  };

  const formatFileSize = (bytes) => {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i))?.toFixed(2)) + ' ' + sizes?.[i];
  };

  const getFileIcon = (type) => {
    if (type?.startsWith('image/')) return Image;
    if (type?.startsWith('video/')) return Video;
    return File;
  };

  return (
    <div>
      <h3 className="text-sm font-medium text-gray-900 mb-3">Media Attachments</h3>
      {/* Drop Zone */}
      <div
        className={`
          relative border-2 border-dashed rounded-lg p-6 text-center transition-colors
          ${dragActive 
            ? 'border-blue-400 bg-blue-50' :'border-gray-300 hover:border-gray-400'
          }
        `}
        onDragEnter={handleDrag}
        onDragLeave={handleDrag}
        onDragOver={handleDrag}
        onDrop={handleDrop}
      >
        <input
          type="file"
          multiple
          onChange={handleChange}
          accept="image/*,video/*"
          className="absolute inset-0 w-full h-full opacity-0 cursor-pointer"
        />
        
        <div className="space-y-2">
          <div className="mx-auto w-12 h-12 bg-gray-100 rounded-lg flex items-center justify-center">
            <Upload className="w-6 h-6 text-gray-400" />
          </div>
          <div>
            <p className="text-sm font-medium text-gray-900">Upload media files</p>
            <p className="text-xs text-gray-500">Drag & drop or click to select</p>
          </div>
          <p className="text-xs text-gray-400">Images, videos up to 10MB each</p>
        </div>
      </div>
      {/* Attached Media List */}
      {attachedMedia?.length > 0 && (
        <div className="mt-4 space-y-3">
          <h4 className="text-sm font-medium text-gray-700">Attached Files ({attachedMedia?.length})</h4>
          
          <div className="space-y-2">
            {attachedMedia?.map((media) => {
              const FileIcon = getFileIcon(media?.type);
              
              return (
                <div 
                  key={media?.id} 
                  className="flex items-center space-x-3 p-3 bg-gray-50 rounded-lg"
                >
                  {/* Preview or Icon */}
                  <div className="flex-shrink-0 w-10 h-10 bg-white rounded border overflow-hidden">
                    {media?.preview ? (
                      <img 
                        src={media?.preview} 
                        alt="Preview" 
                        className="w-full h-full object-cover"
                      />
                    ) : (
                      <div className="w-full h-full flex items-center justify-center">
                        <FileIcon className="w-5 h-5 text-gray-400" />
                      </div>
                    )}
                  </div>
                  {/* File Info */}
                  <div className="flex-1 min-w-0">
                    <p className="text-sm font-medium text-gray-900 truncate">
                      {media?.name}
                    </p>
                    <p className="text-xs text-gray-500">
                      {formatFileSize(media?.size)}
                    </p>
                  </div>
                  {/* Remove Button */}
                  <button
                    onClick={() => removeMedia(media?.id)}
                    className="flex-shrink-0 p-1 hover:bg-gray-200 rounded"
                  >
                    <X className="w-4 h-4 text-gray-400" />
                  </button>
                </div>
              );
            })}
          </div>
        </div>
      )}
      {/* Media Optimization Info */}
      {attachedMedia?.length > 0 && (
        <div className="mt-3 p-3 bg-blue-50 rounded-lg">
          <p className="text-xs text-blue-700">
            <strong>Auto-optimization:</strong> Media will be automatically resized and formatted for each selected platform's requirements.
          </p>
        </div>
      )}
    </div>
  );
};

export default MediaAttachment;