import React, { forwardRef, useRef, useEffect } from 'react';

const MainEditor = forwardRef(({ content, setContent, isMobile = false }, ref) => {
  const textareaRef = useRef(null);

  useEffect(() => {
    if (ref) {
      ref.current = textareaRef?.current;
    }
  }, [ref]);

  // Auto-resize textarea
  useEffect(() => {
    const textarea = textareaRef?.current;
    if (textarea) {
      textarea.style.height = 'auto';
      textarea.style.height = `${textarea?.scrollHeight}px`;
    }
  }, [content]);

  const handleKeyDown = (e) => {
    // Handle tab key for indentation
    if (e?.key === 'Tab') {
      e?.preventDefault();
      const start = e?.target?.selectionStart;
      const end = e?.target?.selectionEnd;
      const newContent = content?.substring(0, start) + '  ' + content?.substring(end);
      setContent(newContent);
      
      // Reset cursor position
      setTimeout(() => {
        e.target.selectionStart = e.target.selectionEnd = start + 2;
      }, 0);
    }
  };

  return (
    <div className="h-full flex flex-col">
      <textarea
        ref={textareaRef}
        value={content}
        onChange={(e) => setContent(e?.target?.value)}
        onKeyDown={handleKeyDown}
        placeholder="What's on your mind? Start writing your post here..."
        className={`
          w-full resize-none border-none outline-none text-gray-900 placeholder-gray-400
          ${isMobile ? 'text-base min-h-[200px]' : 'text-lg min-h-[300px]'}
        `}
        style={{
          fontFamily: '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif',
          lineHeight: '1.5'
        }}
        aria-label="Post content editor"
        role="textbox"
        aria-multiline="true"
      />
      {/* Helper text */}
      <div className="mt-4 text-sm text-gray-500">
        <p>Tips: Use @ for mentions, # for hashtags. Press Ctrl+S to save, Ctrl+N for new post.</p>
      </div>
    </div>
  );
});

MainEditor.displayName = 'MainEditor';

export default MainEditor;