import React, { useState, useEffect } from 'react';
import { Link, useLocation } from 'react-router-dom';
import Icon from '../AppIcon';
import Button from './Button';

const Header = () => {
  const [isMenuOpen, setIsMenuOpen] = useState(false);
  const [isScrolled, setIsScrolled] = useState(false);
  const location = useLocation();

  useEffect(() => {
    const handleScroll = () => {
      setIsScrolled(window.scrollY > 10);
    };

    window.addEventListener('scroll', handleScroll);
    return () => window.removeEventListener('scroll', handleScroll);
  }, []);

  const navigationItems = [
    { name: 'Features', path: '/product-features' },
    { name: 'Pricing', path: '/pricing' },
    { name: 'Stories', path: '/customer-stories' },
    { name: 'Resources', path: '/resources' },
  ];

  const isActivePath = (path) => {
    return location?.pathname === path;
  };

  const toggleMenu = () => {
    setIsMenuOpen(!isMenuOpen);
  };

  const closeMenu = () => {
    setIsMenuOpen(false);
  };

  return (
    <header 
      className={`fixed top-0 left-0 right-0 z-50 transition-all duration-300 ${
        isScrolled 
          ? 'bg-white/95 backdrop-blur-sm creative-confidence-shadow' 
          : 'bg-white'
      }`}
    >
      <div className="max-w-7xl mx-auto">
        <div className="flex items-center justify-between h-16 px-4 sm:px-6 lg:px-8">
          {/* Logo */}
          <Link 
            to="/homepage" 
            className="flex items-center space-x-2 micro-interaction hover:opacity-80"
            onClick={closeMenu}
          >
            <div className="w-8 h-8 bg-gradient-to-br from-primary to-secondary rounded-lg flex items-center justify-center">
              <Icon name="Zap" size={20} color="white" strokeWidth={2.5} />
            </div>
            <span className="text-xl font-bold text-text-primary">
              Social Composer Pro
            </span>
          </Link>

          {/* Desktop Navigation */}
          <nav className="hidden md:flex items-center space-x-8">
            {navigationItems?.map((item) => (
              <Link
                key={item?.name}
                to={item?.path}
                className={`text-sm font-medium micro-interaction hover:text-primary ${
                  isActivePath(item?.path)
                    ? 'text-primary border-b-2 border-primary pb-1' :'text-text-secondary hover:text-text-primary'
                }`}
              >
                {item?.name}
              </Link>
            ))}
          </nav>

          {/* Desktop CTA */}
          <div className="hidden md:flex items-center space-x-4">
            <Button 
              variant="ghost" 
              size="sm"
              className="text-text-secondary hover:text-text-primary"
            >
              Sign In
            </Button>
            <Button 
              variant="default" 
              size="sm"
              asChild
            >
              <Link to="/sign-up">Start Free Trial</Link>
            </Button>
          </div>

          {/* Mobile Menu Button */}
          <button
            onClick={toggleMenu}
            className="md:hidden p-2 rounded-lg micro-interaction hover:bg-muted"
            aria-label="Toggle menu"
          >
            <Icon 
              name={isMenuOpen ? "X" : "Menu"} 
              size={24} 
              color="currentColor" 
            />
          </button>
        </div>

        {/* Mobile Menu */}
        <div 
          className={`md:hidden transition-all duration-300 ease-smooth ${
            isMenuOpen 
              ? 'max-h-96 opacity-100' :'max-h-0 opacity-0 overflow-hidden'
          }`}
        >
          <div className="px-4 py-4 bg-white border-t border-border">
            <nav className="flex flex-col space-y-4">
              {navigationItems?.map((item) => (
                <Link
                  key={item?.name}
                  to={item?.path}
                  onClick={closeMenu}
                  className={`text-base font-medium micro-interaction hover:text-primary ${
                    isActivePath(item?.path)
                      ? 'text-primary' :'text-text-secondary hover:text-text-primary'
                  }`}
                >
                  {item?.name}
                </Link>
              ))}
              <div className="pt-4 border-t border-border">
                <div className="flex flex-col space-y-3">
                  <Button 
                    variant="ghost" 
                    size="sm"
                    fullWidth
                    className="justify-start text-text-secondary hover:text-text-primary"
                  >
                    Sign In
                  </Button>
                  <Button 
                    variant="default" 
                    size="sm"
                    fullWidth
                    asChild
                  >
                    <Link to="/sign-up" onClick={closeMenu}>
                      Start Free Trial
                    </Link>
                  </Button>
                </div>
              </div>
            </nav>
          </div>
        </div>
      </div>
    </header>
  );
};

export default Header;