<?php

namespace App\Services;

use App\Models\AppUser;
use App\Models\Post;
use App\Models\PostMedia;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\File;

class PostMediaService
{
    /**
     * Store uploaded files for a post and return the created media records.
     *
     * @param  UploadedFile[]  $files
     * @return array<PostMedia>
     */
    public function storeFiles(Post $post, array $files, AppUser $user, ?int $startingDisplayOrder = null): array
    {
        if (empty($files)) {
            return [];
        }

        $created = [];
        $currentOrder = $startingDisplayOrder;

        if ($currentOrder === null) {
            $maxOrder = $post->media()->max('display_order');
            $currentOrder = is_null($maxOrder) ? 0 : $maxOrder + 1;
        }

        foreach ($files as $file) {
            if (!$file instanceof UploadedFile) {
                continue;
            }

            $relativeDirectory = "post_uploads/{$post->id}";
            $absoluteDirectory = public_path($relativeDirectory);

            if (!File::exists($absoluteDirectory)) {
                File::makeDirectory($absoluteDirectory, 0755, true);
            }

            $originalName = $file->getClientOriginalName();
            $clientMimeType = $file->getClientMimeType();
            $detectedMimeType = $file->getMimeType();
            $fileSize = $file->getSize();

            $fileName = $file->hashName();
            $file->move($absoluteDirectory, $fileName);

            $relativePath = "{$relativeDirectory}/{$fileName}";

            $created[] = PostMedia::create([
                'post_id' => $post->id,
                'app_user_id' => $user->id,
                'file_name' => $originalName,
                'file_path' => $relativePath,
                'file_url' => asset($relativePath),
                'file_type' => $clientMimeType,
                'file_size' => $fileSize,
                'mime_type' => $detectedMimeType,
                'display_order' => $currentOrder++,
            ]);
        }

        return $created;
    }

    /**
     * Delete the given media record and remove the underlying file.
     */
    public function deleteMedia(PostMedia $media): void
    {
        if ($media->file_path) {
            $absolutePath = public_path($media->file_path);
            if (File::exists($absolutePath)) {
                File::delete($absolutePath);
            }
        }

        $media->delete();
    }

    /**
     * Delete multiple media records.
     *
     * @param  Collection<int, PostMedia>  $mediaCollection
     */
    public function deleteMany(Collection $mediaCollection): void
    {
        $mediaCollection->each(function (PostMedia $media) {
            $this->deleteMedia($media);
        });
    }
}

