<?php

namespace App\Services\Admin;

use App\Models\SubscriptionPlan;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Collection;

class PlanService
{
    /**
     * Get all subscription plans with pagination.
     */
    public function getAllPlans(int $perPage = 20): LengthAwarePaginator
    {
        return SubscriptionPlan::orderBy('created_at', 'desc')->paginate($perPage);
    }

    /**
     * Get active subscription plans.
     */
    public function getActivePlans(): Collection
    {
        return SubscriptionPlan::active()->orderBy('price', 'asc')->get();
    }

    /**
     * Create a new subscription plan.
     */
    public function createPlan(array $data): SubscriptionPlan
    {
        return SubscriptionPlan::create($data);
    }

    /**
     * Update an existing subscription plan.
     */
    public function updatePlan(SubscriptionPlan $plan, array $data): bool
    {
        return $plan->update($data);
    }

    /**
     * Delete a subscription plan.
     */
    public function deletePlan(SubscriptionPlan $plan): bool
    {
        // Check if plan has active subscriptions
        if ($plan->hasActiveSubscriptions()) {
            throw new \Exception('Cannot delete plan with active subscriptions.');
        }

        return $plan->delete();
    }

    /**
     * Get plan statistics.
     */
    public function getPlanStatistics(): array
    {
        return [
            'total_plans' => SubscriptionPlan::count(),
            'active_plans' => SubscriptionPlan::active()->count(),
            'inactive_plans' => SubscriptionPlan::inactive()->count(),
            'total_subscribers' => SubscriptionPlan::withCount('subscriptions')->get()->sum('subscriptions_count'),
            'total_revenue' => SubscriptionPlan::with('subscriptions')->get()->sum('total_revenue'),
        ];
    }

    /**
     * Get plans with subscription counts.
     */
    public function getPlansWithStats(): Collection
    {
        return SubscriptionPlan::withCount(['subscriptions', 'activeSubscriptions'])
            ->orderBy('created_at', 'desc')
            ->get();
    }

    /**
     * Search plans by name.
     */
    public function searchPlans(string $query, int $perPage = 20): LengthAwarePaginator
    {
        return SubscriptionPlan::where('name', 'like', "%{$query}%")
            ->orderBy('created_at', 'desc')
            ->paginate($perPage);
    }

    /**
     * Get plans by status.
     */
    public function getPlansByStatus(string $status, int $perPage = 20): LengthAwarePaginator
    {
        return SubscriptionPlan::where('status', $status)
            ->orderBy('created_at', 'desc')
            ->paginate($perPage);
    }

    /**
     * Toggle plan status.
     */
    public function togglePlanStatus(SubscriptionPlan $plan): bool
    {
        $newStatus = $plan->status === 'active' ? 'inactive' : 'active';
        
        return $plan->update(['status' => $newStatus]);
    }

    /**
     * Duplicate a plan.
     */
    public function duplicatePlan(SubscriptionPlan $plan): SubscriptionPlan
    {
        $newPlanData = $plan->toArray();
        unset($newPlanData['id'], $newPlanData['created_at'], $newPlanData['updated_at']);
        
        $newPlanData['name'] = $plan->name . ' (Copy)';
        $newPlanData['status'] = 'inactive';

        return $this->createPlan($newPlanData);
    }
}
