<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class SubscriptionPlan extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'price',
        'duration',
        'max_posts_per_month',
        'max_social_accounts',
        'ai_credits',
        'features',
        'status',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'price' => 'decimal:2',
        'max_posts_per_month' => 'integer',
        'max_social_accounts' => 'integer',
        'ai_credits' => 'integer',
    ];

    /**
     * Get the subscriptions for the plan.
     */
    public function subscriptions(): HasMany
    {
        return $this->hasMany(Subscription::class);
    }

    /**
     * Get the active subscriptions for the plan.
     */
    public function activeSubscriptions(): HasMany
    {
        return $this->subscriptions()->where('status', 'active');
    }

    /**
     * Scope a query to only include active plans.
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    /**
     * Scope a query to only include inactive plans.
     */
    public function scopeInactive($query)
    {
        return $query->where('status', 'inactive');
    }

    /**
     * Get the formatted price attribute.
     */
    public function getFormattedPriceAttribute(): string
    {
        return '₹' . number_format($this->price, 2);
    }

    /**
     * Get the features as an array.
     */
    public function getFeaturesArrayAttribute(): array
    {
        if (empty($this->features)) {
            return [];
        }

        return array_filter(explode("\n", $this->features));
    }

    /**
     * Check if the plan has active subscriptions.
     */
    public function hasActiveSubscriptions(): bool
    {
        return $this->activeSubscriptions()->exists();
    }

    /**
     * Get the total number of subscribers.
     */
    public function getTotalSubscribersAttribute(): int
    {
        return $this->subscriptions()->count();
    }

    /**
     * Get the total revenue from this plan.
     */
    public function getTotalRevenueAttribute(): float
    {
        return $this->subscriptions()
            ->where('status', 'active')
            ->sum('amount_paid') ?? 0;
    }
}
