<?php

namespace App\Jobs;

use App\Models\Post;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class PublishScheduledPosts implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * Create a new job instance.
     */
    public function __construct()
    {
        //
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        $now = Carbon::now();

        // Find posts scheduled to publish now or in the past
        $posts = Post::where('status', 'scheduled')
            ->whereNotNull('scheduled_at')
            ->where('scheduled_at', '<=', $now)
            ->get();

        if ($posts->isEmpty()) {
            Log::info('PublishScheduledPosts: No scheduled posts found to publish');
            return;
        }

        Log::info('PublishScheduledPosts: Found ' . $posts->count() . ' scheduled post(s) to publish');

        foreach ($posts as $post) {
            try {
                // Update post status to posted
                $post->status = 'posted';
                $post->published_at = $post->scheduled_at ?? now();
                $post->scheduled_at = null; // Clear scheduled_at after publishing
                $post->save();

                Log::info('PublishScheduledPosts: Published post ID ' . $post->id, [
                    'post_id' => $post->id,
                    'published_at' => $post->published_at,
                ]);

                // TODO: Add social media publishing logic here
                // This is where you would integrate with Facebook, Twitter, LinkedIn APIs
                // Example:
                // if ($post->platforms) {
                //     foreach ($post->platforms as $platform) {
                //         $this->publishToPlatform($post, $platform);
                //     }
                // }
            } catch (\Exception $e) {
                Log::error('PublishScheduledPosts: Failed to publish post ID ' . $post->id, [
                    'post_id' => $post->id,
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString(),
                ]);
            }
        }
    }
}

