<?php

namespace App\Http\Requests\Api;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\ValidationException;

class CompleteProfileRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true; // User must be authenticated via middleware
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        $user = $this->user();
        $userId = $user ? $user->id : null;

        return [
            // Basic Information (Required in complete profile)
            'phone' => 'required|string|max:15|unique:app_users,phone,' . $userId,
            
            // Company/Shop Information (Required after activation)
            'company_name' => 'required|string|max:255',
            'company_description' => 'nullable|string|max:1000',
            
            // Contact Information
            'website' => 'nullable|url|max:255',
            'address' => 'nullable|string|max:500',
            'city' => 'nullable|string|max:100',
            'state' => 'nullable|string|max:100',
            'country' => 'nullable|string|max:100',
            'postal_code' => 'nullable|string|max:20',
            
            // Social Media & Marketing
            'social_media_handles' => 'nullable|array',
            'social_media_handles.facebook' => 'nullable|string|max:255',
            'social_media_handles.instagram' => 'nullable|string|max:255',
            'social_media_handles.twitter' => 'nullable|string|max:255',
            'social_media_handles.linkedin' => 'nullable|string|max:255',
            'social_media_handles.youtube' => 'nullable|string|max:255',
            'preferred_social_platforms' => 'nullable|array',
            'preferred_social_platforms.*' => 'string|in:facebook,instagram,twitter,linkedin,youtube,tiktok,pinterest',
        ];
    }

    /**
     * Get custom messages for validator errors.
     *
     * @return array<string, string>
     */
    public function messages(): array
    {
        return [
            'phone.required' => 'Phone number is required.',
            'phone.unique' => 'This phone number is already registered by another user.',
            'phone.max' => 'Phone number must not exceed 15 characters.',
            'company_name.required' => 'Company/Shop name is required.',
            'company_description.max' => 'Company description must not exceed 1000 characters.',
            'website.url' => 'Please enter a valid website URL (e.g., https://example.com).',
            'preferred_social_platforms.*.in' => 'Please select valid social media platforms.',
        ];
    }

    /**
     * Handle a failed validation attempt.
     *
     * @param  \Illuminate\Contracts\Validation\Validator  $validator
     * @return void
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    protected function failedValidation(\Illuminate\Contracts\Validation\Validator $validator)
    {
        throw new ValidationException($validator);
    }
}

