<?php

namespace App\Http\Requests\Admin;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

class UpdatePlanRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return auth()->check() && auth()->user()->hasRole('admin');
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        $planId = $this->route('plan');
        
        return [
            'name' => [
                'required',
                'string',
                'max:100',
                Rule::unique('subscription_plans', 'name')->ignore($planId)
            ],
            'price' => 'required|numeric|min:0|max:999999.99',
            'duration' => 'required|in:monthly,yearly',
            'max_posts_per_month' => 'required|integer|min:0|max:999999',
            'max_social_accounts' => 'required|integer|min:0|max:100',
            'ai_credits' => 'required|integer|min:0|max:999999',
            'features' => 'nullable|string|max:1000',
            'status' => 'required|in:active,inactive',
        ];
    }

    /**
     * Get custom messages for validator errors.
     *
     * @return array<string, string>
     */
    public function messages(): array
    {
        return [
            'name.required' => 'Plan name is required.',
            'name.unique' => 'A plan with this name already exists.',
            'price.required' => 'Price is required.',
            'price.numeric' => 'Price must be a valid number.',
            'price.min' => 'Price must be at least 0.',
            'duration.required' => 'Duration is required.',
            'duration.in' => 'Duration must be either monthly or yearly.',
            'max_posts_per_month.required' => 'Maximum posts per month is required.',
            'max_posts_per_month.integer' => 'Maximum posts must be a whole number.',
            'max_social_accounts.required' => 'Maximum social accounts is required.',
            'max_social_accounts.integer' => 'Maximum social accounts must be a whole number.',
            'ai_credits.required' => 'AI credits is required.',
            'ai_credits.integer' => 'AI credits must be a whole number.',
            'status.required' => 'Status is required.',
            'status.in' => 'Status must be either active or inactive.',
        ];
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array<string, string>
     */
    public function attributes(): array
    {
        return [
            'name' => 'plan name',
            'price' => 'price',
            'duration' => 'duration',
            'max_posts_per_month' => 'maximum posts per month',
            'max_social_accounts' => 'maximum social accounts',
            'ai_credits' => 'AI credits',
            'features' => 'features',
            'status' => 'status',
        ];
    }
}
