<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\AppUser;
use App\Models\Post;
use App\Models\PostMedia;
use App\Services\PostMediaService;
use Illuminate\Http\Exceptions\HttpResponseException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class PostMediaController extends Controller
{
    public function __construct(private readonly PostMediaService $postMediaService)
    {
    }

    public function store(Request $request, Post $post): JsonResponse
    {
        /** @var AppUser $user */
        $user = $request->user();
        $this->assertOwner($user, $post);
        $this->assertDraft($post);

        $validated = $request->validate([
            'files' => ['required', 'array'],
            'files.*' => ['file', 'max:10240'],
            'display_order' => ['nullable', 'integer', 'min:0'],
        ]);

        $files = $request->file('files', []);
        $startingOrder = $validated['display_order'] ?? null;

        $media = $this->postMediaService->storeFiles($post, $files, $user, $startingOrder);

        return response()->json([
            'success' => true,
            'message' => 'Media uploaded successfully',
            'data' => $media,
        ], 201);
    }

    public function destroy(Request $request, Post $post, PostMedia $media): JsonResponse
    {
        /** @var AppUser $user */
        $user = $request->user();
        $this->assertOwner($user, $post);
        $this->assertDraft($post);

        if ($media->post_id !== $post->id) {
            throw new HttpResponseException(response()->json([
                'success' => false,
                'error' => 'Media does not belong to this post',
            ], 404));
        }

        $this->postMediaService->deleteMedia($media);

        return response()->json([
            'success' => true,
            'message' => 'Media deleted successfully',
        ]);
    }

    private function assertOwner(AppUser $user, Post $post): void
    {
        if ($post->app_user_id !== $user->id) {
            throw new HttpResponseException(response()->json([
                'success' => false,
                'error' => "You don't have permission to access this post",
            ], 403));
        }
    }

    private function assertDraft(Post $post): void
    {
        if ($post->status !== 'draft') {
            throw new HttpResponseException(response()->json([
                'success' => false,
                'error' => 'Only draft posts can be modified.',
            ], 400));
        }
    }
}

