<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\AppUser;
use App\Models\Post;
use App\Models\PostMedia;
use App\Services\PostMediaService;
use Illuminate\Http\Exceptions\HttpResponseException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;

class PostController extends Controller
{
    public function __construct(private readonly PostMediaService $postMediaService)
    {
    }

    public function index(Request $request): JsonResponse
    {
        /** @var AppUser $user */
        $user = $request->user();

        $perPage = (int) $request->query('per_page', 20);
        $perPage = max(1, min($perPage, 100));

        $query = Post::with(['media' => fn ($q) => $q->orderBy('display_order')])
            ->withCount('media')
            ->where('app_user_id', $user->id);

        if ($status = $request->query('status')) {
            if (in_array($status, ['draft', 'posted'], true)) {
                $query->where('status', $status);
            }
        }

        if ($category = $request->query('category')) {
            $query->where('category', $category);
        }

        $allowedSorts = ['created_at', 'updated_at', 'published_at'];
        $sort = $request->query('sort', 'created_at');
        if (!in_array($sort, $allowedSorts, true)) {
            $sort = 'created_at';
        }

        $order = strtolower($request->query('order', 'desc'));
        if (!in_array($order, ['asc', 'desc'], true)) {
            $order = 'desc';
        }

        $paginator = $query
            ->orderBy($sort, $order)
            ->paginate($perPage)
            ->appends($request->query());

        return response()->json([
            'success' => true,
            'data' => $paginator->items(),
            'meta' => [
                'current_page' => $paginator->currentPage(),
                'per_page' => $paginator->perPage(),
                'total' => $paginator->total(),
                'last_page' => $paginator->lastPage(),
                'from' => $paginator->firstItem(),
                'to' => $paginator->lastItem(),
            ],
        ]);
    }

    public function store(Request $request): JsonResponse
    {
        /** @var AppUser $user */
        $user = $request->user();

        // Debug: Log all incoming request data BEFORE validation
        \Log::info('Post Store - Request Data (BEFORE validation)', [
            'request_method' => $request->method(),
            'content_type' => $request->header('Content-Type'),
            'request_all_keys' => array_keys($request->all()),
            'request_all' => $request->all(),
            'all_files_keys' => array_keys($request->allFiles()),
            'all_files_count' => count($request->allFiles()),
            'has_file_files' => $request->hasFile('files'),
            'file_files' => $request->file('files') ? 'exists' : 'null',
        ]);

        $validated = $request->validate([
            'content' => ['required', 'string', 'max:5000'],
            'content_html' => ['nullable', 'string', 'max:10000'],
            'status' => ['required', 'in:draft,posted'],
            'category' => ['nullable', 'string', 'max:100'],
            'platforms' => ['nullable'],
            'files' => ['sometimes', 'array'],
            'files.*' => ['file', 'mimes:jpeg,jpg,png,gif,mp4,mov,avi', 'max:10240'],
        ]);

        // Debug: Log after validation
        \Log::info('Post Store - After Validation', [
            'validated_keys' => array_keys($validated),
            'validated_has_files' => array_key_exists('files', $validated),
        ]);

        $platforms = $this->normalizePlatforms($validated['platforms'] ?? null);
        $category = array_key_exists('category', $validated) && $validated['category'] !== null
            ? $validated['category']
            : 'General';
        $status = $validated['status'];
        $publishedAt = $status === 'posted' ? now() : null;

        $files = $this->collectUploadedFiles($request);

        // Debug: Log file collection
        \Log::info('Post Store - Files collected', [
            'files_count' => count($files),
            'all_files_keys' => array_keys($request->allFiles()),
            'request_method' => $request->method(),
            'file_details' => array_map(function ($file) {
                return $file instanceof UploadedFile ? [
                    'name' => $file->getClientOriginalName(),
                    'size' => $file->getSize(),
                    'mime' => $file->getMimeType(),
                ] : 'not_uploaded_file';
            }, $files),
        ]);

        /** @var Post $post */
        $post = DB::transaction(function () use ($user, $validated, $platforms, $category, $status, $publishedAt, $files) {
            $post = Post::create([
                'app_user_id' => $user->id,
                'content' => $validated['content'],
                'content_html' => $validated['content_html'] ?? null,
                'status' => $status,
                'category' => $category,
                'platforms' => $platforms,
                'published_at' => $publishedAt,
            ]);

            $this->postMediaService->storeFiles($post, $files, $user);

            return $post->load(['media' => fn ($q) => $q->orderBy('display_order')]);
        });

        return response()->json([
            'success' => true,
            'message' => 'Post created successfully',
            'data' => $post,
        ], 201);
    }

    public function show(Request $request, Post $post): JsonResponse
    {
        /** @var AppUser $user */
        $user = $request->user();
        $this->assertOwner($user, $post);

        $post->load(['media' => fn ($q) => $q->orderBy('display_order')]);

        return response()->json([
            'success' => true,
            'data' => $post,
        ]);
    }

    public function update(Request $request, Post $post): JsonResponse
    {
        /** @var AppUser $user */
        $user = $request->user();
        $this->assertOwner($user, $post);
        // Allow updating draft and scheduled posts
        $this->assertDraft($post);

        // Normalize remove_media_ids before validation (handles JSON strings from FormData)
        $removeMediaIds = $request->input('remove_media_ids');
        if ($removeMediaIds !== null && !is_array($removeMediaIds)) {
            if (is_string($removeMediaIds) && $removeMediaIds !== '') {
                // Trim whitespace first
                $removeMediaIds = trim($removeMediaIds);
                
                // Try to decode as JSON first (handles "[12]", "[1,2,3]", etc.)
                $decoded = json_decode($removeMediaIds, true);
                if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                    $removeMediaIds = $decoded;
                } else {
                    // Try comma-separated string (handles "1,2,3" or "12")
                    $parts = array_filter(array_map('trim', explode(',', $removeMediaIds)));
                    $removeMediaIds = array_values(array_filter(array_map('intval', $parts), fn($v) => $v > 0));
                }
            } else {
                // Empty string or invalid type - set to null
                $removeMediaIds = null;
            }
            // Replace the input value with normalized array (or null)
            $request->merge(['remove_media_ids' => $removeMediaIds]);
        }

        // Debug: Log all incoming request data BEFORE validation
        \Log::info('Post Update - Request Data (BEFORE validation)', [
            'post_id' => $post->id,
            'request_method' => $request->method(),
            'content_type' => $request->header('Content-Type'),
            'request_all_keys' => array_keys($request->all()),
            'request_all' => $request->all(),
            'all_files_keys' => array_keys($request->allFiles()),
            'all_files_count' => count($request->allFiles()),
            'has_file_files' => $request->hasFile('files'),
            'file_files' => $request->file('files') ? 'exists' : 'null',
        ]);

        // Validate request data - INCLUDING files (same as store method)
        // This ensures Laravel properly processes the files
        $validated = $request->validate([
            'content' => ['sometimes', 'required', 'string', 'max:5000'],
            'content_html' => ['nullable', 'string', 'max:10000'],
            'status' => ['sometimes', 'required', 'in:draft,scheduled,posted'],
            'category' => ['nullable', 'string', 'max:100'],
            'platforms' => ['nullable'],
            'files' => ['sometimes', 'array'],
            'files.*' => ['file', 'mimes:jpeg,jpg,png,gif,mp4,mov,avi', 'max:10240'],
            'remove_media_ids' => ['nullable', 'array'],
            'remove_media_ids.*' => ['integer', Rule::exists('post_media', 'id')],
            'scheduled_at' => ['nullable', 'date', 'after:now'],
        ]);

        // Debug: Log after validation
        \Log::info('Post Update - After Validation', [
            'post_id' => $post->id,
            'validated_keys' => array_keys($validated),
            'validated_has_files' => array_key_exists('files', $validated),
        ]);

        $platforms = array_key_exists('platforms', $validated)
            ? $this->normalizePlatforms($validated['platforms'])
            : $post->platforms;

        // Collect files AFTER validation (same as store method)
        // This works because validation processes the files and makes them available
        $files = $this->collectUploadedFiles($request);

        // Debug: Log file collection
        \Log::info('Post Update - Files collected', [
            'post_id' => $post->id,
            'files_count' => count($files),
            'all_files_keys' => array_keys($request->allFiles()),
            'request_method' => $request->method(),
            'file_details' => array_map(function ($file) {
                return $file instanceof UploadedFile ? [
                    'name' => $file->getClientOriginalName(),
                    'size' => $file->getSize(),
                    'mime' => $file->getMimeType(),
                ] : 'not_uploaded_file';
            }, $files),
        ]);

        DB::transaction(function () use ($post, $validated, $platforms, $files, $user) {
            $updates = [];

            foreach (['content', 'content_html', 'category'] as $field) {
                if (array_key_exists($field, $validated)) {
                    $updates[$field] = $validated[$field];
                }
            }

            if (array_key_exists('platforms', $validated)) {
                $updates['platforms'] = $platforms;
            }

            // Handle status field if provided
            if (array_key_exists('status', $validated)) {
                $updates['status'] = $validated['status'];
                // If status is 'posted', set published_at and clear scheduled_at
                if ($validated['status'] === 'posted') {
                    $updates['published_at'] = now();
                    $updates['scheduled_at'] = null;
                } elseif ($validated['status'] === 'scheduled') {
                    // If status is 'scheduled', clear published_at
                    $updates['published_at'] = null;
                } elseif ($validated['status'] === 'draft') {
                    // If status is 'draft', clear both published_at and scheduled_at
                    $updates['published_at'] = null;
                    $updates['scheduled_at'] = null;
                }
            }

            // Handle scheduled_at field if provided
            if (array_key_exists('scheduled_at', $validated)) {
                if ($validated['scheduled_at'] !== null) {
                    $updates['scheduled_at'] = $validated['scheduled_at'];
                    // If scheduled_at is set, ensure status is 'scheduled'
                    if (!isset($updates['status']) || $updates['status'] !== 'scheduled') {
                        $updates['status'] = 'scheduled';
                        $updates['published_at'] = null;
                    }
                } else {
                    // If scheduled_at is null, clear it
                    $updates['scheduled_at'] = null;
                    // If status was 'scheduled', change to 'draft'
                    if ($post->status === 'scheduled' && !isset($updates['status'])) {
                        $updates['status'] = 'draft';
                    }
                }
            }

            if (!empty($updates)) {
                $post->fill($updates);
                $post->save();
            }

            if (!empty($validated['remove_media_ids'])) {
                $mediaToRemove = PostMedia::where('post_id', $post->id)
                    ->whereIn('id', $validated['remove_media_ids'])
                    ->get();

                $this->postMediaService->deleteMany($mediaToRemove);
            }

            // Store new files if any were uploaded
            if (!empty($files)) {
                $this->postMediaService->storeFiles($post, $files, $user);
            }
        });

        $post->load(['media' => fn ($q) => $q->orderBy('display_order')]);

        return response()->json([
            'success' => true,
            'message' => 'Post updated successfully',
            'data' => $post,
        ]);
    }

    public function destroy(Request $request, Post $post): JsonResponse
    {
        /** @var AppUser $user */
        $user = $request->user();
        $this->assertOwner($user, $post);

        DB::transaction(function () use ($post) {
            $post->load('media');
            $post->media->each(function (PostMedia $media) {
                $this->postMediaService->deleteMedia($media);
            });

            $post->delete();
        });

        return response()->json([
            'success' => true,
            'message' => 'Post deleted successfully',
        ]);
    }

    public function publish(Request $request, Post $post): JsonResponse
    {
        /** @var AppUser $user */
        $user = $request->user();
        $this->assertOwner($user, $post);
        $this->assertDraft($post);

        $validated = $request->validate([
            'platforms' => ['nullable'],
            'scheduled_at' => ['nullable', 'date', 'after:now'],
        ]);

        $platforms = array_key_exists('platforms', $validated)
            ? $this->normalizePlatforms($validated['platforms'])
            : $post->platforms;

        // Check if scheduling for later
        if (isset($validated['scheduled_at']) && $validated['scheduled_at'] !== null) {
            // Schedule for later
            $post->update([
                'status' => 'scheduled',
                'platforms' => $platforms,
                'scheduled_at' => $validated['scheduled_at'],
                'published_at' => null, // Clear published_at if it was set
            ]);

            $post->load(['media' => fn ($q) => $q->orderBy('display_order')]);

            return response()->json([
                'success' => true,
                'message' => 'Post scheduled successfully',
                'data' => $post,
            ]);
        }

        // Publish immediately
        $post->update([
            'status' => 'posted',
            'platforms' => $platforms,
            'published_at' => now(),
            'scheduled_at' => null, // Clear scheduled_at if it was set
        ]);

        $post->load(['media' => fn ($q) => $q->orderBy('display_order')]);

        return response()->json([
            'success' => true,
            'message' => 'Post published successfully',
            'data' => $post,
        ]);
    }

    private function assertOwner(AppUser $user, Post $post): void
    {
        if ($post->app_user_id !== $user->id) {
            throw new HttpResponseException(response()->json([
                'success' => false,
                'error' => "You don't have permission to access this post",
            ], 403));
        }
    }

    private function assertDraft(Post $post): void
    {
        if (!in_array($post->status, ['draft', 'scheduled'], true)) {
            throw new HttpResponseException(response()->json([
                'success' => false,
                'error' => "Cannot update post with status '{$post->status}'. Only draft and scheduled posts can be updated.",
            ], 400));
        }
    }

    private function normalizePlatforms(mixed $platforms): ?array
    {
        if ($platforms === null || $platforms === '') {
            return null;
        }

        if (is_string($platforms)) {
            $decoded = json_decode($platforms, true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                $platforms = $decoded;
            } else {
                $platforms = [$platforms];
            }
        }

        if ($platforms instanceof \Illuminate\Support\Collection) {
            $platforms = $platforms->all();
        }

        if (!is_array($platforms)) {
            return null;
        }

        $filtered = array_values(array_filter(array_map('strval', $platforms), function ($value) {
            return $value !== '';
        }));

        return empty($filtered) ? null : $filtered;
    }

    /**
     * Log the structure of files in the request for debugging.
     */
    private function logFilesStructure(array $allFiles): array
    {
        $structure = [];
        foreach ($allFiles as $key => $value) {
            if ($value instanceof UploadedFile) {
                $structure[$key] = [
                    'type' => 'UploadedFile',
                    'name' => $value->getClientOriginalName(),
                    'size' => $value->getSize(),
                    'mime' => $value->getMimeType(),
                ];
            } elseif (is_array($value)) {
                $structure[$key] = [
                    'type' => 'array',
                    'count' => count($value),
                    'items' => array_map(function ($item, $idx) {
                        if ($item instanceof UploadedFile) {
                            return [
                                'index' => $idx,
                                'type' => 'UploadedFile',
                                'name' => $item->getClientOriginalName(),
                                'size' => $item->getSize(),
                            ];
                        }
                        return [
                            'index' => $idx,
                            'type' => gettype($item),
                            'class' => is_object($item) ? get_class($item) : null,
                        ];
                    }, $value, array_keys($value)),
                ];
            } else {
                $structure[$key] = [
                    'type' => gettype($value),
                    'class' => is_object($value) ? get_class($value) : null,
                    'value' => (string) $value,
                ];
            }
        }
        return $structure;
    }

    /**
     * Normalize the uploaded files payload into a flat array.
     * Handles various file upload formats including PUT requests with multipart/form-data.
     *
     * @return UploadedFile[]
     */
    private function collectUploadedFiles(Request $request, string $key = 'files'): array
    {
        // In multipart/form-data, when frontend sends 'files[]', Laravel parses it as 'files' (array)
        // Try multiple variations to catch all cases
        
        // 1. Try 'files' (standard key - Laravel automatically handles 'files[]' as 'files')
        $files = $request->file($key);
        
        // 2. Try 'files[]' explicitly (some edge cases)
        if ($files === null) {
            $files = $request->file($key . '[]');
        }
        
        // 3. Most reliable: Check allFiles() and collect any files that match our key pattern
        // This works even if the key has brackets or variations, and handles PUT requests better
        $allFiles = $request->allFiles();
        if (!empty($allFiles)) {
            $collected = [];
            foreach ($allFiles as $fileKey => $value) {
                // Match 'files', 'files[]', or any variation
                $normalizedKey = rtrim($fileKey, '[]');
                if ($normalizedKey === $key) {
                    if ($value instanceof UploadedFile) {
                        $collected[] = $value;
                    } elseif (is_array($value)) {
                        foreach ($value as $file) {
                            if ($file instanceof UploadedFile) {
                                $collected[] = $file;
                            }
                        }
                    }
                }
            }
            if (!empty($collected)) {
                return $collected;
            }
        }
        
        // 4. If we found files via direct access, process them
        if ($files !== null) {
            if ($files instanceof UploadedFile) {
                return [$files];
            }
            if (is_array($files)) {
                return array_values(array_filter($files, fn ($file) => $file instanceof UploadedFile));
            }
        }

        // 5. For PUT/PATCH requests, also check if files are in the request input
        // Some clients send files differently with PUT requests
        if (in_array($request->method(), ['PUT', 'PATCH'], true)) {
            $inputFiles = $request->input($key);
            if (is_array($inputFiles)) {
                $collected = [];
                foreach ($inputFiles as $file) {
                    if ($file instanceof UploadedFile) {
                        $collected[] = $file;
                    }
                }
                if (!empty($collected)) {
                    return $collected;
                }
            }
        }

        return [];
    }
}
