<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Category;
use Illuminate\Http\Request;

class CategoryController extends Controller
{
    /**
     * Get list of categories
     * 
     * Returns default categories and categories created by the authenticated user
     * Requires authentication
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        $user = $request->user();
        
        // Get default categories (is_default = 1) 
        // and categories created by the authenticated user (user_id = authenticated user's id)
        $categories = Category::where(function ($query) use ($user) {
            // Default categories: is_default = 1
            $query->where('is_default', '1')
            // OR user's own categories: user_id matches authenticated user
            ->orWhere('user_id', $user->id);
        })
        ->orderBy('category_name', 'asc')
        ->get();

        return response()->json([
            'success' => true,
            'message' => 'Categories retrieved successfully.',
            'data' => [
                'categories' => $categories->map(function ($category) {
                    return [
                        'id' => $category->id,
                        'category_name' => $category->category_name,
                        'is_default' => (bool) $category->is_default,
                        'user_id' => $category->user_id,
                        'is_own_category' => $category->user_id !== null, // Indicates if it's user's own category
                        'created_at' => $category->created_at?->toIso8601String(),
                        'updated_at' => $category->updated_at?->toIso8601String(),
                    ];
                })
            ]
        ], 200);
    }

    /**
     * Create a new category
     * 
     * Creates a category for the authenticated user
     * Requires authentication
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        // Validate the request
        $validated = $request->validate([
            'category_name' => 'required|string|max:255',
        ]);

        $user = $request->user();

        // Create the category with user_id and is_default = 0 for user-created categories
        $category = Category::create([
            'category_name' => $validated['category_name'],
            'is_default' => '0', // User-created categories are not default
            'user_id' => $user->id,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Category created successfully.',
            'data' => [
                'category' => [
                    'id' => $category->id,
                    'category_name' => $category->category_name,
                    'is_default' => (bool) $category->is_default,
                    'user_id' => $category->user_id,
                    'is_own_category' => true,
                    'created_at' => $category->created_at?->toIso8601String(),
                    'updated_at' => $category->updated_at?->toIso8601String(),
                ]
            ]
        ], 201);
    }

    /**
     * Delete a category
     * 
     * Deletes a category that belongs to the authenticated user
     * Users can only delete their own categories, not default categories
     * Requires authentication
     * 
     * @param Request $request
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy(Request $request, $id)
    {
        $user = $request->user();

        // Find the category
        $category = Category::find($id);

        // Check if category exists
        if (!$category) {
            return response()->json([
                'success' => false,
                'message' => 'Category not found.',
            ], 404);
        }

        // Check if the category belongs to the authenticated user
        // Users can only delete their own categories (not default categories)
        if ($category->user_id !== $user->id) {
            return response()->json([
                'success' => false,
                'message' => 'You do not have permission to delete this category.',
            ], 403);
        }

        // Delete the category
        $category->delete();

        return response()->json([
            'success' => true,
            'message' => 'Category deleted successfully.',
        ], 200);
    }
}

