<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\Api\LoginRequest;
use App\Http\Requests\Api\RegisterRequest;
use App\Http\Requests\Api\ForgotPasswordRequest;
use App\Http\Requests\Api\ResetPasswordRequest;
use App\Http\Requests\Api\ResendVerificationRequest;
use App\Http\Requests\Api\CompleteProfileRequest;
use App\Http\Requests\Api\UpdateProfileRequest;
use Illuminate\Http\Request;
use App\Models\AppUser;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Carbon\Carbon;
use App\Mail\VerifyEmailMail;
use App\Mail\ResetPasswordMail;

class AuthController extends Controller
{
    /**
     * Handle user registration (Step 1: Minimal Registration)
     * 
     * Flow:
     * 1. User registers with minimal info (name, email, password)
     * 2. Account created with status 'inactive'
     * 3. Verification email sent with activation link
     * 4. After email verification, status becomes 'active'
     * 5. User can then complete profile with phone and company details
     * 
     * @param RegisterRequest $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function register(RegisterRequest $request)
    {
        // Request is already validated by RegisterRequest
        $validated = $request->validated();

        // Generate email verification token
        $verificationToken = Str::random(64);

        // Create user with minimal information
        // Status is set to 'inactive' until email is verified
        // Phone number will be collected in complete profile step
        $user = AppUser::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']),
            'status' => 'inactive', // Inactive until email verified
            'email_verified_at' => null,
        ]);

        // Store verification token in password_reset_tokens table (reusing for email verification)
        DB::table('password_reset_tokens')->insert([
            'email' => $user->email,
            'token' => Hash::make($verificationToken),
            'created_at' => now(),
        ]);

        // Send verification email
        // SECURITY IMPROVEMENT: Only include token in URL, not email
        // This prevents email exposure in browser history, server logs, etc.
        $frontendUrl = config('app.frontend_url', 'http://localhost:3000');
        $verificationUrl = $frontendUrl . '/verify?token=' . $verificationToken;
        
        try {
            Mail::to($user->email)->send(new VerifyEmailMail($verificationUrl, $user->name));
        } catch (\Exception $e) {
            // Log the error but don't fail registration
            // In production, you might want to log this to a file or monitoring service
            Log::error('Failed to send verification email: ' . $e->getMessage(), [
                'user_id' => $user->id,
                'email' => $user->email,
            ]);
        }

        return response()->json([
            'success' => true,
            'message' => 'Registration successful! Please check your email to activate your account.',
            'data' => [
                'user' => [
                    'id' => $user->id,
                    'name' => $user->name,
                    'email' => $user->email,
                    'status' => $user->status,
                    'email_verified' => false,
                ],
                // Development only - remove in production
                'debug' => [
                    'verification_token' => $verificationToken,
                    'verification_url' => $frontendUrl . '/verify?token=' . $verificationToken,
                    'note' => 'This debug field should be removed in production. User must verify email before account is activated.'
                ]
            ]
        ], 201);
    }

    /**
     * Handle user login for frontend (Next.js/Web)
     * 
     * @param LoginRequest $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function login(LoginRequest $request)
    {
        // Use AppUser guard for authentication
        $credentials = $request->only('email', 'password');
        $user = AppUser::where('email', $credentials['email'])->first();

        // Step 1: Check if user exists and password is correct
        if (!$user || !Hash::check($credentials['password'], $user->password)) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid email or password.',
                'errors' => [
                    'email' => ['The provided credentials do not match our records.']
                ]
            ], 401);
        }

        /** @var \App\Models\AppUser $user */
        
        // Step 2: Check if email is verified (required for login)
        if (!$user->email_verified_at) {
            return response()->json([
                'success' => false,
                'message' => 'Please verify your email address before logging in. Check your inbox for the activation link.',
                'errors' => [
                    'email' => ['Email verification required. Please check your inbox for the activation link.']
                ]
            ], 403);
        }

        // Step 3: Check if user account status is active
        // Status can be: 'active', 'inactive', 'suspended'
        // Only 'active' users can login
        if ($user->status !== 'active') {
            return response()->json([
                'success' => false,
                'message' => 'Your account is not active. Please contact support.',
            ], 403);
        }
        

        // Create API token with expiration based on "Remember Me"
        $rememberMe = $request->boolean('remember_me', false);
        
        // If "Remember Me" is checked, token expires in 30 days, otherwise 7 days
        $expiresAt = $rememberMe 
            ? now()->addDays(30)  // 30 days for "Remember Me"
            : now()->addDays(7);   // 7 days for normal login
        
        $token = $user->createToken(
            'web_app_token',
            ['*'], // All abilities
            $expiresAt
        )->plainTextToken;

        return response()->json([
            'success' => true,
            'message' => 'Login successful.',
            'data' => [
                'token' => $token,
                'token_expires_at' => $expiresAt->toIso8601String(),
                'remember_me' => $rememberMe,
                'user' => [
                    'id' => $user->id,
                    'name' => $user->name,
                    'email' => $user->email,
                    'email_verified' => $user->email_verified_at !== null,
                    'timezone' => $user->timezone,
                    'status' => $user->status,
                ]
            ]
        ], 200);
    }

    /**
     * Get authenticated user profile
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function profile(Request $request)
    {
        return response()->json([
            'success' => true,
            'data' => $request->user()
        ]);
    }

    /**
     * Handle user logout
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function logout(Request $request)
    {
        // Revoke the current token
        $request->user()->currentAccessToken()->delete();

        return response()->json([
            'success' => true,
            'message' => 'Logged out successfully.'
        ]);
    }

    /**
     * Handle forgot password request
     * 
     * SECURITY NOTE: This endpoint always returns success, even if email doesn't exist.
     * This prevents email enumeration attacks. The debug field is only for development.
     * 
     * IMPROVED SECURITY FLOW:
     * 1. User requests password reset with email
     * 2. If email exists: Generate token, store in DB, send email with frontend link
     * 3. If email doesn't exist: Return same success message (security)
     * 4. Email contains: http://FRONTEND/reset-password?token=xxx (ONLY TOKEN - more secure)
     * 5. User clicks link → Opens Next.js frontend page
     * 6. Frontend calls GET /api/validate-reset-token?token=xxx to check if token is valid
     * 7. If valid, frontend displays password reset form
     * 8. User submits form → Frontend makes POST request to /api/reset-password with token and password
     * 
     * @param ForgotPasswordRequest $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function forgotPassword(ForgotPasswordRequest $request)
    {
        $email = $request->validated()['email'];
        $frontendUrl = config('app.frontend_url', 'http://localhost:3000');
        $token = null;
        
        // Check if user exists
        $user = AppUser::where('email', $email)->first();
        
        // Always return success message (security best practice - don't reveal if email exists)
        // But only send email if user actually exists
        if ($user) {
            // Generate password reset token
            $token = Str::random(64);
            
            // Store token in password_reset_tokens table
            // Using the same table structure as Laravel's default
            DB::table('password_reset_tokens')->updateOrInsert(
                ['email' => $email],
                [
                    'token' => Hash::make($token),
                    'created_at' => now(),
                ]
            );
            
            // Send email with reset link
            // SECURITY IMPROVEMENT: Only include token in URL, not email
            // This prevents email exposure in browser history, server logs, etc.
            // Format: http://your-frontend-domain.com/reset-password?token={token}
            // The frontend will validate the token and retrieve the email from the backend
            $resetUrl = $frontendUrl . '/reset-password?token=' . $token;
            
            try {
                Mail::to($user->email)->send(new ResetPasswordMail($resetUrl, $user->name));
            } catch (\Exception $e) {
                // Log the error but don't fail the request (security best practice)
                Log::error('Failed to send password reset email: ' . $e->getMessage(), [
                    'email' => $email,
                ]);
            }
        }
        
        // Always return same response for security (prevents email enumeration)
        // In production, remove the debug field completely!
        
        return response()->json([
            'success' => true,
            'message' => 'If an account exists with that email, we have sent a password reset link.',
            // REMOVE THIS IN PRODUCTION - only for development/testing
            // debug: null means email doesn't exist (for testing only)
            'debug' => $user ? [
                'token' => $token ?? null,
                // This URL opens the FRONTEND page (GET request), not the API endpoint
                'reset_url' => $frontendUrl . '/reset-password?token=' . ($token ?? ''),
                'note' => 'This debug field should be removed in production. Email link opens frontend page, which validates token then POSTs to /api/reset-password'
            ] : null
        ], 200);
    }

    /**
     * Validate reset token (called when user clicks reset link)
     * 
     * This endpoint is called by the FRONTEND (Next.js) when user:
     * 1. Clicks email link (opens frontend page with token in URL)
     * 2. Frontend immediately calls this endpoint to validate token and check expiration
     * 3. If valid, frontend displays password reset form
     * 4. If invalid/expired, frontend shows error message
     * 
     * Request: GET /api/validate-reset-token?token=xxx
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function validateResetToken(Request $request)
    {
        $request->validate([
            'token' => 'required|string',
        ]);

        $token = $request->input('token');
        
        // Find reset tokens created in the last 24 hours (optimization)
        // We need to iterate because tokens are hashed in DB
        // Only check recent tokens to improve performance
        $resetRecords = DB::table('password_reset_tokens')
            ->where('created_at', '>=', now()->subHours(24))
            ->orderBy('created_at', 'desc')
            ->get();
        
        $validRecord = null;
        foreach ($resetRecords as $record) {
            if (Hash::check($token, $record->token)) {
                $validRecord = $record;
                break;
            }
        }
        
        if (!$validRecord) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid or expired reset token.',
                'errors' => [
                    'token' => ['This password reset token is invalid or has expired.']
                ],
                'valid' => false,
                'expired' => false,
            ], 422);
        }
        
        // Check if token is expired (60 minutes default)
        $tokenAge = Carbon::parse($validRecord->created_at)->diffInMinutes(now());
        $isExpired = $tokenAge > 60;
        
        if ($isExpired) {
            // Delete expired token
            DB::table('password_reset_tokens')->where('email', $validRecord->email)->delete();
            
            return response()->json([
                'success' => false,
                'message' => 'This password reset token has expired. Please request a new one.',
                'errors' => [
                    'token' => ['This password reset token has expired. Please request a new password reset link.']
                ],
                'valid' => false,
                'expired' => true,
            ], 422);
        }
        
        // Token is valid and not expired
        // Return success with masked email for display purposes (optional)
        $email = $validRecord->email;
        $maskedEmail = $this->maskEmail($email);
        
        return response()->json([
            'success' => true,
            'message' => 'Reset token is valid.',
            'valid' => true,
            'expired' => false,
            'email' => $maskedEmail, // Masked email for display (e.g., u***@example.com)
            'expires_in_minutes' => 60 - $tokenAge, // Remaining time
        ], 200);
    }

    /**
     * Handle password reset
     * 
     * IMPROVED SECURITY FLOW:
     * This endpoint is called by the FRONTEND (Next.js) after user:
     * 1. Clicks email link (opens frontend page with token in URL)
     * 2. Frontend validates token via GET /api/validate-reset-token
     * 3. If valid, frontend displays password reset form
     * 4. User submits form → Frontend makes POST request to this endpoint
     * 
     * Request body: { token, password, password_confirmation }
     * Note: Email is retrieved from token record, not from request (more secure)
     * 
     * @param ResetPasswordRequest $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function resetPassword(ResetPasswordRequest $request)
    {
        $validated = $request->validated();
        $token = $validated['token'];
        $password = $validated['password'];
        
        // Find reset tokens created in the last 24 hours (optimization)
        // We need to iterate because tokens are hashed in DB
        // Only check recent tokens to improve performance
        $resetRecords = DB::table('password_reset_tokens')
            ->where('created_at', '>=', now()->subHours(24))
            ->orderBy('created_at', 'desc')
            ->get();
        
        $validRecord = null;
        foreach ($resetRecords as $record) {
            if (Hash::check($token, $record->token)) {
                $validRecord = $record;
                break;
            }
        }
        
        if (!$validRecord) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid or expired reset token.',
                'errors' => [
                    'token' => ['This password reset token is invalid or has expired.']
                ]
            ], 422);
        }
        
        // Get email from token record (more secure - email not in request)
        $email = $validRecord->email;
        
        // Check if token is expired (60 minutes default)
        $tokenAge = Carbon::parse($validRecord->created_at)->diffInMinutes(now());
        if ($tokenAge > 60) {
            // Delete expired token
            DB::table('password_reset_tokens')->where('email', $email)->delete();
            
            return response()->json([
                'success' => false,
                'message' => 'This password reset token has expired. Please request a new one.',
                'errors' => [
                    'token' => ['This password reset token has expired. Please request a new password reset link.']
                ]
            ], 422);
        }
        
        // Find user and update password
        $user = AppUser::where('email', $email)->first();
        
        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'User not found.',
            ], 404);
        }
        
        // Update password
        $user->password = Hash::make($password);
        $user->save();
        
        // Delete used reset token
        DB::table('password_reset_tokens')->where('email', $email)->delete();
        
        // Optionally: Revoke all existing tokens to force re-login
        // $user->tokens()->delete();
        
        return response()->json([
            'success' => true,
            'message' => 'Your password has been reset successfully. You can now login with your new password.',
        ], 200);
    }

    /**
     * Mask email address for display (privacy)
     * Example: user@example.com -> u***@example.com
     * 
     * @param string $email
     * @return string
     */
    private function maskEmail(string $email): string
    {
        if (strpos($email, '@') === false) {
            return $email;
        }
        
        [$localPart, $domain] = explode('@', $email, 2);
        
        if (strlen($localPart) <= 1) {
            $maskedLocal = '*';
        } else {
            $maskedLocal = substr($localPart, 0, 1) . str_repeat('*', min(3, strlen($localPart) - 1));
        }
        
        return $maskedLocal . '@' . $domain;
    }

    /**
     * Verify email address (Step 2: Email Activation)
     * 
     * IMPROVED SECURITY FLOW:
     * 1. User clicks link from email: /verify?token=xxx (ONLY TOKEN)
     * 2. Frontend page loads and calls this endpoint
     * 3. This function validates token, checks expiration, and verifies email
     * 
     * Request body: { token }
     * Note: Email is retrieved from token record, not from request (more secure)
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function verifyEmail(Request $request)
    {
        $request->validate([
            'token' => 'required|string',
        ]);

        $token = $request->input('token');

        // Find verification tokens created in the last 48 hours (optimization)
        // We need to iterate because tokens are hashed in DB
        $verificationRecords = DB::table('password_reset_tokens')
            ->where('created_at', '>=', now()->subHours(48))
            ->orderBy('created_at', 'desc')
            ->get();
        
        $validRecord = null;
        foreach ($verificationRecords as $record) {
            if (Hash::check($token, $record->token)) {
                $validRecord = $record;
                break;
            }
        }

        if (!$validRecord) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid or expired verification link.',
                'errors' => [
                    'token' => ['This verification link is invalid or has expired.']
                ]
            ], 422);
        }

        // Get email from token record (more secure - email not in request)
        $email = $validRecord->email;

        // Check if token is expired (24 hours)
        $tokenAge = Carbon::parse($validRecord->created_at)->diffInHours(now());
        if ($tokenAge > 24) {
            DB::table('password_reset_tokens')->where('email', $email)->delete();
            
            return response()->json([
                'success' => false,
                'message' => 'This verification link has expired. Please request a new one.',
                'errors' => [
                    'token' => ['This verification link has expired. Please register again or contact support.']
                ]
            ], 422);
        }

        // Find user
        $user = AppUser::where('email', $email)->first();

        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'User not found.',
            ], 404);
        }

        // Check if already verified
        if ($user->email_verified_at) {
            return response()->json([
                'success' => true,
                'message' => 'Email is already verified. You can now login.',
                'already_verified' => true,
            ], 200);
        }

        // Verify email and activate account
        $user->email_verified_at = now();
        $user->status = 'active'; // Activate account
        $user->save();

        // Delete used verification token
        DB::table('password_reset_tokens')->where('email', $email)->delete();

        return response()->json([
            'success' => true,
            'message' => 'Email verified successfully! Your account is now activated. Please complete your profile to get started.',
            'already_verified' => false,
            'data' => [
                'user' => [
                    'id' => $user->id,
                    'name' => $user->name,
                    'email' => $user->email,
                    'email_verified' => true,
                    'status' => $user->status,
                    'profile_completed' => !empty($user->company_name), // Check if profile is completed
                ]
            ]
        ], 200);
    }

    /**
     * Resend email verification link
     * 
     * SECURITY NOTE: This endpoint always returns success, even if email doesn't exist or is already verified.
     * This prevents email enumeration attacks. The debug field is only for development.
     * 
     * FLOW:
     * 1. User requests resend verification with email
     * 2. If email exists and not verified: Generate new token, store in DB, send email with frontend link
     * 3. If email doesn't exist or already verified: Return same success message (security)
     * 4. Email contains: http://FRONTEND/verify?token=xxx (ONLY TOKEN - more secure)
     * 
     * @param ResendVerificationRequest $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function resendVerification(ResendVerificationRequest $request)
    {
        $email = $request->validated()['email'];
        $frontendUrl = config('app.frontend_url', 'http://localhost:3000');
        $token = null;
        
        // Check if user exists
        $user = AppUser::where('email', $email)->first();
        
        // Always return success message (security best practice - don't reveal if email exists)
        // But only send email if user actually exists and email is not verified
        if ($user) {
            // Check if email is already verified
            if ($user->email_verified_at) {
                // Email already verified - return success but don't send email
                return response()->json([
                    'success' => true,
                    'message' => 'Your email address is already verified. You can now login.',
                    'already_verified' => true,
                    // REMOVE THIS IN PRODUCTION - only for development/testing
                    'debug' => [
                        'note' => 'This debug field should be removed in production. Email is already verified.'
                    ]
                ], 200);
            }
            
            // Generate new verification token
            $token = Str::random(64);
            
            // Delete any existing verification token for this email
            DB::table('password_reset_tokens')->where('email', $email)->delete();
            
            // Store new token in password_reset_tokens table (reusing for email verification)
            DB::table('password_reset_tokens')->insert([
                'email' => $user->email,
                'token' => Hash::make($token),
                'created_at' => now(),
            ]);
            
            // Send verification email
            // SECURITY IMPROVEMENT: Only include token in URL, not email
            // This prevents email exposure in browser history, server logs, etc.
            $verificationUrl = $frontendUrl . '/verify?token=' . $token;
            
            try {
                Mail::to($user->email)->send(new VerifyEmailMail($verificationUrl, $user->name));
            } catch (\Exception $e) {
                // Log the error but don't fail the request (security best practice)
                Log::error('Failed to send verification email: ' . $e->getMessage(), [
                    'user_id' => $user->id,
                    'email' => $email,
                ]);
            }
        }
        
        // Always return same response for security (prevents email enumeration)
        // In production, remove the debug field completely!
        
        return response()->json([
            'success' => true,
            'message' => 'If an account exists with that email and needs verification, we have sent a verification link.',
            'already_verified' => false,
            // REMOVE THIS IN PRODUCTION - only for development/testing
            // debug: null means email doesn't exist or already verified (for testing only)
            'debug' => $user && !$user->email_verified_at ? [
                'token' => $token ?? null,
                // This URL opens the FRONTEND page (GET request), not the API endpoint
                'verification_url' => $frontendUrl . '/verify?token=' . ($token ?? ''),
                'note' => 'This debug field should be removed in production. Email link opens frontend page, which validates token then POSTs to /api/verify-email'
            ] : null
        ], 200);
    }

    /**
     * Complete profile with phone and company details (Step 3: After Activation)
     * 
     * User must be authenticated and email verified
     * Phone number is required and collected in this step
     * 
     * @param CompleteProfileRequest $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function completeProfile(CompleteProfileRequest $request)
    {
        $user = $request->user();
        $validated = $request->validated();

        // Check if email is verified
        if (!$user->email_verified_at) {
            return response()->json([
                'success' => false,
                'message' => 'Please verify your email address first.',
            ], 403);
        }

        // Update user with phone and company information
        $user->update([
            'phone' => $validated['phone'],
            'company_name' => $validated['company_name'],
            'company_description' => $validated['company_description'] ?? null,
            'website' => $validated['website'] ?? null,
            'address' => $validated['address'] ?? null,
            'city' => $validated['city'] ?? null,
            'state' => $validated['state'] ?? null,
            'country' => $validated['country'] ?? 'India',
            'postal_code' => $validated['postal_code'] ?? null,
            'social_media_handles' => $validated['social_media_handles'] ?? null,
            'preferred_social_platforms' => $validated['preferred_social_platforms'] ?? null,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Profile completed successfully! You can now start posting to social media platforms.',
            'data' => [
                'user' => [
                    'id' => $user->id,
                    'name' => $user->name,
                    'email' => $user->email,
                    'company' => [
                        'company_name' => $user->company_name,
                        'preferred_platforms' => $user->preferred_social_platforms,
                    ]
                ]
            ]
        ], 200);
    }

    /**
     * Update user profile information
     * 
     * User must be authenticated
     * Email cannot be updated (account identifier)
     * Password should be updated via separate change password endpoint
     * Status cannot be updated by user (admin only)
     * 
     * @param UpdateProfileRequest $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateProfile(UpdateProfileRequest $request)
    {
        $user = $request->user();
        $validated = $request->validated();
        $updateData = [];

        // REQUIRED FIELDS: name, phone - Only update if value provided, never clear
        if (isset($validated['name']) && trim($validated['name']) !== '') {
            $updateData['name'] = trim($validated['name']);
        }
        if (isset($validated['phone']) && trim($validated['phone']) !== '') {
            $updateData['phone'] = trim($validated['phone']);
        }

        // OPTIONAL FIELDS: Can be updated or cleared (set to null if empty)
        // Basic Info
        if ($request->has('timezone')) {
            $updateData['timezone'] = !empty(trim($validated['timezone'] ?? '')) 
                ? trim($validated['timezone']) 
                : 'UTC';
        }

        // Company Info
        if ($request->has('company_name')) {
            $updateData['company_name'] = !empty(trim($validated['company_name'] ?? '')) 
                ? trim($validated['company_name']) 
                : null;
        }
        if ($request->has('company_description')) {
            $updateData['company_description'] = !empty(trim($validated['company_description'] ?? '')) 
                ? trim($validated['company_description']) 
                : null;
        }

        // Contact Info
        if ($request->has('website')) {
            $updateData['website'] = !empty(trim($validated['website'] ?? '')) 
                ? trim($validated['website']) 
                : null;
        }
        if ($request->has('address')) {
            $updateData['address'] = !empty(trim($validated['address'] ?? '')) 
                ? trim($validated['address']) 
                : null;
        }
        if ($request->has('city')) {
            $updateData['city'] = !empty(trim($validated['city'] ?? '')) 
                ? trim($validated['city']) 
                : null;
        }
        if ($request->has('state')) {
            $updateData['state'] = !empty(trim($validated['state'] ?? '')) 
                ? trim($validated['state']) 
                : null;
        }
        if ($request->has('country')) {
            $updateData['country'] = !empty(trim($validated['country'] ?? '')) 
                ? trim($validated['country']) 
                : null;
        }
        if ($request->has('postal_code')) {
            $updateData['postal_code'] = !empty(trim($validated['postal_code'] ?? '')) 
                ? trim($validated['postal_code']) 
                : null;
        }

        // Social Media
        if ($request->has('social_media_handles')) {
            if (isset($validated['social_media_handles']) && is_array($validated['social_media_handles'])) {
                $handles = array_filter($validated['social_media_handles'], fn($v) => !empty(trim($v ?? '')));
                $updateData['social_media_handles'] = !empty($handles) ? $handles : null;
            } else {
                $updateData['social_media_handles'] = null;
            }
        }
        if ($request->has('preferred_social_platforms')) {
            $updateData['preferred_social_platforms'] = !empty($validated['preferred_social_platforms'] ?? []) 
                ? $validated['preferred_social_platforms'] 
                : null;
        }

        // Update user
        if (!empty($updateData)) {
            $user->update($updateData);
        }

        // Refresh user to get updated data
        $user->refresh();

        return response()->json([
            'success' => true,
            'message' => 'Profile updated successfully.',
            'data' => [
                'user' => [
                    'id' => $user->id,
                    'name' => $user->name,
                    'email' => $user->email,
                    'phone' => $user->phone,
                    'timezone' => $user->timezone,
                    'email_verified' => $user->email_verified_at !== null,
                    'status' => $user->status,
                    'company' => [
                        'company_name' => $user->company_name,
                        'company_description' => $user->company_description,
                        'website' => $user->website,
                        'address' => $user->address,
                        'city' => $user->city,
                        'state' => $user->state,
                        'country' => $user->country,
                        'postal_code' => $user->postal_code,
                        'social_media_handles' => $user->social_media_handles,
                        'preferred_platforms' => $user->preferred_social_platforms,
                    ]
                ]
            ]
        ], 200);
    }
}
