<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class UserController extends Controller
{
    public function index()
    {
        $users = User::with(['subscription.plan'])->orderBy('id', 'asc')->paginate(20);
        return view('admin.users.index', compact('users'));
    }


    public function store(Request $request)
    {
        // echo "<pre>"; print_r($request->all()); exit;
        $data = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|string|min:6|confirmed',
            'role' => 'required|in:subscriber,admin',
            'timezone' => 'nullable|string|max:50',
            // accept both legacy 'suspended' and current 'banned' from UI
            'status' => 'nullable|in:active,inactive,banned,suspended',
        ]);

        $data['password'] = Hash::make($data['password']);
        // persist selected role from validated input
        $data['role'] = $data['role'] ?? 'subscriber';
        $data['timezone'] = $data['timezone'] ?? 'UTC';
        $data['status'] = $data['status'] ?? 'active';

        // use Eloquent create to trigger timestamps and model events
        $user = User::create($data);

        return redirect()->route('users.index')->with('success', 'User created successfully.');
    }


    public function show(User $user)
    {
        return response()->json($user);
    }


    public function update(Request $request, User $user)
    {
        $data = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'password' => 'nullable|string|min:6|confirmed',
            'timezone' => 'nullable|string|max:50',
            // accept both legacy 'suspended' and current 'banned' from UI
            'status' => 'nullable|in:active,inactive,banned,suspended',
        ]);

        if (!empty($data['password'])) {
            $data['password'] = Hash::make($data['password']);
        } else {
            unset($data['password']);
        }

        // Disallow role changes via this endpoint; always enforce subscriber for non-admins
        unset($data['role']);
        if (isset($data['timezone']) && $data['timezone'] === '') {
            unset($data['timezone']);
        }
        if (isset($data['status']) && $data['status'] === '') {
            unset($data['status']);
        }

        $user->update($data);

        return redirect()->route('users.index')->with('success', 'User updated successfully.');
    }

    public function destroy(User $user)
    {
        if ($user->role === 'admin') {
            return redirect()->route('users.index')->with('success', 'Admin accounts cannot be deleted.');
        }
        $user->delete();
        return redirect()->route('users.index')->with('success', 'User deleted successfully.');
    }

    public function data(Request $request)
    {
        $draw = (int) $request->input('draw', 1);
        $start = (int) $request->input('start', 0);
        $length = (int) $request->input('length', 10);
        $searchValue = $request->input('search.value');

        $columns = [
            0 => 'name',
            1 => 'email',
            2 => 'role',
            3 => 'timezone',
            4 => 'status',
            5 => 'created_at',
        ];

        $orderColumnIndex = (int) data_get($request->input('order'), '0.column', 0);
        $orderDir = data_get($request->input('order'), '0.dir', 'asc') === 'desc' ? 'desc' : 'asc';
        $orderColumn = $columns[$orderColumnIndex] ?? 'name';

        $baseQuery = User::query();

        $recordsTotal = (clone $baseQuery)->count();

        if (!empty($searchValue)) {
            $baseQuery->where(function ($q) use ($searchValue) {
                $q->where('name', 'like', "%{$searchValue}%")
                  ->orWhere('email', 'like', "%{$searchValue}%")
                  ->orWhere('role', 'like', "%{$searchValue}%")
                  ->orWhere('timezone', 'like', "%{$searchValue}%")
                  ->orWhere('status', 'like', "%{$searchValue}%");
            });
        }

        $recordsFiltered = (clone $baseQuery)->count();

        $users = $baseQuery
            ->orderBy($orderColumn, $orderDir)
            ->skip($start)
            ->take($length)
            ->get();

        $data = $users->map(function (User $user) {
            return [
                'name' => e($user->name),
                'email' => e($user->email),
                'role' => e($user->role),
                'timezone' => e($user->timezone),
                'status' => ucfirst(e($user->status)),
                'created' => optional($user->created_at)->format('Y-m-d'),
                'actions' => view('admin.users.partials.actions', compact('user'))->render(),
            ];
        });

        return response()->json([
            'draw' => $draw,
            'recordsTotal' => $recordsTotal,
            'recordsFiltered' => $recordsFiltered,
            'data' => $data,
        ]);
    }
}
