<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\StorePlanRequest;
use App\Http\Requests\Admin\UpdatePlanRequest;
use App\Models\SubscriptionPlan;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;

class PlanController extends Controller
{
    /**
     * Display a listing of the subscription plans.
     */
    public function index(): View
    {
        $plans = SubscriptionPlan::orderBy('created_at', 'desc')->paginate(20);
        
        return view('admin.plans.index', compact('plans'));
    }

    /**
     * Store a newly created subscription plan.
     */
    public function store(StorePlanRequest $request): RedirectResponse
    {
        $validated = $request->validated();
        
        SubscriptionPlan::create($validated);

        return redirect()
            ->route('plans.index')
            ->with('success', 'Subscription plan created successfully.');
    }

    /**
     * Display the specified subscription plan.
     */
    public function show(SubscriptionPlan $plan): JsonResponse
    {
        return response()->json([
            'success' => true,
            'data' => $plan
        ]);
    }

    /**
     * Update the specified subscription plan.
     */
    public function update(UpdatePlanRequest $request, SubscriptionPlan $plan): RedirectResponse
    {
        $validated = $request->validated();
        
        $plan->update($validated);

        return redirect()
            ->route('plans.index')
            ->with('success', 'Subscription plan updated successfully.');
    }

    /**
     * Remove the specified subscription plan.
     */
    public function destroy(SubscriptionPlan $plan): RedirectResponse
    {
        // Check if plan has active subscriptions
        if ($plan->subscriptions()->where('status', 'active')->exists()) {
            return redirect()
                ->route('plans.index')
                ->with('error', 'Cannot delete plan with active subscriptions.');
        }

        $plan->delete();

        return redirect()
            ->route('plans.index')
            ->with('success', 'Subscription plan deleted successfully.');
    }

    /**
     * Data endpoint for DataTables (server-side processing).
     */
    public function data(Request $request): JsonResponse
    {
        $draw = (int) $request->input('draw', 1);
        $start = (int) $request->input('start', 0);
        $length = (int) $request->input('length', 10);
        $searchValue = $request->input('search.value');

        $columns = [
            0 => 'name',
            1 => 'price',
            2 => 'duration',
            3 => 'status',
            4 => 'created_at',
        ];

        $orderColumnIndex = (int) data_get($request->input('order'), '0.column', 0);
        $orderDir = data_get($request->input('order'), '0.dir', 'asc') === 'desc' ? 'desc' : 'asc';
        $orderColumn = $columns[$orderColumnIndex] ?? 'name';

        $baseQuery = SubscriptionPlan::query();

        $recordsTotal = (clone $baseQuery)->count();

        if (!empty($searchValue)) {
            $baseQuery->where(function ($q) use ($searchValue) {
                $q->where('name', 'like', "%{$searchValue}%")
                  ->orWhere('duration', 'like', "%{$searchValue}%")
                  ->orWhere('status', 'like', "%{$searchValue}%");
            });
        }

        $recordsFiltered = (clone $baseQuery)->count();

        $plans = $baseQuery
            ->orderBy($orderColumn, $orderDir)
            ->skip($start)
            ->take($length)
            ->get();

        $data = $plans->map(function (SubscriptionPlan $plan) {
            return [
                'name' => e($plan->name),
                'price' => e($plan->formatted_price ?? $plan->getFormattedPriceAttribute()),
                'duration' => ucfirst(e($plan->duration)),
                'status' => ucfirst(e($plan->status)),
                'created' => optional($plan->created_at)->format('Y-m-d'),
                'actions' => view('admin.plans.partials.actions', compact('plan'))->render(),
            ];
        });

        return response()->json([
            'draw' => $draw,
            'recordsTotal' => $recordsTotal,
            'recordsFiltered' => $recordsFiltered,
            'data' => $data,
        ]);
    }
}


