<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Category;
use Illuminate\Http\Request;

class CategoryController extends Controller
{
    public function index()
    {
        $categories = Category::orderBy('id', 'asc')->paginate(20);
        return view('admin.categories.index', compact('categories'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'category_name' => 'required|string|max:255',
            'is_default' => 'nullable',
            'user_id' => 'nullable|integer',
        ]);

        // Set defaults - checkbox sends value "1" if checked, nothing if unchecked
        $data['is_default'] = isset($data['is_default']) && $data['is_default'] == '1' ? '1' : '0'; // Default to 1 as per requirement
        $data['user_id'] = $data['user_id'] ?? null;

        Category::create($data);

        return redirect()->route('categories.index')->with('success', 'Category created successfully.');
    }

    public function show(Category $category)
    {
        return response()->json($category);
    }

    public function update(Request $request, Category $category)
    {
        $data = $request->validate([
            'category_name' => 'required|string|max:255',
            'is_default' => 'nullable',
            'user_id' => 'nullable|integer',
        ]);

        // Handle is_default - checkbox sends value "1" if checked, nothing if unchecked
        $data['is_default'] = isset($data['is_default']) && $data['is_default'] == '1' ? '1' : '0';

        // Handle user_id - set to null if empty
        if (isset($data['user_id']) && $data['user_id'] === '') {
            $data['user_id'] = null;
        }

        $category->update($data);

        return redirect()->route('categories.index')->with('success', 'Category updated successfully.');
    }

    public function destroy(Category $category)
    {
        $category->delete();
        return redirect()->route('categories.index')->with('success', 'Category deleted successfully.');
    }

    public function data(Request $request)
    {
        $draw = (int) $request->input('draw', 1);
        $start = (int) $request->input('start', 0);
        $length = (int) $request->input('length', 10);
        $searchValue = $request->input('search.value');

        $columns = [
            0 => 'category_name',
            1 => 'is_default',
            2 => 'user_id',
            3 => 'created_at',
        ];

        $orderColumnIndex = (int) data_get($request->input('order'), '0.column', 0);
        $orderDir = data_get($request->input('order'), '0.dir', 'asc') === 'desc' ? 'desc' : 'asc';
        $orderColumn = $columns[$orderColumnIndex] ?? 'category_name';

        $baseQuery = Category::query();

        $recordsTotal = (clone $baseQuery)->count();

        if (!empty($searchValue)) {
            $baseQuery->where(function ($q) use ($searchValue) {
                $q->where('category_name', 'like', "%{$searchValue}%")
                  ->orWhere('is_default', 'like', "%{$searchValue}%")
                  ->orWhere('user_id', 'like', "%{$searchValue}%");
            });
        }

        $recordsFiltered = (clone $baseQuery)->count();

        $categories = $baseQuery
            ->orderBy($orderColumn, $orderDir)
            ->skip($start)
            ->take($length)
            ->get();

        $data = $categories->map(function (Category $category) {
            return [
                'name' => e($category->category_name),
                'is_default' => $category->is_default ? 'Yes' : 'No',
                'user_id' => $category->user_id ?? 'N/A',
                'created' => optional($category->created_at)->format('Y-m-d'),
                'actions' => view('admin.categories.partials.actions', compact('category'))->render(),
            ];
        });

        return response()->json([
            'draw' => $draw,
            'recordsTotal' => $recordsTotal,
            'recordsFiltered' => $recordsFiltered,
            'data' => $data,
        ]);
    }
}

