<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AppUser;
use App\Models\Subscription;
use App\Models\SubscriptionPlan;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class AppUserController extends Controller
{
    public function index()
    {
        $users = AppUser::with(['subscription.plan'])
            ->orderBy('id', 'asc')
            ->paginate(20);
        return view('admin.appusers.index', compact('users'));
    }

    public function show(AppUser $appuser)
    {
        $appuser->load(['subscription.plan']);
        return response()->json([
            'user' => $appuser,
            'subscription' => $appuser->subscription
        ]);
    }

    public function update(Request $request, AppUser $appuser)
    {

        $data = $request->validate([
            // Basic Information
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:app_users,email,' . $appuser->id,
            'password' => 'nullable|string|min:6|confirmed',
            'timezone' => 'nullable|string|max:50',
            'status' => 'nullable|in:active,inactive,suspended',
            // Company/Shop Information
            'company_name' => 'nullable|string|max:255',
            'company_description' => 'nullable|string',
            // Contact Information
            'phone' => 'nullable|string|max:20',
            'website' => 'nullable|url|max:255',
            'address' => 'nullable|string',
            'city' => 'nullable|string|max:100',
            'state' => 'nullable|string|max:100',
            'country' => 'nullable|string|max:100',
            'postal_code' => 'nullable|string|max:20',
            // Social Media & Marketing
            'social_media_handles' => 'nullable|string',
            'preferred_social_platforms' => 'nullable|string',
            // Subscription fields
            'subscription_id' => 'nullable|exists:subscriptions,id',
            'plan_id' => 'nullable|exists:subscription_plans,id',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'subscription_status' => 'nullable|in:active,expired,cancelled',
            'payment_method' => 'nullable|in:stripe,paypal,manual',
            'transaction_id' => 'nullable|string|max:150',
        ]);

        // Update user fields - include all app_users table fields
        $userData = [
            'name' => $data['name'],
            'email' => $data['email'],
            'timezone' => $data['timezone'] ?? 'UTC',
            'status' => $data['status'] ?? 'inactive',
            'company_name' => $data['company_name'] ?? null,
            'company_description' => $data['company_description'] ?? null,
            'phone' => $data['phone'] ?? null,
            'website' => $data['website'] ?? null,
            'address' => $data['address'] ?? null,
            'city' => $data['city'] ?? null,
            'state' => $data['state'] ?? null,
            'country' => $data['country'] ?? null,
            'postal_code' => $data['postal_code'] ?? null,
            'social_media_handles' => $this->parseSocialMediaHandles($data['social_media_handles'] ?? null),
            'preferred_social_platforms' => isset($data['preferred_social_platforms']) && !empty($data['preferred_social_platforms']) ? 
                (is_string($data['preferred_social_platforms']) ? 
                    (str_starts_with(trim($data['preferred_social_platforms']), '[') ? json_decode($data['preferred_social_platforms'], true) : 
                        array_filter(array_map('trim', explode(',', $data['preferred_social_platforms'])))) : 
                    $data['preferred_social_platforms']) : null,
        ];

        if (!empty($data['password'])) {
            $userData['password'] = Hash::make($data['password']);
        }

        $appuser->update($userData);

        // Handle subscription update or creation
        if (isset($data['plan_id']) && $data['plan_id']) {
            // Check if subscription exists
            if (isset($data['subscription_id']) && $data['subscription_id']) {
                $subscription = Subscription::find($data['subscription_id']);
                if ($subscription && $subscription->user_id === $appuser->id) {
                    // Update existing subscription
                    $subscriptionData = [
                        'plan_id' => $data['plan_id'],
                        'start_date' => $data['start_date'] ?? $subscription->start_date,
                        'end_date' => $data['end_date'] ?? $subscription->end_date,
                        'status' => $data['subscription_status'] ?? $subscription->status,
                        'payment_method' => $data['payment_method'] ?? $subscription->payment_method,
                        'transaction_id' => $data['transaction_id'] ?? $subscription->transaction_id,
                    ];
                    $subscription->update($subscriptionData);
                }
            } else {
                // Create new subscription if it doesn't exist
                $subscriptionData = [
                    'user_id' => $appuser->id,
                    'plan_id' => $data['plan_id'],
                    'start_date' => $data['start_date'] ?? now()->toDateString(),
                    'end_date' => $data['end_date'] ?? now()->addMonth()->toDateString(),
                    'status' => $data['subscription_status'] ?? 'active',
                    'payment_method' => $data['payment_method'] ?? 'manual',
                    'transaction_id' => $data['transaction_id'] ?? null,
                ];
                Subscription::create($subscriptionData);
            }
        } elseif (isset($data['subscription_id']) && $data['subscription_id']) {
            // Update subscription without changing plan
            $subscription = Subscription::find($data['subscription_id']);
            if ($subscription && $subscription->user_id === $appuser->id) {
                $subscriptionData = [];
                if (isset($data['start_date'])) $subscriptionData['start_date'] = $data['start_date'];
                if (isset($data['end_date'])) $subscriptionData['end_date'] = $data['end_date'];
                if (isset($data['subscription_status'])) $subscriptionData['status'] = $data['subscription_status'];
                if (isset($data['payment_method'])) $subscriptionData['payment_method'] = $data['payment_method'];
                if (isset($data['transaction_id'])) $subscriptionData['transaction_id'] = $data['transaction_id'];
                
                if (!empty($subscriptionData)) {
                    $subscription->update($subscriptionData);
                }
            }
        }

        return redirect()->route('appusers.index')->with('success', 'App User updated successfully.');
    }

    public function getPlanDetails($planId)
    {
        $plan = SubscriptionPlan::find($planId);
        if (!$plan) {
            return response()->json(['error' => 'Plan not found'], 404);
        }

        return response()->json([
            'id' => $plan->id,
            'name' => $plan->name,
            'duration' => $plan->duration,
        ]);
    }

    /**
     * Parse social media handles from various formats to JSON object
     */
    private function parseSocialMediaHandles($input)
    {
        if (empty($input)) {
            return null;
        }

        if (!is_string($input)) {
            return $input;
        }

        $trimmed = trim($input);

        // If it's already a JSON object, parse it
        if (str_starts_with($trimmed, '{')) {
            $decoded = json_decode($trimmed, true);
            return $decoded ?: null;
        }

        // If it's a JSON array, parse it (fallback)
        if (str_starts_with($trimmed, '[')) {
            $decoded = json_decode($trimmed, true);
            return $decoded ?: null;
        }

        // Parse "platform: handle" format
        $pairs = array_filter(array_map('trim', explode(',', $trimmed)));
        $obj = [];
        foreach ($pairs as $pair) {
            $parts = explode(':', $pair, 2);
            if (count($parts) === 2) {
                $platform = trim($parts[0]);
                $handle = trim($parts[1]);
                if ($platform && $handle) {
                    $obj[strtolower($platform)] = $handle;
                }
            }
        }

        return !empty($obj) ? $obj : null;
    }

    public function data(Request $request)
    {
        $draw = (int) $request->input('draw', 1);
        $start = (int) $request->input('start', 0);
        $length = (int) $request->input('length', 10);
        $searchValue = $request->input('search.value');

        $columns = [
            0 => 'name',
            1 => 'email',
            2 => 'timezone',
            3 => 'status',
            4 => 'created_at',
        ];

        $orderColumnIndex = (int) data_get($request->input('order'), '0.column', 0);
        $orderDir = data_get($request->input('order'), '0.dir', 'asc') === 'desc' ? 'desc' : 'asc';
        $orderColumn = $columns[$orderColumnIndex] ?? 'name';

        // Get all app users
        $baseQuery = AppUser::with(['subscription.plan']);

        $recordsTotal = (clone $baseQuery)->count();

        if (!empty($searchValue)) {
            $baseQuery->where(function ($q) use ($searchValue) {
                $q->where('name', 'like', "%{$searchValue}%")
                  ->orWhere('email', 'like', "%{$searchValue}%")
                  ->orWhere('timezone', 'like', "%{$searchValue}%")
                  ->orWhere('status', 'like', "%{$searchValue}%");
            });
        }

        $recordsFiltered = (clone $baseQuery)->count();

        $users = $baseQuery
            ->orderBy($orderColumn, $orderDir)
            ->skip($start)
            ->take($length)
            ->get();

        $data = $users->map(function (AppUser $user) {
            $subscription = $user->subscription;
            return [
                'name' => e($user->name),
                'email' => e($user->email),
                'subscription_plan' => $subscription && $subscription->plan ? e($subscription->plan->name) : 'N/A',
                'subscription_status' => $subscription ? ucfirst(e($subscription->status)) : 'N/A',
                'start_date' => $subscription && $subscription->start_date ? optional($subscription->start_date)->format('Y-m-d') : 'N/A',
                'end_date' => $subscription && $subscription->end_date ? optional($subscription->end_date)->format('Y-m-d') : 'N/A',
                'status' => ucfirst(e($user->status)),
                'created' => optional($user->created_at)->format('Y-m-d'),
                'actions' => view('admin.appusers.partials.actions', compact('user'))->render(),
            ];
        });

        return response()->json([
            'draw' => $draw,
            'recordsTotal' => $recordsTotal,
            'recordsFiltered' => $recordsFiltered,
            'data' => $data,
        ]);
    }
}

