# Social Media Content Generation Platform - Workflow Documentation

## Table of Contents
1. [Project Overview](#project-overview)
2. [Architecture Overview](#architecture-overview)
3. [Authentication & Authorization Workflows](#authentication--authorization-workflows)
4. [Admin Dashboard Workflows](#admin-dashboard-workflows)
5. [API Workflows (User-Facing)](#api-workflows-user-facing)
6. [Subscription Management Workflows](#subscription-management-workflows)
7. [Post Management Workflows](#post-management-workflows)
8. [Social Media Integration Workflows](#social-media-integration-workflows)
9. [Analytics & Reporting Workflows](#analytics--reporting-workflows)
10. [Data Flow Diagrams](#data-flow-diagrams)
11. [Technical Stack](#technical-stack)
12. [Deployment & Development Workflows](#deployment--development-workflows)

---

## Project Overview

**Platform Name**: Social Media Content Generation Platform  
**Framework**: Laravel 11.x  
**Type**: Multi-tenant SaaS platform for social media content creation and scheduling

### Core Features
- ✅ User authentication (API & Web)
- ✅ Role-based access control (Admin/Subscriber)
- ✅ Subscription plan management
- ✅ User management
- ✅ Post creation and scheduling
- ✅ Social media account integration (Facebook, Instagram, LinkedIn)
- ✅ Analytics and reporting
- ✅ AI-powered content generation (placeholder)
- ✅ Media management

### User Roles
1. **Admin** - Full system access, manages users, plans, subscriptions
2. **Subscriber** - Platform users who create and schedule posts

---

## Architecture Overview

### Technology Stack
- **Backend**: Laravel 11.x (PHP 8.2+)
- **Frontend**: Tailwind CSS, TailAdmin Template
- **Database**: MySQL/SQLite
- **Authentication**: Laravel Sanctum (API), Session-based (Web)
- **Queue System**: Laravel Queue (for scheduled posts)
- **Build Tool**: Vite

### Directory Structure
```
app/
├── Http/
│   ├── Controllers/
│   │   ├── Admin/      # Admin dashboard controllers
│   │   └── Api/         # API endpoints for users
│   ├── Middleware/      # Role-based access control
│   └── Requests/        # Form validation
├── Models/              # Eloquent models
├── Services/            # Business logic services
│   ├── Admin/
│   ├── Ai/
│   ├── Analytics/
│   └── Social/
└── Helpers/             # Helper classes

routes/
├── web.php              # Web routes (admin panel)
└── api.php              # API routes (user-facing)

database/
├── migrations/          # Database schema
└── seeders/            # Database seeders
```

---

## Authentication & Authorization Workflows

### 1. Admin Authentication Flow

**Route**: `/admin/login`

#### Workflow Steps:
1. **Admin accesses login page**
   - GET `/admin/login` → `AuthController@showLoginForm`
   - Returns `admin.login` view

2. **Admin submits credentials**
   - POST `/admin/login` → `AuthController@login`
   - Validates request using `LoginRequest`
   - Authenticates user via Laravel's Auth system
   - Validates role is 'admin'

3. **Session creation**
   - Regenerates session ID for security
   - Stores authenticated user in session

4. **Redirect**
   - On success: Redirects to `/admin/dashboard`
   - On failure: Returns to login with error message

5. **Logout**
   - POST `/admin/logout` → `AuthController@logout`
   - Invalidates session
   - Regenerates CSRF token
   - Redirects to `/admin/login`

**Middleware Protection**: `auth` + `role:admin`

### 2. API Authentication Flow

**Routes**: `/api/register`, `/api/login`

#### Registration Workflow:
```
1. POST /api/register
   ├── Request: { name, email, password }
   ├── Validates input
   ├── Creates User with role='user'
   ├── Hashes password
   └── Returns: { user } (201 Created)
```

#### Login Workflow:
```
1. POST /api/login
   ├── Request: { email, password }
   ├── Validates credentials via Auth::attempt()
   ├── Creates Sanctum token
   └── Returns: { token, user }

2. Subsequent API requests
   ├── Header: Authorization: Bearer {token}
   ├── Middleware: auth:sanctum
   └── Access granted to protected routes
```

### 3. Authorization Middleware

**RoleMiddleware** (`app/Http/Middleware/RoleMiddleware.php`)
- Checks user role matches required role
- Returns 403 Unauthorized if role mismatch
- Applied to admin routes via `role:admin`

---

## Admin Dashboard Workflows

### 1. Dashboard Overview

**Route**: `/admin/dashboard`  
**Controller**: `DashboardController@index`  
**Middleware**: `auth`, `role:admin`

#### Workflow:
```
1. Admin accesses dashboard
   ├── Retrieves statistics:
   │   ├── Total users count
   │   ├── Total subscription plans count
   │   └── Active subscriptions count
   └── Renders dashboard view with stats
```

### 2. User Management Workflow

**Routes**: `/admin/users` (Resource)

#### List Users (`GET /admin/users`)
```
1. Load users with pagination
   ├── Eager loads: subscription, subscription.plan
   ├── Paginates: 20 per page
   └── Returns: users.index view
```

#### DataTable Endpoint (`GET /admin/users/data`)
```
1. Handles DataTables AJAX request
   ├── Parameters: draw, start, length, search, order
   ├── Searchable columns: name, email, role, timezone, status
   ├── Returns JSON: { draw, recordsTotal, recordsFiltered, data }
   └── Data includes: name, email, role, timezone, status, created, actions
```

#### Create User (`POST /admin/users`)
```
1. Validates input
   ├── name: required|string|max:255
   ├── email: required|email|unique
   ├── password: required|min:6|confirmed
   ├── timezone: optional|string|max:50
   └── status: optional|in:active,inactive,suspended

2. Creates user
   ├── Hashes password
   ├── Sets role: 'subscriber'
   ├── Sets defaults: timezone='UTC', status='active'
   └── Redirects: users.index with success message
```

#### Update User (`PUT /admin/users/{user}`)
```
1. Validates input (same as create)
2. Updates user
   ├── Hashes password if provided
   ├── Prevents role changes (always subscriber)
   └── Redirects: users.index with success message
```

#### Delete User (`DELETE /admin/users/{user}`)
```
1. Validates deletion
   ├── Prevents admin account deletion
   └── Deletes user and related data
```

### 3. Subscription Plan Management Workflow

**Routes**: `/admin/plans` (Resource)

#### List Plans (`GET /admin/plans`)
```
1. Loads subscription plans
   ├── Orders by: created_at DESC
   ├── Paginates: 20 per page
   └── Returns: plans.index view
```

#### Create Plan (`POST /admin/plans`)
```
1. Validates via StorePlanRequest
   ├── name: required|string|max:100
   ├── price: required|numeric|min:0
   ├── duration: required|in:monthly,yearly
   ├── max_posts_per_month: required|integer|min:0
   ├── max_social_accounts: required|integer|min:0
   ├── ai_credits: required|integer|min:0
   ├── features: optional|string
   └── status: optional|in:active,inactive

2. Creates plan
   └── Redirects: plans.index with success message
```

#### Update Plan (`PUT /admin/plans/{plan}`)
```
1. Validates via UpdatePlanRequest
2. Updates plan
   └── Redirects: plans.index with success message
```

#### Delete Plan (`DELETE /admin/plans/{plan}`)
```
1. Validates deletion
   ├── Checks for active subscriptions
   ├── Prevents deletion if active subscriptions exist
   └── Deletes plan if safe
```

### 4. User Subscription Management Workflow

**Routes**: `/admin/users/{user}/subscription`

#### View Subscription (`GET /admin/users/{user}/subscription`)
```
1. Retrieves user's latest subscription
   ├── Loads subscription with plan relationship
   ├── Retrieves all available plans
   └── Returns JSON: { subscription, plans }
```

#### Create Subscription (`POST /admin/users/{user}/subscription`)
```
1. Validates input
   ├── plan_id: required|exists:subscription_plans,id
   ├── start_date: required|date
   ├── end_date: required|date|after_or_equal:start_date
   ├── status: required|in:active,expired,cancelled
   ├── payment_method: required|in:stripe,paypal,manual
   └── transaction_id: optional|string|max:150

2. Creates subscription
   ├── Sets user_id from route parameter
   └── Redirects: users.index with success message
```

#### Update Subscription (`PUT /admin/users/{user}/subscription/{subscription}`)
```
1. Validates input (same as create)
2. Updates subscription
   └── Redirects: users.index with success message
```

### 5. Reports Workflow

**Route**: `/admin/reports`  
**Controller**: `ReportController@index`

**Status**: Endpoint exists, implementation pending

---

## API Workflows (User-Facing)

### 1. User Profile Workflow

**Route**: `GET /api/user`  
**Middleware**: `auth:sanctum`

```
1. Validates Sanctum token
2. Returns authenticated user profile
   └── Response: { user object }
```

### 2. Post Management Workflow

**Routes**: `/api/posts` (Resource)

#### List Posts (`GET /api/posts`)
```
1. Authenticates user via Sanctum
2. Retrieves all posts for authenticated user
   └── Returns: [ { post objects } ]
```

#### Create Post (`POST /api/posts`)
```
1. Validates authentication
2. Creates post
   ├── Fields: user_id, content, status, scheduled_at
   ├── user_id: auto-assigned from authenticated user
   └── Returns: { post object }
```

#### View Post (`GET /api/posts/{post}`)
```
1. Validates authentication
2. Returns single post
   └── Response: { post object }
```

#### Update Post (`PUT /api/posts/{post}`)
```
1. Validates authentication
2. Updates post
   └── Returns: { updated post object }
```

#### Delete Post (`DELETE /api/posts/{post}`)
```
1. Validates authentication
2. Deletes post
   └── Returns: 204 No Content
```

### 3. Media Management Workflow

**Routes**: `/api/media` (Resource)

**Status**: Endpoint exists, implementation pending

### 4. Analytics Workflow

**Route**: `GET /api/analytics`  
**Controller**: `AnalyticsController@index`  
**Middleware**: `auth:sanctum`

**Status**: Endpoint exists, implementation pending

### 5. Social Account Management Workflow

**Route**: `/api/social-accounts`  
**Controller**: `SocialAccountController`

**Status**: Endpoint exists, implementation pending

---

## Subscription Management Workflows

### 1. Subscription Plan Structure

**Model**: `SubscriptionPlan`

**Attributes**:
- `name`: Plan name (e.g., "Basic", "Pro", "Enterprise")
- `price`: Decimal (10,2)
- `duration`: Enum ('monthly', 'yearly')
- `max_posts_per_month`: Integer
- `max_social_accounts`: Integer
- `ai_credits`: Integer
- `features`: LongText (newline-separated)
- `status`: Enum ('active', 'inactive')

### 2. Subscription Lifecycle

**Model**: `Subscription`

**Attributes**:
- `user_id`: Foreign key to users
- `plan_id`: Foreign key to subscription_plans
- `start_date`: Date
- `end_date`: Date
- `status`: Enum ('active', 'expired', 'cancelled')
- `payment_method`: Enum ('stripe', 'paypal', 'manual')
- `transaction_id`: String (nullable)

#### Subscription States:
```
1. Active
   ├── User can create posts
   ├── Within post limits
   └── Access to features based on plan

2. Expired
   ├── End date passed
   ├── User needs to renew
   └── Access limited

3. Cancelled
   ├── Manually cancelled
   ├── Access revoked
   └── May have grace period
```

### 3. Subscription Validation Workflow

**Before Post Creation**:
```
1. Check user has active subscription
   ├── Query: Subscription::where('user_id', $userId)
   │   ->where('status', 'active')
   │   ->where('end_date', '>=', now())
   │   ->latest()
   │   ->first()
   └── If null: Return error "No active subscription"

2. Check post limits
   ├── Count posts this month
   ├── Compare with plan.max_posts_per_month
   └── If exceeded: Return error "Post limit reached"

3. Check social account limits
   ├── Count connected social accounts
   ├── Compare with plan.max_social_accounts
   └── If exceeded: Return error "Social account limit reached"
```

---

## Post Management Workflows

### 1. Post Creation Workflow

**Model**: `Post`

**Attributes**:
- `user_id`: Foreign key to users
- `content`: Text (post content)
- `status`: Enum (pending, scheduled, published, failed)
- `scheduled_at`: DateTime (nullable)

#### Workflow Steps:
```
1. User creates post via API
   ├── POST /api/posts
   ├── Validates authentication
   ├── Validates subscription status
   ├── Validates post limits
   └── Creates post record

2. Post Scheduling Decision
   ├── If scheduled_at is null:
   │   └── Post immediately (status: 'pending')
   ├── If scheduled_at is future:
   │   ├── Status: 'scheduled'
   │   └── Queue job for scheduled time
   └── If scheduled_at is past:
       └── Reject with error

3. Post Publishing
   ├── For immediate posts:
   │   ├── Queue job processes immediately
   │   ├── Publishes to selected platforms
   │   └── Updates status: 'published'
   └── For scheduled posts:
       ├── Queue job scheduled for scheduled_at
       ├── Publishes at scheduled time
       └── Updates status: 'published'
```

### 2. Post Status Workflow

```
pending → processing → published
                    ↓
                 failed
                 
scheduled → [time reached] → processing → published
                            ↓
                         failed
```

### 3. Scheduled Post Execution

**Queue System**: Laravel Queue

```
1. Job scheduled via Laravel scheduler
   ├── Checks for posts where:
   │   ├── status = 'scheduled'
   │   ├── scheduled_at <= now()
   │   └── scheduled_at is not null

2. Job executes
   ├── Retrieves post
   ├── Retrieves user's social accounts
   ├── Publishes to each platform
   ├── Updates post status
   └── Logs success/failure
```

---

## Social Media Integration Workflows

### 1. Social Account Connection

**Model**: `SocialAccount`

**Supported Platforms**:
- Facebook (`FacebookService`)
- Instagram (`InstagramService`)
- LinkedIn (`LinkedInService`)

#### Connection Workflow:
```
1. User initiates OAuth flow
   ├── Redirects to platform OAuth URL
   ├── User authorizes application
   └── Platform redirects back with code

2. Exchange code for token
   ├── Backend exchanges code for access token
   ├── Stores token securely
   └── Creates SocialAccount record

3. SocialAccount stored
   ├── user_id: Foreign key
   ├── platform: Enum ('facebook', 'instagram', 'linkedin')
   ├── platform_user_id: String
   ├── access_token: Encrypted string
   ├── refresh_token: Encrypted string (optional)
   ├── expires_at: DateTime (optional)
   └── status: Enum ('active', 'expired', 'revoked')
```

### 2. Post Publishing Workflow

**Service Classes**: `FacebookService`, `InstagramService`, `LinkedInService`

#### Publishing Steps:
```
1. Post ready for publishing
   ├── Post status: 'pending' or 'scheduled'
   └── scheduled_at <= now()

2. Retrieve user's active social accounts
   ├── Query: SocialAccount::where('user_id', $userId)
   │   ->where('status', 'active')
   │   ->get()

3. For each platform:
   ├── Validate access token
   │   ├── Check expires_at
   │   ├── Refresh if needed
   │   └── Revoke if expired
   ├── Format post content
   │   ├── Platform-specific formatting
   │   ├── Add media if attached
   │   └── Handle hashtags/mentions
   ├── Publish via platform API
   │   ├── POST to platform API
   │   ├── Handle rate limits
   │   └── Handle errors
   └── Update post status
       ├── Success: 'published'
       └── Failure: 'failed' + error message
```

### 3. Token Refresh Workflow

```
1. Check token expiration
   ├── If expires_at <= now() + 1 hour:
   │   └── Trigger refresh

2. Refresh token
   ├── Call platform refresh endpoint
   ├── Update access_token
   ├── Update expires_at
   └── Update refresh_token if provided

3. Handle refresh failure
   ├── Mark account as 'expired'
   ├── Notify user
   └── Prevent publishing until re-authorized
```

---

## Analytics & Reporting Workflows

### 1. User Analytics Workflow

**Model**: `Analytics`

**Metrics Tracked**:
- Posts created
- Posts published
- Posts failed
- Posts scheduled
- Engagement metrics (likes, comments, shares)
- Best posting times
- Platform performance

#### Data Collection:
```
1. Post Creation
   ├── Log: Analytics::create([
   │   'user_id' => $userId,
   │   'event' => 'post_created',
   │   'data' => ['post_id' => $postId]
   │ ])

2. Post Published
   ├── Log: Analytics::create([
   │   'user_id' => $userId,
   │   'event' => 'post_published',
   │   'data' => [
   │     'post_id' => $postId,
   │     'platform' => $platform,
   │     'timestamp' => now()
   │   ]
   │ ])

3. Engagement Metrics
   ├── Periodic sync from platforms
   ├── Update Analytics records
   └── Aggregate for reporting
```

### 2. Admin Reporting Workflow

**Route**: `/admin/reports`  
**Controller**: `ReportController@index`

**Reports Available**:
- Total users by status
- Subscription revenue
- Active subscriptions
- Posts published (total/per user)
- Platform distribution
- User growth over time

---

## Data Flow Diagrams

### 1. User Registration & Subscription Flow

```
User Registration
    ↓
Create User Account (role: user)
    ↓
Browse Subscription Plans
    ↓
Select Plan
    ↓
Create Subscription (manual/stripe/paypal)
    ↓
Subscription Active
    ↓
Access Platform Features
```

### 2. Post Creation & Publishing Flow

```
User Creates Post
    ↓
Validate Subscription Status
    ↓
Validate Post Limits
    ↓
Save Post (status: pending/scheduled)
    ↓
[If Scheduled] Queue Job
    ↓
[When Time Reached] Process Job
    ↓
Retrieve User's Social Accounts
    ↓
For Each Platform:
    ├── Validate Token
    ├── Format Content
    ├── Publish via API
    └── Update Post Status
    ↓
Success: Status = 'published'
Failure: Status = 'failed' + Log Error
```

### 3. Admin User Management Flow

```
Admin Login
    ↓
Access User Management
    ↓
View Users List (with filters/search)
    ↓
[Create] → Add New User → Set Subscription
[Edit] → Update User → Modify Subscription
[Delete] → Validate → Delete User
[View Subscription] → Manage Subscription
    ↓
Update Complete
```

---

## Technical Stack

### Backend
- **Framework**: Laravel 11.x
- **PHP**: 8.2+
- **Database**: MySQL/SQLite
- **Authentication**: Laravel Sanctum (API), Session (Web)
- **Queue**: Laravel Queue (Database driver)

### Frontend
- **CSS Framework**: Tailwind CSS 3.4+
- **Admin Template**: TailAdmin
- **Build Tool**: Vite 6.0+
- **JavaScript**: Vanilla JS + DataTables

### Dependencies
- **laravel/sanctum**: API authentication
- **datatables.net**: DataTables integration
- **tailwindcss**: Utility-first CSS

### Development Tools
- **Laravel Pint**: Code formatting
- **Laravel Pail**: Log viewing
- **PHPUnit**: Testing
- **Concurrently**: Parallel task runner

---

## Deployment & Development Workflows

### 1. Initial Setup Workflow

```bash
# 1. Install dependencies
composer install
npm install

# 2. Configure environment
cp .env.example .env
php artisan key:generate

# 3. Configure database in .env
DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=social_platform
DB_USERNAME=root
DB_PASSWORD=

# 4. Run migrations and seeders
php artisan migrate --seed

# 5. Build assets
npm run build  # Production
# OR
npm run dev    # Development (with hot reload)

# 6. Start development server
php artisan serve
```

### 2. Development Workflow

```bash
# Run all services concurrently
composer dev

# This runs:
# - php artisan serve (Laravel server)
# - php artisan queue:listen (Queue worker)
# - php artisan pail (Log viewer)
# - npm run dev (Vite dev server)
```

### 3. Database Migration Workflow

```bash
# Create migration
php artisan make:migration create_table_name

# Run migrations
php artisan migrate

# Rollback last migration
php artisan migrate:rollback

# Fresh migration (WARNING: Drops all tables)
php artisan migrate:fresh --seed
```

### 4. Queue Processing Workflow

```bash
# Start queue worker
php artisan queue:work

# Process failed jobs
php artisan queue:retry all

# View failed jobs
php artisan queue:failed
```

### 5. Asset Building Workflow

```bash
# Development (watch mode)
npm run dev

# Production build
npm run build

# Assets will be in public/build/
```

### 6. Testing Workflow

```bash
# Run all tests
php artisan test

# Run specific test
php artisan test --filter TestName
```

### 7. Code Quality Workflow

```bash
# Format code
./vendor/bin/pint

# Lint code
./vendor/bin/pint --test
```

---

## Security Considerations

### 1. Authentication Security
- Passwords hashed with bcrypt
- Session regeneration on login
- CSRF protection on forms
- Sanctum token expiration

### 2. Authorization Security
- Role-based access control
- Middleware protection on routes
- Policy checks where applicable

### 3. Data Security
- SQL injection prevention (Eloquent ORM)
- XSS prevention (Blade escaping)
- Encrypted social account tokens
- Secure API endpoints

### 4. Rate Limiting
- API rate limiting (to be implemented)
- Queue job rate limiting
- Social platform API rate limits

---

## Future Enhancements

### Planned Features
1. **AI Content Generation**
   - Integration with AI service
   - Content suggestions
   - Image generation

2. **Advanced Analytics**
   - Real-time engagement tracking
   - Performance insights
   - Predictive analytics

3. **Enhanced Social Integration**
   - Twitter/X integration
   - TikTok integration
   - YouTube integration

4. **Automation Features**
   - Auto-posting based on schedule
   - Content recycling
   - Best time posting

5. **Team Collaboration**
   - Multi-user accounts
   - Content approval workflows
   - Team analytics

---

## API Reference

### Authentication Endpoints

| Method | Endpoint | Description | Auth Required |
|--------|----------|-------------|---------------|
| POST | `/api/register` | Register new user | No |
| POST | `/api/login` | Login user | No |
| GET | `/api/user` | Get user profile | Yes |

### Post Endpoints

| Method | Endpoint | Description | Auth Required |
|--------|----------|-------------|---------------|
| GET | `/api/posts` | List user posts | Yes |
| POST | `/api/posts` | Create post | Yes |
| GET | `/api/posts/{id}` | Get post | Yes |
| PUT | `/api/posts/{id}` | Update post | Yes |
| DELETE | `/api/posts/{id}` | Delete post | Yes |

### Media Endpoints

| Method | Endpoint | Description | Auth Required |
|--------|----------|-------------|---------------|
| GET | `/api/media` | List media | Yes |
| POST | `/api/media` | Upload media | Yes |

### Analytics Endpoints

| Method | Endpoint | Description | Auth Required |
|--------|----------|-------------|---------------|
| GET | `/api/analytics` | Get analytics | Yes |

---

## Admin Panel Routes

| Route | Method | Controller | Description |
|-------|--------|------------|-------------|
| `/admin/login` | GET | AuthController@showLoginForm | Login page |
| `/admin/login` | POST | AuthController@login | Login action |
| `/admin/logout` | POST | AuthController@logout | Logout |
| `/admin/dashboard` | GET | DashboardController@index | Dashboard |
| `/admin/users` | GET | UserController@index | List users |
| `/admin/users` | POST | UserController@store | Create user |
| `/admin/users/{id}` | PUT | UserController@update | Update user |
| `/admin/users/{id}` | DELETE | UserController@destroy | Delete user |
| `/admin/users/data` | GET | UserController@data | Users DataTable |
| `/admin/plans` | GET | PlanController@index | List plans |
| `/admin/plans` | POST | PlanController@store | Create plan |
| `/admin/plans/{id}` | PUT | PlanController@update | Update plan |
| `/admin/plans/{id}` | DELETE | PlanController@destroy | Delete plan |
| `/admin/users/{id}/subscription` | GET | SubscriptionController@show | View subscription |
| `/admin/users/{id}/subscription` | POST | SubscriptionController@store | Create subscription |
| `/admin/users/{id}/subscription/{sub}` | PUT | SubscriptionController@update | Update subscription |
| `/admin/reports` | GET | ReportController@index | Reports |

---

## Error Handling

### API Error Responses

```json
{
  "message": "Error description",
  "errors": {
    "field": ["Error message"]
  }
}
```

### HTTP Status Codes
- `200`: Success
- `201`: Created
- `204`: No Content
- `401`: Unauthorized
- `403`: Forbidden
- `404`: Not Found
- `422`: Validation Error
- `500`: Server Error

---

## Conclusion

This workflow document provides a comprehensive overview of the Social Media Content Generation Platform. It covers all major workflows, from user authentication to post publishing, subscription management, and admin operations.

For implementation details, refer to the actual controller and service files in the codebase.

---

**Document Version**: 1.0  
**Last Updated**: 2025-01-XX  
**Maintained By**: Development Team

