# Token Authentication Guide - How to Use API Tokens

## Overview
After login, you receive a token. This guide shows you how to use that token to authenticate API requests.

---

## Step 1: Get Token from Login

### Login Request
```http
POST http://localhost:8000/api/login
Content-Type: application/json

{
    "email": "ganesh@yopmail.com",
    "password": "123456"
}
```

### Login Response
```json
{
    "success": true,
    "message": "Login successful.",
    "data": {
        "token": "1|7eda8f53e2f0f3dcb2b52b0f196c4ed92ed21d815517ef0a7d707396cb7d0b17",
        "user": {
            "id": 1,
            "name": "Ganesh",
            "email": "ganesh@yopmail.com",
            "email_verified": false,
            "timezone": "UTC",
            "status": "active"
        }
    }
}
```

**Save the token**: `1|7eda8f53e2f0f3dcb2b52b0f196c4ed92ed21d815517ef0a7d707396cb7d0b17`

---

## Step 2: Use Token in API Requests

### How to Send Token

Add the token to the **Authorization** header in this format:

```
Authorization: Bearer {your_token_here}
```

**Important**: 
- Use the word `Bearer` (with capital B)
- Add a space after `Bearer`
- Then paste your full token

---

## Step 3: Testing in Postman

### Example: Get User Profile

1. **Method**: `GET`
2. **URL**: `http://localhost:8000/api/user`
3. **Headers Tab**:
   - Key: `Authorization`
   - Value: `Bearer 1|7eda8f53e2f0f3dcb2b52b0f196c4ed92ed21d815517ef0a7d707396cb7d0b17`
   - Also add: `Accept: application/json`

4. **Click Send**

### Postman Screenshot Guide

```
┌─────────────────────────────────────────┐
│ GET  http://localhost:8000/api/user     │
├─────────────────────────────────────────┤
│ Headers                                 │
├─────────────────────────────────────────┤
│ Authorization │ Bearer 1|7eda8f53...    │
│ Accept       │ application/json        │
└─────────────────────────────────────────┘
```

### Expected Response (200 OK)
```json
{
    "success": true,
    "data": {
        "id": 1,
        "name": "Ganesh",
        "email": "ganesh@yopmail.com",
        "email_verified": false,
        "timezone": "UTC",
        "status": "active"
    }
}
```

### Error Response (401 Unauthorized)
If token is missing or invalid:
```json
{
    "message": "Unauthenticated."
}
```

---

## Step 4: Using Token in Next.js Frontend

### Setup API Client

Create `lib/api.ts`:

```typescript
import axios from 'axios';

const API_BASE_URL = process.env.NEXT_PUBLIC_API_URL || 'http://localhost:8000';

const apiClient = axios.create({
  baseURL: `${API_BASE_URL}/api`,
  headers: {
    'Content-Type': 'application/json',
    'Accept': 'application/json',
  },
});

// Request interceptor - Automatically add token to all requests
apiClient.interceptors.request.use(
  (config) => {
    // Get token from localStorage
    const token = localStorage.getItem('auth_token');
    
    if (token) {
      // Add token to Authorization header
      config.headers.Authorization = `Bearer ${token}`;
    }
    
    return config;
  },
  (error) => {
    return Promise.reject(error);
  }
);

// Response interceptor - Handle 401 errors
apiClient.interceptors.response.use(
  (response) => response,
  (error) => {
    // If token is invalid or expired
    if (error.response?.status === 401) {
      // Clear token and redirect to login
      localStorage.removeItem('auth_token');
      localStorage.removeItem('user');
      window.location.href = '/login';
    }
    return Promise.reject(error);
  }
);

export default apiClient;
```

### Login and Store Token

```typescript
// Login function
const login = async (email: string, password: string) => {
  try {
    const response = await apiClient.post('/login', {
      email,
      password,
    });

    if (response.data.success) {
      // Store token in localStorage
      localStorage.setItem('auth_token', response.data.data.token);
      localStorage.setItem('user', JSON.stringify(response.data.data.user));
      
      return response.data;
    }
  } catch (error) {
    console.error('Login error:', error);
    throw error;
  }
};
```

### Make Authenticated Requests

```typescript
// Get user profile (token automatically added by interceptor)
const getUserProfile = async () => {
  try {
    const response = await apiClient.get('/user');
    return response.data;
  } catch (error) {
    console.error('Error fetching profile:', error);
    throw error;
  }
};

// Logout (token automatically added)
const logout = async () => {
  try {
    await apiClient.post('/logout');
  } finally {
    // Clear token regardless of API call success
    localStorage.removeItem('auth_token');
    localStorage.removeItem('user');
  }
};
```

### React Component Example

```typescript
'use client';

import { useEffect, useState } from 'react';
import apiClient from '@/lib/api';

export default function Dashboard() {
  const [user, setUser] = useState(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    // Check if token exists
    const token = localStorage.getItem('auth_token');
    
    if (!token) {
      window.location.href = '/login';
      return;
    }

    // Fetch user profile (token automatically added)
    const fetchUser = async () => {
      try {
        const response = await apiClient.get('/user');
        setUser(response.data.data);
      } catch (error) {
        console.error('Error:', error);
        // Token might be invalid, redirect to login
        window.location.href = '/login';
      } finally {
        setLoading(false);
      }
    };

    fetchUser();
  }, []);

  if (loading) return <div>Loading...</div>;
  if (!user) return null;

  return (
    <div>
      <h1>Welcome, {user.name}!</h1>
      <p>Email: {user.email}</p>
    </div>
  );
}
```

---

## Step 5: Protected Routes in Laravel

### How Routes Are Protected

In `routes/api.php`:

```php
// Public routes (no token required)
Route::post('/login', [AuthController::class, 'login']);
Route::post('/register', [AuthController::class, 'register']);

// Protected routes (token required)
Route::middleware('auth:sanctum')->group(function () {
    Route::get('/user', [AuthController::class, 'profile']);
    Route::post('/logout', [AuthController::class, 'logout']);
    Route::apiResource('posts', PostController::class);
    Route::get('/analytics', [AnalyticsController::class, 'index']);
});
```

### How Middleware Works

1. **Request comes in** with `Authorization: Bearer {token}` header
2. **Sanctum middleware** (`auth:sanctum`) extracts the token
3. **Looks up token** in `personal_access_tokens` table
4. **Validates token** (checks hash, expiration, etc.)
5. **If valid**: Loads the user and makes available via `$request->user()`
6. **If invalid**: Returns 401 Unauthorized

### Accessing Authenticated User in Controller

```php
public function profile(Request $request)
{
    // Get authenticated user
    $user = $request->user(); // Returns AppUser instance
    
    return response()->json([
        'success' => true,
        'data' => $user
    ]);
}
```

---

## Step 6: Testing All Protected Endpoints

### 1. Get User Profile
```http
GET http://localhost:8000/api/user
Authorization: Bearer {token}
Accept: application/json
```

### 2. Logout
```http
POST http://localhost:8000/api/logout
Authorization: Bearer {token}
Accept: application/json
```

### 3. Get Posts (if exists)
```http
GET http://localhost:8000/api/posts
Authorization: Bearer {token}
Accept: application/json
```

### 4. Create Post (if exists)
```http
POST http://localhost:8000/api/posts
Authorization: Bearer {token}
Content-Type: application/json
Accept: application/json

{
    "title": "My Post",
    "content": "Post content here"
}
```

---

## Common Issues and Solutions

### Issue 1: 401 Unauthorized

**Causes:**
- Token missing from header
- Token format wrong (missing `Bearer ` prefix)
- Token expired or revoked
- Token doesn't exist in database

**Solutions:**
- Check header: `Authorization: Bearer {token}` (with space after Bearer)
- Login again to get new token
- Check token exists in `personal_access_tokens` table

### Issue 2: Token Not Working

**Check:**
1. Token format: `{id}|{token}` (should have `|` separator)
2. Full token copied (not truncated)
3. Header name: `Authorization` (not `Auth` or `Token`)
4. Header value: `Bearer {token}` (with space)

### Issue 3: CORS Errors

**Solution**: Configure CORS in Laravel `config/cors.php`:
```php
'allowed_origins' => [
    'http://localhost:3000', // Next.js dev server
],
```

---

## Token Lifecycle

### 1. Create Token (Login)
```php
$token = $user->createToken('web_app_token')->plainTextToken;
// Returns: "1|7eda8f53..."
```

### 2. Use Token (API Requests)
```http
Authorization: Bearer 1|7eda8f53...
```

### 3. Validate Token (Middleware)
- Sanctum automatically validates token
- Loads user if valid
- Returns 401 if invalid

### 4. Revoke Token (Logout)
```php
$request->user()->currentAccessToken()->delete();
```

---

## Best Practices

### 1. Store Token Securely
```typescript
// ✅ Good: localStorage (for web apps)
localStorage.setItem('auth_token', token);

// ❌ Bad: Don't store in global variables
window.token = token; // Don't do this
```

### 2. Add Token Automatically
```typescript
// ✅ Good: Use axios interceptor
apiClient.interceptors.request.use((config) => {
  const token = localStorage.getItem('auth_token');
  if (token) {
    config.headers.Authorization = `Bearer ${token}`;
  }
  return config;
});
```

### 3. Handle Token Expiration
```typescript
// ✅ Good: Check 401 and redirect
apiClient.interceptors.response.use(
  (response) => response,
  (error) => {
    if (error.response?.status === 401) {
      localStorage.removeItem('auth_token');
      window.location.href = '/login';
    }
    return Promise.reject(error);
  }
);
```

### 4. Clear Token on Logout
```typescript
// ✅ Good: Clear token from storage
const logout = async () => {
  await apiClient.post('/logout');
  localStorage.removeItem('auth_token');
  localStorage.removeItem('user');
};
```

---

## Quick Reference

### Token Format
```
Authorization: Bearer {token_id}|{plain_token}
```

### Postman Setup
```
Header: Authorization
Value: Bearer 1|7eda8f53e2f0f3dcb2b52b0f196c4ed92ed21d815517ef0a7d707396cb7d0b17
```

### Next.js Setup
```typescript
// Store token
localStorage.setItem('auth_token', token);

// Use in requests (automatic with interceptor)
apiClient.get('/user'); // Token added automatically
```

### Laravel Controller
```php
// Get authenticated user
$user = $request->user(); // AppUser instance
```

---

## Summary

✅ **Get Token**: Login endpoint returns token
✅ **Store Token**: Save in localStorage (Next.js) or variable
✅ **Send Token**: Add to `Authorization: Bearer {token}` header
✅ **Automatic**: Use axios interceptor for automatic token injection
✅ **Protected Routes**: Use `auth:sanctum` middleware
✅ **Access User**: Use `$request->user()` in controllers

The token is now working! Use it in the `Authorization` header for all protected API requests.

