# Schedule Post Implementation Guide

## 📋 Current State Analysis

### What's Already Implemented:

1. ✅ **UI Components**: Schedule date/time pickers exist in `PostComposer.tsx`

   - Radio buttons: "Post Now" vs "Schedule for Later"
   - Date picker and time picker (shown when "Schedule for Later" is selected)
   - State management: `scheduleType`, `scheduleDate`, `scheduleTime`

2. ✅ **Current Publish Flow**:
   - `handlePublish()` in `app/compose/page.tsx`
   - Calls `publishPost()` from `lib/postsService.ts`
   - Immediately publishes to backend via `PUT /api/posts/{id}/publish`

### What's Missing:

1. ❌ **Post Interface**: No `scheduled_at` field in `Post` interface
2. ❌ **API Integration**: Schedule data not sent to backend
3. ❌ **Backend Support**: No `scheduled_at` field in database/API
4. ❌ **Background Jobs**: No queue system to publish at scheduled time
5. ❌ **Validation**: No date/time validation (must be future date)

---

## 🎯 Implementation Approaches

### **Approach 1: Scheduled Status (Recommended)**

**Concept**: Add a new status `"scheduled"` and use `scheduled_at` field.

**Flow**:

1. User selects "Schedule for Later" → sets `scheduled_at` datetime
2. Post status = `"scheduled"` (not `"draft"` or `"posted"`)
3. Backend stores post with `scheduled_at` timestamp
4. Background job (Laravel Queue/Cron) checks for posts where:
   - `status = "scheduled"`
   - `scheduled_at <= NOW()`
5. Job publishes post (changes status to `"posted"` and sets `published_at`)

**Pros**:

- Clear status separation
- Easy to query scheduled posts
- Can show "Scheduled" posts in UI separately
- Can edit/delete scheduled posts before they publish

**Cons**:

- Requires new status value
- Need to update all status checks

---

### **Approach 2: Draft with Scheduled At**

**Concept**: Keep status as `"draft"` but add `scheduled_at` field.

**Flow**:

1. User selects "Schedule for Later" → sets `scheduled_at` datetime
2. Post status = `"draft"` with `scheduled_at` set
3. Backend stores post with `scheduled_at` timestamp
4. Background job checks for posts where:
   - `status = "draft"`
   - `scheduled_at IS NOT NULL`
   - `scheduled_at <= NOW()`
5. Job publishes post (changes status to `"posted"`)

**Pros**:

- No new status needed
- Simpler implementation
- Scheduled posts appear in drafts list

**Cons**:

- Harder to distinguish scheduled vs regular drafts
- Need to filter by `scheduled_at IS NOT NULL`

---

### **Approach 3: Separate Schedule Endpoint**

**Concept**: Create new endpoint `POST /api/posts/{id}/schedule` instead of using publish.

**Flow**:

1. User selects "Schedule for Later" → sets `scheduled_at` datetime
2. Frontend calls `schedulePost(id, scheduled_at)` instead of `publishPost()`
3. Backend stores post with `scheduled_at` and status = `"scheduled"`
4. Background job publishes at scheduled time

**Pros**:

- Clear separation of concerns
- Can have different validation rules
- Easier to track scheduled vs immediate publishes

**Cons**:

- More API endpoints to maintain
- More frontend logic

---

## 🚀 Recommended Implementation (Approach 1)

### Step 1: Update Frontend Types

**File**: `lib/postsService.ts`

```typescript
export interface Post {
  id: number;
  app_user_id: number;
  content: string;
  content_html?: string;
  status: "draft" | "scheduled" | "posted"; // Add "scheduled"
  category: string;
  platforms: string[];
  published_at: string | null;
  scheduled_at: string | null; // Add this field
  created_at: string;
  updated_at: string;
  media?: PostMedia[];
  media_count?: number;
}

export interface CreatePostData {
  content: string;
  content_html?: string;
  category?: string;
  platforms?: string[];
  files?: File[];
  scheduled_at?: string; // Add this
}

export interface UpdatePostData {
  content?: string;
  content_html?: string;
  category?: string;
  platforms?: string[];
  files?: File[];
  remove_media_ids?: number[];
  scheduled_at?: string; // Add this
}
```

### Step 2: Update PostComposer to Pass Schedule Data

**File**: `components/SocialFlow/PostComposer.tsx`

Add callback to pass schedule data to parent:

```typescript
interface PostComposerProps {
  // ... existing props
  onScheduleChange?: (scheduleData: {
    type: "now" | "later";
    date?: string;
    time?: string;
  }) => void;
}

// In component:
useEffect(() => {
  if (onScheduleChange) {
    onScheduleChange({
      type: scheduleType as "now" | "later",
      date: scheduleType === "later" ? scheduleDate : undefined,
      time: scheduleType === "later" ? scheduleTime : undefined,
    });
  }
}, [scheduleType, scheduleDate, scheduleTime, onScheduleChange]);
```

### Step 3: Update Compose Page to Handle Schedule

**File**: `app/compose/page.tsx`

```typescript
const [scheduleData, setScheduleData] = useState<{
  type: "now" | "later";
  date?: string;
  time?: string;
}>({ type: "now" });

// In handlePublish:
const handlePublish = async () => {
  // ... validation ...

  // Calculate scheduled_at if scheduling
  let scheduledAt: string | undefined;
  if (scheduleData.type === "later" && scheduleData.date && scheduleData.time) {
    // Combine date and time into ISO string
    const dateTime = new Date(`${scheduleData.date}T${scheduleData.time}`);

    // Validate it's in the future
    if (dateTime <= new Date()) {
      setPublishStatus({
        type: "error",
        message: "Scheduled time must be in the future.",
      });
      return;
    }

    scheduledAt = dateTime.toISOString();
  }

  // ... rest of publish logic ...

  // When calling publishPost, include scheduled_at
  const publishResponse = await publishPost(
    postId,
    selectedPlatforms.length > 0 ? selectedPlatforms : undefined,
    scheduledAt // Pass scheduled_at
  );
};
```

### Step 4: Update publishPost Function

**File**: `lib/postsService.ts`

```typescript
export async function publishPost(
  id: number,
  platforms?: string[],
  scheduledAt?: string // Add this parameter
): Promise<ApiResponse<PostResponse>> {
  // ... existing code ...

  const body: any = {};
  if (platforms && platforms.length > 0) {
    body.platforms = platforms;
  }
  if (scheduledAt) {
    body.scheduled_at = scheduledAt; // Add scheduled_at to request
  }

  // ... rest of function ...
}
```

### Step 5: Backend Changes Required

**Laravel Backend** (`app/Http/Controllers/Api/PostController.php`):

1. **Add `scheduled_at` to database migration**:

```php
Schema::table('posts', function (Blueprint $table) {
    $table->timestamp('scheduled_at')->nullable()->after('published_at');
    $table->index('scheduled_at'); // For efficient querying
});
```

2. **Update Post Model** (`app/Models/Post.php`):

```php
protected $fillable = [
    // ... existing fields
    'scheduled_at',
];

protected $casts = [
    'scheduled_at' => 'datetime',
    'published_at' => 'datetime',
];
```

3. **Update Publish Endpoint** (`update()` or `publish()` method):

```php
public function publish(Request $request, $id)
{
    $post = Post::findOrFail($id);

    // Validate scheduled_at if provided
    $validated = $request->validate([
        'platforms' => 'sometimes|array',
        'platforms.*' => 'string',
        'scheduled_at' => 'sometimes|date|after:now',
    ]);

    if (isset($validated['scheduled_at'])) {
        // Schedule for later
        $post->scheduled_at = $validated['scheduled_at'];
        $post->status = 'scheduled'; // New status
        $post->save();

        return response()->json([
            'success' => true,
            'message' => 'Post scheduled successfully',
            'data' => $post->load('media'),
        ]);
    } else {
        // Publish immediately
        $post->status = 'posted';
        $post->published_at = now();
        $post->scheduled_at = null;
        $post->save();

        return response()->json([
            'success' => true,
            'message' => 'Post published successfully',
            'data' => $post->load('media'),
        ]);
    }
}
```

4. **Create Background Job** (`app/Jobs/PublishScheduledPosts.php`):

```php
<?php

namespace App\Jobs;

use App\Models\Post;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Carbon\Carbon;

class PublishScheduledPosts implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function handle()
    {
        $now = Carbon::now();

        // Find posts scheduled to publish now or in the past
        $posts = Post::where('status', 'scheduled')
            ->where('scheduled_at', '<=', $now)
            ->get();

        foreach ($posts as $post) {
            $post->status = 'posted';
            $post->published_at = $post->scheduled_at ?? now();
            $post->scheduled_at = null;
            $post->save();

            // TODO: Add social media publishing logic here
        }
    }
}
```

5. **Create Console Command** (`app/Console/Commands/PublishScheduledPosts.php`):

```php
<?php

namespace App\Console\Commands;

use App\Jobs\PublishScheduledPosts;
use Illuminate\Console\Command;

class PublishScheduledPostsCommand extends Command
{
    protected $signature = 'posts:publish-scheduled';
    protected $description = 'Publish scheduled posts';

    public function handle()
    {
        dispatch(new PublishScheduledPosts());
        $this->info('Scheduled posts check completed');
    }
}
```

6. **Add to Cron** (`app/Console/Kernel.php`):

```php
protected function schedule(Schedule $schedule)
{
    // Run every minute to check for scheduled posts
    $schedule->command('posts:publish-scheduled')
        ->everyMinute();
}
```

---

## 📊 Alternative Publishing Methods

### Method 1: Immediate Publish (Current)

- User clicks "Publish Post" → Post published immediately
- Status: `"posted"`, `published_at` = now()

### Method 2: Scheduled Publish (New)

- User selects "Schedule for Later" → Post scheduled
- Status: `"scheduled"`, `scheduled_at` = future datetime
- Background job publishes at scheduled time

### Method 3: Draft with Auto-Publish

- Save as draft → Later manually publish
- Status: `"draft"` → User clicks "Publish" → Status: `"posted"`

### Method 4: Queue-Based Publishing

- Use Laravel Queue for all publishes (immediate or scheduled)
- Better for handling social media API rate limits
- Can retry failed publishes

---

## 🔧 Implementation Checklist

### Frontend:

- [ ] Update `Post` interface to include `scheduled_at` and `"scheduled"` status
- [ ] Update `PostComposer` to pass schedule data to parent
- [ ] Update `handlePublish` to calculate and validate `scheduled_at`
- [ ] Update `publishPost` function to accept `scheduled_at` parameter
- [ ] Add date/time validation (must be future)
- [ ] Update UI to show scheduled posts differently
- [ ] Add ability to edit/delete scheduled posts

### Backend:

- [ ] Add `scheduled_at` column to `posts` table
- [ ] Update Post model to include `scheduled_at`
- [ ] Update publish endpoint to handle `scheduled_at`
- [ ] Add validation for `scheduled_at` (must be future)
- [ ] Create background job to publish scheduled posts
- [ ] Create console command to run job
- [ ] Add cron schedule to run every minute
- [ ] Update API responses to include `scheduled_at`

### Testing:

- [ ] Test immediate publish (no schedule)
- [ ] Test scheduled publish (future date/time)
- [ ] Test validation (past date/time should fail)
- [ ] Test background job publishes at correct time
- [ ] Test editing scheduled post
- [ ] Test deleting scheduled post
- [ ] Test timezone handling

---

## 🎨 UI/UX Considerations

1. **Schedule Date/Time Picker**:

   - Set minimum date to today
   - Set minimum time to current time if today
   - Show timezone to user
   - Add validation messages

2. **Scheduled Posts List**:

   - Show scheduled posts separately from drafts
   - Display scheduled date/time
   - Show countdown "Publishing in X hours"
   - Allow editing/canceling scheduled posts

3. **Status Indicators**:

   - Draft: Gray badge
   - Scheduled: Blue badge with clock icon
   - Posted: Green badge

4. **Notifications**:
   - Show success message when scheduled
   - Send notification when post publishes (optional)

---

## 📝 Example API Request/Response

### Request (Schedule for Later):

```json
PUT /api/posts/123/publish
{
  "platforms": ["facebook", "twitter"],
  "scheduled_at": "2025-12-01T14:30:00Z"
}
```

### Response:

```json
{
  "success": true,
  "message": "Post scheduled successfully",
  "data": {
    "id": 123,
    "status": "scheduled",
    "scheduled_at": "2025-12-01T14:30:00Z",
    "published_at": null
    // ... other fields
  }
}
```

---

## 🚨 Important Notes

1. **Timezone Handling**: Always store `scheduled_at` in UTC, convert to user's timezone in UI
2. **Validation**: Always validate `scheduled_at` is in the future
3. **Background Jobs**: Use Laravel Queue for reliability (not just cron)
4. **Error Handling**: Handle cases where scheduled post fails to publish
5. **Retry Logic**: Implement retry for failed social media publishes
6. **Rate Limiting**: Consider social media API rate limits when scheduling

---

## 🔄 Migration Path

1. **Phase 1**: Add `scheduled_at` field (nullable)
2. **Phase 2**: Update API to accept `scheduled_at`
3. **Phase 3**: Add background job
4. **Phase 4**: Update UI to show scheduled posts
5. **Phase 5**: Add edit/cancel scheduled posts feature
