# Scheduled Posts - Quick Start Guide

## 🎯 Simple Explanation

**How it works:**

1. User schedules a post → Post saved with `status='scheduled'` and `scheduled_at` time
2. Every minute, Laravel checks for posts ready to publish
3. When found, posts are automatically published (`status='posted'`)

---

## ⚡ Quick Setup (3 Steps)

### Step 1: Run Migration

```bash
php artisan migrate
```

### Step 2: Choose Your Method

#### **Method A: Simple (No Queue) - Recommended for Development**

The scheduler is already configured to run synchronously. Just set up the cron:

**Linux/Mac:**

```bash
crontab -e
# Add this line:
* * * * * cd /path-to-project && php artisan schedule:run >> /dev/null 2>&1
```

**Windows:**
Run this in PowerShell (keep it running):

```powershell
while ($true) {
    php artisan schedule:run
    Start-Sleep -Seconds 60
}
```

**That's it!** Posts will publish automatically every minute.

---

#### **Method B: With Queue (Production)**

1. **Start Queue Worker:**

```bash
php artisan queue:work
```

2. **Update scheduler** (in `routes/console.php`):

```php
// Remove --sync flag
Schedule::command('posts:publish-scheduled')
    ->everyMinute()
    ->withoutOverlapping()
    ->runInBackground();
```

3. **Set up cron** (same as Method A)

---

## 🧪 Test It

### 1. Create a Scheduled Post

```bash
PUT /api/posts/1/publish
{
  "scheduled_at": "2025-01-23T15:05:00Z"  # 5 minutes from now
}
```

### 2. Run Command Manually

```bash
# Synchronous (no queue needed)
php artisan posts:publish-scheduled --sync

# Or with queue
php artisan posts:publish-scheduled
```

### 3. Check Results

```bash
# Check logs
tail -f storage/logs/laravel.log

# Or check database
php artisan tinker
>>> App\Models\Post::where('status', 'scheduled')->count()
```

---

## 📊 How It Works (Visual)

```
┌─────────────────────────────────────────┐
│  User Schedules Post                    │
│  PUT /api/posts/1/publish               │
│  { scheduled_at: "2025-01-23 15:00" }   │
└──────────────┬──────────────────────────┘
               │
               ▼
┌─────────────────────────────────────────┐
│  Post Saved                             │
│  status: 'scheduled'                    │
│  scheduled_at: "2025-01-23 15:00"      │
└──────────────┬──────────────────────────┘
               │
               │ (Wait until scheduled time)
               │
               ▼
┌─────────────────────────────────────────┐
│  Laravel Scheduler (Every Minute)      │
│  php artisan schedule:run              │
└──────────────┬──────────────────────────┘
               │
               ▼
┌─────────────────────────────────────────┐
│  Command: posts:publish-scheduled      │
│  Checks: scheduled_at <= NOW()         │
└──────────────┬──────────────────────────┘
               │
               ▼
┌─────────────────────────────────────────┐
│  Job: PublishScheduledPosts            │
│  Finds posts ready to publish          │
└──────────────┬──────────────────────────┘
               │
               ▼
┌─────────────────────────────────────────┐
│  Post Published!                       │
│  status: 'posted'                       │
│  published_at: now()                    │
│  scheduled_at: null                      │
└─────────────────────────────────────────┘
```

---

## 🔍 Verify It's Working

### Check Scheduler

```bash
php artisan schedule:list
```

You should see:

```
posts:publish-scheduled --sync  ... Every minute
```

### Check Logs

```bash
tail -f storage/logs/laravel.log | grep PublishScheduledPosts
```

You should see:

```
PublishScheduledPosts: Found 1 scheduled post(s) to publish
PublishScheduledPosts: Published post ID 1
```

### Check Database

```bash
php artisan tinker
```

```php
// See scheduled posts
Post::where('status', 'scheduled')->get();

// See recently published posts
Post::where('status', 'posted')
    ->where('published_at', '>=', now()->subHour())
    ->get();
```

---

## ⚙️ Configuration

### Change Check Frequency

In `routes/console.php`:

```php
// Every 5 minutes instead of every minute
Schedule::command('posts:publish-scheduled --sync')
    ->everyFiveMinutes()
    ->withoutOverlapping();
```

### Switch Between Sync/Queue

**Sync Mode (Simple):**

```php
Schedule::command('posts:publish-scheduled --sync')
```

**Queue Mode (Production):**

```php
Schedule::command('posts:publish-scheduled')
    ->runInBackground();
```

---

## 🚨 Troubleshooting

### Posts Not Publishing?

1. **Check if scheduler is running:**

    ```bash
    php artisan schedule:run -v
    ```

2. **Run command manually:**

    ```bash
    php artisan posts:publish-scheduled --sync
    ```

3. **Check logs:**

    ```bash
    tail -f storage/logs/laravel.log
    ```

4. **Verify post is scheduled:**
    ```bash
    php artisan tinker
    >>> $post = Post::find(1);
    >>> $post->status;  // Should be 'scheduled'
    >>> $post->scheduled_at;  // Should be a future date
    ```

---

## ✅ Checklist

-   [ ] Migration run: `php artisan migrate`
-   [ ] Scheduler configured in `routes/console.php`
-   [ ] Cron set up (or manual testing)
-   [ ] Test post created with `scheduled_at`
-   [ ] Command tested manually
-   [ ] Logs checked
-   [ ] Posts publishing correctly

---

## 📝 Summary

**Minimum Setup:**

1. Run migration
2. Set up cron to run `php artisan schedule:run` every minute
3. Done!

**The system will:**

-   Check every minute for scheduled posts
-   Automatically publish posts when `scheduled_at <= NOW()`
-   Log everything for debugging

**No queue worker needed** if you use `--sync` flag (default in current setup).

That's it! Your scheduled posts feature is ready to use! 🎉
