# Scheduled Posts - Environment Setup Guide

## 🎯 Recommendation

### **Development: Use Synchronous (--sync)**

✅ **Recommended for Development**

-   Simple setup - no queue worker needed
-   Easier debugging - see errors immediately
-   No additional processes to manage
-   Perfect for local development

### **Production: Use Queue System**

✅ **Recommended for Production**

-   More reliable - handles failures better
-   Scalable - can process multiple jobs
-   Non-blocking - doesn't slow down scheduler
-   Better for high-traffic applications

---

## ⚙️ Current Configuration

The system is **automatically configured** based on your environment:

-   **Development** (`APP_ENV=local`): Uses `--sync` (synchronous)
-   **Production** (`APP_ENV=production`): Uses queue system

**No manual changes needed!** It switches automatically.

---

## 📋 Setup by Environment

### **Development Setup**

#### 1. Set Environment

In `.env`:

```env
APP_ENV=local
```

#### 2. Run Migration

```bash
php artisan migrate
```

#### 3. Set Up Scheduler (Choose One)

**Option A: Manual Testing (Easiest)**

```bash
# Run manually when needed
php artisan schedule:run
```

**Option B: Automatic (Cron)**

```bash
# Linux/Mac
crontab -e
# Add:
* * * * * cd /path-to-project && php artisan schedule:run >> /dev/null 2>&1

# Windows (PowerShell - keep running)
while ($true) {
    php artisan schedule:run
    Start-Sleep -Seconds 60
}
```

**That's it!** No queue worker needed for development.

---

### **Production Setup**

#### 1. Set Environment

In `.env`:

```env
APP_ENV=production
QUEUE_CONNECTION=database
```

#### 2. Run Migration

```bash
php artisan migrate
```

#### 3. Start Queue Worker

**Using Supervisor (Recommended):**

Create `/etc/supervisor/conf.d/laravel-worker.conf`:

```ini
[program:laravel-worker]
process_name=%(program_name)s_%(process_num)02d
command=php /path-to-project/artisan queue:work database --sleep=3 --tries=3 --max-time=3600
autostart=true
autorestart=true
stopasgroup=true
killasgroup=true
user=www-data
numprocs=1
redirect_stderr=true
stdout_logfile=/path-to-project/storage/logs/worker.log
stopwaitsecs=3600
```

Then:

```bash
sudo supervisorctl reread
sudo supervisorctl update
sudo supervisorctl start laravel-worker:*
```

**Or manually:**

```bash
php artisan queue:work
```

#### 4. Set Up Scheduler (Cron)

```bash
crontab -e
# Add:
* * * * * cd /path-to-project && php artisan schedule:run >> /dev/null 2>&1
```

---

## 🔄 Manual Override

If you want to manually control which method to use:

### Force Synchronous (Even in Production)

In `routes/console.php`:

```php
Schedule::command('posts:publish-scheduled --sync')
    ->everyMinute()
    ->withoutOverlapping();
```

### Force Queue (Even in Development)

In `routes/console.php`:

```php
Schedule::command('posts:publish-scheduled')
    ->everyMinute()
    ->withoutOverlapping()
    ->runInBackground();
```

---

## 📊 Comparison

| Feature                 | Development (Sync) | Production (Queue)       |
| ----------------------- | ------------------ | ------------------------ |
| **Setup Complexity**    | ⭐ Simple          | ⭐⭐ Medium              |
| **Queue Worker Needed** | ❌ No              | ✅ Yes                   |
| **Performance**         | ⭐⭐ Good          | ⭐⭐⭐ Excellent         |
| **Error Visibility**    | ⭐⭐⭐ Immediate   | ⭐⭐ Logs                |
| **Scalability**         | ⭐ Limited         | ⭐⭐⭐ High              |
| **Reliability**         | ⭐⭐ Good          | ⭐⭐⭐ Excellent         |
| **Best For**            | Local dev, testing | Production, high traffic |

---

## ✅ Quick Checklist

### Development

-   [ ] `APP_ENV=local` in `.env`
-   [ ] Migration run
-   [ ] Scheduler set up (cron or manual)
-   [ ] **No queue worker needed**

### Production

-   [ ] `APP_ENV=production` in `.env`
-   [ ] `QUEUE_CONNECTION=database` in `.env`
-   [ ] Migration run
-   [ ] Queue worker running (Supervisor or manual)
-   [ ] Scheduler set up (cron)

---

## 🧪 Testing

### Test Development Mode

```bash
# Set environment
APP_ENV=local

# Run command (should use --sync)
php artisan posts:publish-scheduled

# Should see: "Running synchronously..."
```

### Test Production Mode

```bash
# Set environment
APP_ENV=production

# Start queue worker first
php artisan queue:work

# Run command (should use queue)
php artisan posts:publish-scheduled

# Should see: "Dispatching to queue..."
```

---

## 🎯 Summary

**The system automatically chooses the best method:**

-   **Development** → Synchronous (simple, no queue worker)
-   **Production** → Queue (reliable, scalable)

**You don't need to change anything!** Just set `APP_ENV` correctly and the system handles the rest.

---

## 🚨 Troubleshooting

### Development: Command Not Running

-   Check `APP_ENV=local` in `.env`
-   Run manually: `php artisan posts:publish-scheduled --sync`
-   Check logs: `tail -f storage/logs/laravel.log`

### Production: Jobs Not Processing

-   Check `APP_ENV=production` in `.env`
-   Check queue worker is running: `ps aux | grep "queue:work"`
-   Check `QUEUE_CONNECTION=database` in `.env`
-   Check failed jobs: `php artisan queue:failed`

---

**That's it!** The system is smart enough to use the right method for each environment automatically. 🎉
