# Scheduled Posts Backend Implementation Summary

## ✅ Implementation Complete

This document summarizes the backend implementation for the scheduled posts feature as per the `SCHEDULE_POST_IMPLEMENTATION_GUIDE.md`.

## 📋 What Was Implemented

### 1. Database Migration ✅

**File**: `database/migrations/2025_01_23_000000_add_scheduled_at_to_posts_table.php`

-   Added `scheduled_at` timestamp column (nullable)
-   Updated `status` enum to include `'scheduled'` (now: `'draft'`, `'scheduled'`, `'posted'`)
-   Added indexes for efficient querying:
    -   Index on `scheduled_at`
    -   Composite index on `['status', 'scheduled_at']`

### 2. Post Model Updates ✅

**File**: `app/Models/Post.php`

-   Added `scheduled_at` to `$fillable` array
-   Added `scheduled_at` to `casts()` as `datetime`

### 3. PostController Updates ✅

**File**: `app/Http/Controllers/Api/PostController.php`

#### Publish Method (`publish()`)

-   Added validation for `scheduled_at` parameter (must be future date)
-   Handles two scenarios:
    -   **Immediate Publish**: If `scheduled_at` is not provided, publishes immediately
    -   **Scheduled Publish**: If `scheduled_at` is provided, sets status to `'scheduled'` and stores the scheduled time
-   Returns appropriate success messages for both scenarios

#### Update Method (`update()`)

-   Updated validation to allow `'scheduled'` status
-   Added validation for `scheduled_at` field
-   Handles `scheduled_at` updates:
    -   If `scheduled_at` is set, automatically sets status to `'scheduled'`
    -   If `scheduled_at` is cleared, changes status from `'scheduled'` to `'draft'`
-   Properly manages `published_at` and `scheduled_at` relationships

#### AssertDraft Method

-   Updated to allow both `'draft'` and `'scheduled'` posts to be updated
-   Changed error message to reflect this change

### 4. Background Job ✅

**File**: `app/Jobs/PublishScheduledPosts.php`

-   Finds all posts with:
    -   `status = 'scheduled'`
    -   `scheduled_at IS NOT NULL`
    -   `scheduled_at <= NOW()`
-   Publishes each post by:
    -   Setting `status` to `'posted'`
    -   Setting `published_at` to the `scheduled_at` value (or now if null)
    -   Clearing `scheduled_at`
-   Includes comprehensive logging for debugging
-   Includes error handling for individual post failures
-   TODO comment for future social media integration

### 5. Console Command ✅

**File**: `app/Console/Commands/PublishScheduledPostsCommand.php`

-   Command signature: `posts:publish-scheduled`
-   Dispatches the `PublishScheduledPosts` job
-   Provides console feedback
-   Returns proper exit codes (SUCCESS/FAILURE)

### 6. Laravel Scheduler ✅

**File**: `routes/console.php`

-   Scheduled command to run every minute
-   Uses `withoutOverlapping()` to prevent concurrent executions
-   Runs in background for better performance

## 🔌 API Endpoints

### Publish Post (with Scheduling)

```
PUT /api/posts/{post}/publish
```

**Request Body:**

```json
{
    "platforms": ["facebook", "twitter"], // Optional
    "scheduled_at": "2025-12-01T14:30:00Z" // Optional - if provided, schedules post
}
```

**Response (Scheduled):**

```json
{
    "success": true,
    "message": "Post scheduled successfully",
    "data": {
        "id": 123,
        "status": "scheduled",
        "scheduled_at": "2025-12-01T14:30:00Z",
        "published_at": null
        // ... other fields
    }
}
```

**Response (Immediate Publish):**

```json
{
    "success": true,
    "message": "Post published successfully",
    "data": {
        "id": 123,
        "status": "posted",
        "scheduled_at": null,
        "published_at": "2025-01-23T10:00:00Z"
        // ... other fields
    }
}
```

### Update Post (with Scheduled At)

```
PUT /api/posts/{post}
```

**Request Body:**

```json
{
    "content": "Updated content",
    "scheduled_at": "2025-12-01T15:00:00Z" // Optional - updates scheduled time
}
```

## 🚀 How It Works

1. **User Schedules Post**: Frontend sends `PUT /api/posts/{id}/publish` with `scheduled_at`
2. **Backend Stores**: Post status set to `'scheduled'`, `scheduled_at` stored
3. **Background Job**: Runs every minute via Laravel scheduler
4. **Job Checks**: Finds posts where `scheduled_at <= NOW()`
5. **Auto-Publish**: Changes status to `'posted'`, sets `published_at`, clears `scheduled_at`

## 📝 Important Notes

1. **Timezone**: Always store `scheduled_at` in UTC. Frontend should convert to user's timezone for display.

2. **Validation**:

    - `scheduled_at` must be in the future (validated with `after:now`)
    - Only `'draft'` and `'scheduled'` posts can be updated
    - Only `'draft'` posts can be published (via `assertDraft`)

3. **Scheduler Setup**:

    - For production, ensure Laravel scheduler is running:

    ```bash
    * * * * * cd /path-to-project && php artisan schedule:run >> /dev/null 2>&1
    ```

    - Or use a process manager like Supervisor

4. **Queue System**:

    - The job uses Laravel's queue system
    - Ensure queue worker is running: `php artisan queue:work`
    - Or use `dispatchSync()` if you want synchronous execution

5. **Future Enhancements**:
    - Social media publishing integration (marked with TODO in job)
    - Retry logic for failed publishes
    - Notifications when posts are published
    - Timezone handling improvements

## 🧪 Testing

### Test Immediate Publish

```bash
PUT /api/posts/1/publish
{
  "platforms": ["facebook"]
}
```

### Test Scheduled Publish

```bash
PUT /api/posts/1/publish
{
  "platforms": ["facebook"],
  "scheduled_at": "2025-12-01T14:30:00Z"
}
```

### Test Update Scheduled Post

```bash
PUT /api/posts/1
{
  "scheduled_at": "2025-12-01T15:00:00Z"
}
```

### Test Background Job

```bash
php artisan posts:publish-scheduled
```

## ✅ Checklist

-   [x] Database migration created
-   [x] Post model updated
-   [x] Publish endpoint handles scheduling
-   [x] Update endpoint handles scheduled_at
-   [x] Background job created
-   [x] Console command created
-   [x] Scheduler configured
-   [x] Validation implemented
-   [x] Error handling implemented
-   [x] Logging implemented

## 🔄 Next Steps (Frontend)

The frontend needs to:

1. Update `Post` interface to include `scheduled_at` and `'scheduled'` status
2. Update `publishPost()` function to accept `scheduled_at` parameter
3. Pass `scheduled_at` from `PostComposer` to `publishPost()`
4. Display scheduled posts in UI with scheduled time
5. Allow editing/canceling scheduled posts

See `SCHEDULE_POST_IMPLEMENTATION_GUIDE.md` for frontend implementation details.
