# Sanctum Token Setup Guide

## Problem
Error: `Table 'personal_access_tokens' doesn't exist`

This happens because the Sanctum migration hasn't been run yet.

## Solution

### Step 1: Migration File
✅ **Created**: `database/migrations/2019_12_14_000001_create_personal_access_tokens_table.php`

### Step 2: Run Migration
```bash
php artisan migrate
```

This will create the `personal_access_tokens` table.

---

## How Sanctum Tokens Work

### Token Creation Process

When you call `$user->createToken('web_app_token')`:

1. **Token Generation**:
   - Generates a random token string
   - Hashes the token using SHA-256
   - Creates a unique token ID

2. **Database Storage**:
   - Inserts record into `personal_access_tokens` table
   - Stores:
     - `tokenable_type`: Model class (e.g., `App\Models\AppUser`)
     - `tokenable_id`: User ID
     - `name`: Token name (e.g., 'web_app_token')
     - `token`: Hashed token (64 characters)
     - `abilities`: Permissions (usually `["*"]`)
     - `created_at`, `updated_at`

3. **Return Value**:
   - Returns plain text token: `{token_id}|{plain_token}`
   - Format: `1|abcdef1234567890...`
   - **Important**: This plain text token is only returned once!

### Token Format

```
{token_id}|{plain_text_token}
```

Example:
```
1|7eda8f53e2f0f3dcb2b52b0f196c4ed92ed21d815517ef0a7d707396cb7d0b17
```

- **Before `|`**: Token ID (used to lookup in database)
- **After `|`**: Plain text token (hashed before storage)

### Token Validation

When a request comes with `Authorization: Bearer {token}`:

1. Extract token ID (before `|`)
2. Look up token in `personal_access_tokens` table
3. Hash the provided plain text token
4. Compare with stored hash
5. If match, load the associated user model
6. Make user available via `$request->user()`

---

## Database Table Structure

### `personal_access_tokens` Table

```sql
CREATE TABLE personal_access_tokens (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    tokenable_type VARCHAR(255) NOT NULL,  -- Model class
    tokenable_id BIGINT UNSIGNED NOT NULL, -- User ID
    name VARCHAR(255) NOT NULL,            -- Token name
    token VARCHAR(64) UNIQUE NOT NULL,     -- Hashed token
    abilities TEXT NULL,                    -- JSON permissions
    last_used_at TIMESTAMP NULL,           -- Last usage
    expires_at TIMESTAMP NULL,             -- Expiration
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL,
    INDEX idx_tokenable (tokenable_type, tokenable_id)
);
```

### Example Record

```php
[
    'id' => 1,
    'tokenable_type' => 'App\Models\AppUser',
    'tokenable_id' => 1,
    'name' => 'web_app_token',
    'token' => '7eda8f53e2f0f3dcb2b52b0f196c4ed92ed21d815517ef0a7d707396cb7d0b17',
    'abilities' => '["*"]',
    'last_used_at' => '2025-11-07 12:45:00',
    'expires_at' => null,
    'created_at' => '2025-11-07 12:37:12',
    'updated_at' => '2025-11-07 12:45:00',
]
```

---

## Token Lifecycle

### 1. Create Token (Login)
```php
$token = $user->createToken('web_app_token')->plainTextToken;
// Returns: "1|7eda8f53e2f0f3dcb2b52b0f196c4ed92ed21d815517ef0a7d707396cb7d0b17"
```

### 2. Use Token (API Request)
```http
GET /api/user
Authorization: Bearer 1|7eda8f53e2f0f3dcb2b52b0f196c4ed92ed21d815517ef0a7d707396cb7d0b17
```

### 3. Revoke Token (Logout)
```php
$request->user()->currentAccessToken()->delete();
// Deletes the token from database
```

---

## Security Features

1. **Hashing**: Tokens are hashed before storage (one-way encryption)
2. **Unique**: Each token is unique and cannot be reverse-engineered
3. **Revocable**: Tokens can be deleted to invalidate them
4. **Expirable**: Optional expiration dates can be set
5. **Scoped**: Tokens can have specific abilities/permissions

---

## Troubleshooting

### Error: Table doesn't exist
**Solution**: Run `php artisan migrate`

### Error: Token not working
**Check**:
1. Token is in correct format: `{id}|{token}`
2. Token exists in database
3. Token hasn't been revoked
4. Token hasn't expired (if expiration set)

### Multiple Tokens Per User
- Users can have multiple tokens
- Each token is independent
- Useful for multiple devices/sessions

---

## Best Practices

1. **Token Naming**: Use descriptive names
   - `'web_app_token'` - Web application
   - `'mobile_app_token'` - Mobile app
   - `'api_token'` - API access

2. **Token Expiration**: Set expiration for security
   ```php
   $token = $user->createToken('web_app_token', ['*'], now()->addDays(30));
   ```

3. **Token Cleanup**: Periodically clean expired tokens
   ```php
   PersonalAccessToken::where('expires_at', '<', now())->delete();
   ```

4. **Revoke All Tokens**: On password change
   ```php
   $user->tokens()->delete();
   ```

---

## Summary

✅ **Migration Created**: `2019_12_14_000001_create_personal_access_tokens_table.php`
✅ **Next Step**: Run `php artisan migrate`
✅ **How It Works**: Tokens stored in `personal_access_tokens` table
✅ **Security**: Tokens are hashed before storage
✅ **Usage**: Send token in `Authorization: Bearer {token}` header

