# Remember Me & Forgot Password Implementation Guide

## Overview

This guide explains how "Remember Me" and "Forgot Password" features work in the API.

---

## 1. Remember Me Feature

### How It Works

**Remember Me** allows users to stay logged in for an extended period without re-entering credentials.

### Backend Implementation

#### Token Expiration Strategy

-   **With "Remember Me" checked**: Token expires in **30 days**
-   **Without "Remember Me"**: Token expires in **7 days**

#### Login Request Format

```http
POST /api/login
Content-Type: application/json

{
    "email": "user@example.com",
    "password": "password123",
    "remember_me": true  // or false
}
```

#### Login Response

```json
{
    "success": true,
    "message": "Login successful.",
    "data": {
        "token": "1|abcdef1234567890...",
        "token_expires_at": "2025-02-20T10:30:00+00:00",
        "remember_me": true,
        "user": {
            "id": 1,
            "name": "John Doe",
            "email": "user@example.com",
            "email_verified": true,
            "timezone": "UTC",
            "status": "active"
        }
    }
}
```

### Frontend Implementation (Next.js)

#### Storage Strategy

**Recommended: Store token in localStorage or sessionStorage**

```javascript
// After successful login
const response = await fetch("/api/login", {
    method: "POST",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify({
        email: email,
        password: password,
        remember_me: rememberMe, // from checkbox
    }),
});

const data = await response.json();

if (data.success) {
    // Store token
    if (rememberMe) {
        // Store in localStorage (persists across browser sessions)
        localStorage.setItem("auth_token", data.data.token);
        localStorage.setItem("token_expires_at", data.data.token_expires_at);
    } else {
        // Store in sessionStorage (cleared when browser closes)
        sessionStorage.setItem("auth_token", data.data.token);
        sessionStorage.setItem("token_expires_at", data.data.token_expires_at);
    }

    // Store user data
    localStorage.setItem("user", JSON.stringify(data.data.user));
}
```

#### Using Token in API Requests

```javascript
// Get token from storage
const token =
    localStorage.getItem("auth_token") || sessionStorage.getItem("auth_token");

// Include in API requests
fetch("/api/user", {
    headers: {
        Authorization: `Bearer ${token}`,
        "Content-Type": "application/json",
    },
});
```

#### Token Expiration Check

```javascript
// Check if token is expired before making requests
function isTokenExpired() {
    const expiresAt =
        localStorage.getItem("token_expires_at") ||
        sessionStorage.getItem("token_expires_at");

    if (!expiresAt) return true;

    return new Date(expiresAt) < new Date();
}

// Auto-logout if expired
if (isTokenExpired()) {
    localStorage.removeItem("auth_token");
    sessionStorage.removeItem("auth_token");
    // Redirect to login
    window.location.href = "/login";
}
```

### Security Considerations

1. **Token Storage**:

    - ✅ **localStorage**: Persists across sessions (good for "Remember Me")
    - ✅ **sessionStorage**: Cleared on browser close (good for normal login)
    - ❌ **Cookies**: Can be more secure with HttpOnly flag, but requires backend changes

2. **Token Expiration**: Tokens automatically expire based on "Remember Me" selection

3. **Logout**: Always delete tokens from storage on logout

---

## 2. Forgot Password Feature

### How It Works

Users can reset their password if they've forgotten it by:

1. Requesting a password reset link via email
2. Clicking the link (contains a secure token)
3. Setting a new password

### Backend Implementation

#### Step 1: Request Password Reset

**Endpoint**: `POST /api/forgot-password`

**Request Format**:

```http
POST /api/forgot-password
Content-Type: application/json

{
    "email": "user@example.com"
}
```

**Response** (Always returns success for security):

```json
{
    "success": true,
    "message": "If an account exists with that email, we have sent a password reset link.",
    "debug": {
        "token": "abc123...",
        "reset_url": "http://your-app.com/reset-password?token=abc123...&email=user@example.com"
    }
}
```

**Note**: The `debug` field is only included in development. Remove it in production!

#### Step 2: Reset Password

**Endpoint**: `POST /api/reset-password`

**Request Format**:

```http
POST /api/reset-password
Content-Type: application/json

{
    "email": "user@example.com",
    "token": "abc123...",
    "password": "newpassword123",
    "password_confirmation": "newpassword123"
}
```

**Success Response**:

```json
{
    "success": true,
    "message": "Your password has been reset successfully. You can now login with your new password."
}
```

**Error Response** (Invalid/Expired Token):

```json
{
    "success": false,
    "message": "Invalid or expired reset token.",
    "errors": {
        "token": ["This password reset token is invalid or has expired."]
    }
}
```

### Frontend Implementation (Next.js)

#### Step 1: Forgot Password Page

```javascript
// pages/forgot-password.js
const ForgotPassword = () => {
    const [email, setEmail] = useState("");
    const [loading, setLoading] = useState(false);
    const [message, setMessage] = useState("");

    const handleSubmit = async (e) => {
        e.preventDefault();
        setLoading(true);
        setMessage("");

        try {
            const response = await fetch("/api/forgot-password", {
                method: "POST",
                headers: { "Content-Type": "application/json" },
                body: JSON.stringify({ email }),
            });

            const data = await response.json();

            if (data.success) {
                setMessage(data.message);
                // In development, you might show the reset URL from debug
                if (data.debug) {
                    console.log("Reset URL:", data.debug.reset_url);
                }
            }
        } catch (error) {
            setMessage("An error occurred. Please try again.");
        } finally {
            setLoading(false);
        }
    };

    return (
        <form onSubmit={handleSubmit}>
            <input
                type="email"
                value={email}
                onChange={(e) => setEmail(e.target.value)}
                placeholder="Enter your email"
                required
            />
            <button type="submit" disabled={loading}>
                {loading ? "Sending..." : "Send Reset Link"}
            </button>
            {message && <p>{message}</p>}
        </form>
    );
};
```

#### Step 2: Reset Password Page

```javascript
// pages/reset-password.js
const ResetPassword = () => {
    const router = useRouter();
    const { token, email } = router.query;

    const [password, setPassword] = useState("");
    const [passwordConfirmation, setPasswordConfirmation] = useState("");
    const [loading, setLoading] = useState(false);
    const [message, setMessage] = useState("");
    const [errors, setErrors] = useState({});

    const handleSubmit = async (e) => {
        e.preventDefault();
        setLoading(true);
        setMessage("");
        setErrors({});

        // Frontend validation
        if (password !== passwordConfirmation) {
            setErrors({ password: ["Passwords do not match"] });
            setLoading(false);
            return;
        }

        try {
            const response = await fetch("/api/reset-password", {
                method: "POST",
                headers: { "Content-Type": "application/json" },
                body: JSON.stringify({
                    email,
                    token,
                    password,
                    password_confirmation: passwordConfirmation,
                }),
            });

            const data = await response.json();

            if (data.success) {
                setMessage(data.message);
                // Redirect to login after 2 seconds
                setTimeout(() => {
                    router.push("/login");
                }, 2000);
            } else {
                setErrors(data.errors || {});
                setMessage(data.message);
            }
        } catch (error) {
            setMessage("An error occurred. Please try again.");
        } finally {
            setLoading(false);
        }
    };

    return (
        <form onSubmit={handleSubmit}>
            <input
                type="password"
                value={password}
                onChange={(e) => setPassword(e.target.value)}
                placeholder="New Password"
                required
            />
            <input
                type="password"
                value={passwordConfirmation}
                onChange={(e) => setPasswordConfirmation(e.target.value)}
                placeholder="Confirm Password"
                required
            />
            {errors.password && <p>{errors.password[0]}</p>}
            <button type="submit" disabled={loading}>
                {loading ? "Resetting..." : "Reset Password"}
            </button>
            {message && <p>{message}</p>}
        </form>
    );
};
```

### Email Integration (Production)

**Important**: Currently, the API returns the reset token in the response for development. In production, you must:

1. **Configure Email** in `.env`:

```env
MAIL_MAILER=smtp
MAIL_HOST=smtp.gmail.com
MAIL_PORT=587
MAIL_USERNAME=your-email@gmail.com
MAIL_PASSWORD=your-app-password
MAIL_ENCRYPTION=tls
MAIL_FROM_ADDRESS=noreply@yourdomain.com
MAIL_FROM_NAME="${APP_NAME}"
```

2. **Create Email Template**:

```php
// app/Mail/ResetPasswordMail.php
namespace App\Mail;

use Illuminate\Bus\Queueable;
use Illuminate\Mail\Mailable;
use Illuminate\Queue\SerializesModels;

class ResetPasswordMail extends Mailable
{
    use Queueable, SerializesModels;

    public $resetUrl;

    public function __construct($resetUrl)
    {
        $this->resetUrl = $resetUrl;
    }

    public function build()
    {
        return $this->subject('Reset Your Password')
                    ->view('emails.reset-password')
                    ->with(['resetUrl' => $this->resetUrl]);
    }
}
```

3. **Update AuthController** (remove debug, add email):

```php
// In forgotPassword method, replace debug section with:
$resetUrl = config('app.frontend_url') . '/reset-password?token=' . $token . '&email=' . urlencode($email);
Mail::to($user->email)->send(new ResetPasswordMail($resetUrl));
```

### Security Features

1. **Token Expiration**: Reset tokens expire after 60 minutes
2. **One-Time Use**: Tokens are deleted after successful password reset
3. **Hashed Storage**: Tokens are hashed before storing in database
4. **Email Privacy**: API doesn't reveal if email exists (security best practice)

---

## Testing in Postman

### Remember Me - Login

```http
POST http://social-platform/api/login
Content-Type: application/json

{
    "email": "user@example.com",
    "password": "password123",
    "remember_me": true
}
```

### Forgot Password

```http
POST http://social-platform/api/forgot-password
Content-Type: application/json

{
    "email": "user@example.com"
}
```

### Reset Password

```http
POST http://social-platform/api/reset-password
Content-Type: application/json

{
    "email": "user@example.com",
    "token": "token_from_forgot_password_response",
    "password": "newpassword123",
    "password_confirmation": "newpassword123"
}
```

---

## Summary

### Remember Me

-   ✅ Token expiration: 30 days (with remember_me) or 7 days (without)
-   ✅ Frontend stores token in localStorage (remember_me) or sessionStorage (normal)
-   ✅ Token automatically expires based on expiration date

### Forgot Password

-   ✅ Secure token generation (64 characters, hashed)
-   ✅ Token expiration (60 minutes)
-   ✅ One-time use tokens
-   ✅ Email integration ready (configure in production)

Both features are fully implemented and ready to use!
