# Post and Draft Storage System - Summary

## 📋 Overview

This document provides a quick overview of the post and draft storage system. For detailed documentation, see the files listed below.

## 🎯 System Requirements

1. ✅ **Unified Table**: Single `posts` table for both drafts and published posts
2. ✅ **Status Field**: `status` field (`'draft'` or `'posted'`) differentiates posts
3. ✅ **Separate Media Storage**: `post_media` table for multiple images per post
4. ✅ **Edit Restrictions**: Only draft posts can be edited
5. ✅ **List View**: Show all posts (drafts + published) with edit option only for drafts
6. ✅ **No Social Media Integration**: Focus on storage and management flow first

## 📁 Documentation Files

### 1. **POST_DRAFT_API_DOCUMENTATION.md**

**Complete API Documentation**

- Database schema (SQL)
- All Laravel API endpoints with request/response examples
- Frontend integration guide (Next.js API routes)
- Error handling
- Implementation checklist

### 2. **LARAVEL_API_ENDPOINTS_REFERENCE.md**

**Quick Reference Guide**

- All API endpoints at a glance
- Request/response formats
- Status rules table
- Common use cases
- Laravel code examples

### 3. **POST_DRAFT_WORKFLOW_DIAGRAM.md**

**Visual Workflow Guide**

- ASCII flow diagrams
- Status state machine
- Data flow diagrams
- Component structure
- Decision points

### 4. **POST_AND_DRAFT_STORAGE.md**

**Original Analysis**

- Current implementation status
- What's missing
- Recommended implementation (original)

## 🗄️ Database Schema Summary

### Posts Table

```sql
CREATE TABLE posts (
    id BIGINT UNSIGNED PRIMARY KEY,
    user_id BIGINT UNSIGNED NOT NULL,
    content TEXT NOT NULL,
    content_html TEXT NULL,
    status ENUM('draft', 'posted') DEFAULT 'draft',
    category VARCHAR(100) NULL,
    platforms JSON NULL,
    published_at DATETIME NULL,
    created_at TIMESTAMP,
    updated_at TIMESTAMP
);
```

### Post Media Table

```sql
CREATE TABLE post_media (
    id BIGINT UNSIGNED PRIMARY KEY,
    post_id BIGINT UNSIGNED NOT NULL,
    user_id BIGINT UNSIGNED NOT NULL,
    file_name VARCHAR(255) NOT NULL,
    file_path VARCHAR(500) NOT NULL,
    file_url VARCHAR(500) NULL,
    file_type VARCHAR(50) NULL,
    file_size BIGINT NULL,
    display_order INT DEFAULT 0,
    created_at TIMESTAMP
);
```

## 🔌 API Endpoints Summary

### Posts

- `POST /api/posts` - Create draft post
- `GET /api/posts` - List all posts (with filters)
- `GET /api/posts/{id}` - Get single post
- `PUT /api/posts/{id}` - Update draft post (draft only)
- `PUT /api/posts/{id}/publish` - Publish draft
- `DELETE /api/posts/{id}` - Delete post

### Media

- `POST /api/posts/{id}/media` - Upload media to post
- `DELETE /api/posts/{id}/media/{media_id}` - Delete media

## 🔄 Key Workflows

### 1. Create Draft

```
User → Compose → Save as Draft → POST /api/posts → Laravel saves (status='draft')
```

### 2. Publish Draft

```
User → Posts List → Click "Publish" → PUT /api/posts/{id}/publish → Status changes to 'posted'
```

### 3. Edit Draft

```
User → Posts List → Click "Edit" (draft only) → Load into composer → Update → Save
```

### 4. View List

```
User → Posts List → GET /api/posts → Display all posts → Show "Edit" only for drafts
```

## 🔒 Status Rules

| Action    | Draft | Posted |
| --------- | ----- | ------ |
| View      | ✅    | ✅     |
| Edit      | ✅    | ❌     |
| Delete    | ✅    | ✅     |
| Publish   | ✅    | ❌     |
| Add Media | ✅    | ❌     |

## 🚀 Implementation Steps

1. **Database Setup** (Laravel)

   - Create migrations for `posts` and `post_media` tables
   - Run migrations

2. **Laravel Backend**

   - Create `Post` and `PostMedia` models
   - Create `PostController` and `PostMediaController`
   - Define API routes
   - Implement file storage

3. **Next.js API Routes**

   - Create proxy routes in `app/api/posts/`
   - Handle authentication
   - Forward requests to Laravel

4. **Frontend Pages**

   - Update `app/compose/page.tsx` (add save draft)
   - Create `app/posts/page.tsx` (post list)
   - Add edit functionality

5. **Testing**
   - Test all CRUD operations
   - Test status restrictions
   - Test media upload/delete

## 📝 Quick Start

### Laravel Routes

```php
Route::middleware('auth:sanctum')->group(function () {
    Route::apiResource('posts', PostController::class);
    Route::put('posts/{id}/publish', [PostController::class, 'publish']);
    Route::post('posts/{id}/media', [PostMediaController::class, 'store']);
    Route::delete('posts/{id}/media/{media_id}', [PostMediaController::class, 'destroy']);
});
```

### Next.js API Route Example

```typescript
// app/api/posts/route.ts
export async function POST(request: NextRequest) {
  const session = await getServerSession(authOptions);
  const formData = await request.formData();
  formData.append("user_id", session.user.id.toString());
  formData.append("status", "draft");

  const response = await serverApi.request("/api/posts", {
    method: "POST",
    body: formData,
  });

  return NextResponse.json(response);
}
```

## 🎨 Frontend Features

### Post List Page (`/posts`)

- Display all posts (drafts + published)
- Filter by status
- "Edit" button (draft posts only)
- "Publish" button (draft posts only)
- "Delete" button (all posts)
- Show media thumbnails
- Pagination

### Compose Page (`/compose`)

- Create new draft
- Edit existing draft (if `?edit={id}`)
- Save as draft
- Publish post
- Upload multiple images
- Preview post

## ⚠️ Important Notes

1. **Authentication**: All endpoints require Bearer token
2. **Authorization**: Users can only access their own posts
3. **Status Validation**: Backend must check status before allowing edits
4. **File Storage**: Configure Laravel storage (local or S3)
5. **Media Cleanup**: Delete files when post/media is deleted
6. **No Social Media**: Social media integration excluded for now

## 📚 Next Steps

1. Read `POST_DRAFT_API_DOCUMENTATION.md` for complete specs
2. Review `POST_DRAFT_WORKFLOW_DIAGRAM.md` for visual understanding
3. Use `LARAVEL_API_ENDPOINTS_REFERENCE.md` as quick reference
4. Implement Laravel backend APIs
5. Create Next.js API proxy routes
6. Build frontend pages
7. Test complete flow

## 🔗 Related Files

- `POST_DRAFT_API_DOCUMENTATION.md` - Full API documentation
- `LARAVEL_API_ENDPOINTS_REFERENCE.md` - API quick reference
- `POST_DRAFT_WORKFLOW_DIAGRAM.md` - Visual workflows
- `POST_AND_DRAFT_STORAGE.md` - Original analysis

---

**For questions or clarifications, refer to the detailed documentation files listed above.**
