# Post and Draft Storage System - Complete Documentation

## 📋 Table of Contents

1. [Overview](#overview)
2. [Database Schema](#database-schema)
3. [System Workflow](#system-workflow)
4. [Laravel API Endpoints](#laravel-api-endpoints)
5. [Frontend Integration](#frontend-integration)
6. [API Request/Response Examples](#api-requestresponse-examples)
7. [Error Handling](#error-handling)
8. [Implementation Checklist](#implementation-checklist)

---

## Overview

This system allows users to:

- Create and save posts as **drafts** (work in progress)
- **Publish** posts (change status from draft to posted)
- View a **unified list** of all posts (both drafts and published)
- **Edit only draft posts** (published posts are read-only)
- Attach **multiple media files** (images/videos) to each post
- Store media files separately with proper relationships

### Key Design Decisions

1. **Single Table Approach**: Both drafts and published posts are stored in the same `posts` table, differentiated by a `status` field (`draft` or `posted`)
2. **Separate Media Storage**: Media files are stored in a separate `post_media` table with a foreign key relationship
3. **Status-Based Editing**: Only posts with `status = 'draft'` can be edited
4. **No Social Media Integration Yet**: Focus on storage and management flow first

---

## Database Schema

### 1. Posts Table

Stores both draft and published posts in a single table.

```sql
CREATE TABLE posts (
    id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
    user_id BIGINT UNSIGNED NOT NULL,
    content TEXT NOT NULL,
    content_html TEXT NULL,
    status ENUM('draft', 'posted') NOT NULL DEFAULT 'draft',
    category VARCHAR(100) NULL DEFAULT 'General',
    platforms JSON NULL COMMENT 'Selected platforms for publishing: ["facebook", "twitter"]',
    published_at DATETIME NULL COMMENT 'Timestamp when post was published (NULL for drafts)',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_status (status),
    INDEX idx_user_status (user_id, status),
    INDEX idx_published_at (published_at),
    INDEX idx_created_at (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
```

**Field Descriptions:**

- `id`: Unique post identifier
- `user_id`: Owner of the post
- `content`: Plain text content of the post
- `content_html`: HTML formatted content (for rich text)
- `status`: Either `'draft'` or `'posted'`
- `category`: Post category (e.g., "General", "Marketing", "News")
- `platforms`: JSON array of selected platforms (stored for reference, not used for publishing yet)
- `published_at`: Timestamp when status changed to 'posted' (NULL for drafts)
- `created_at`: When the post was first created
- `updated_at`: Last modification timestamp

### 2. Post Media Table

Stores media files separately, allowing multiple files per post.

```sql
CREATE TABLE post_media (
    id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
    post_id BIGINT UNSIGNED NOT NULL,
    user_id BIGINT UNSIGNED NOT NULL,
    file_name VARCHAR(255) NOT NULL,
    file_path VARCHAR(500) NOT NULL COMMENT 'Storage path relative to storage root',
    file_url VARCHAR(500) NULL COMMENT 'Public URL for accessing the file',
    file_type VARCHAR(50) NULL COMMENT 'MIME type: image/jpeg, video/mp4, etc.',
    file_size BIGINT UNSIGNED NULL COMMENT 'File size in bytes',
    mime_type VARCHAR(100) NULL,
    display_order INT UNSIGNED DEFAULT 0 COMMENT 'Order for displaying multiple images',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

    FOREIGN KEY (post_id) REFERENCES posts(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_post_id (post_id),
    INDEX idx_user_id (user_id),
    INDEX idx_display_order (post_id, display_order)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
```

**Field Descriptions:**

- `id`: Unique media identifier
- `post_id`: Associated post
- `user_id`: Owner (for security/access control)
- `file_name`: Original filename
- `file_path`: Storage path (e.g., `posts/123/image_abc123.jpg`)
- `file_url`: Public URL (e.g., `https://domain.com/storage/posts/123/image_abc123.jpg`)
- `file_type`: MIME type for proper handling
- `file_size`: Size in bytes
- `display_order`: Order for displaying multiple images (0, 1, 2, ...)

---

## System Workflow

### 1. Creating a Draft Post

```
User composes post in frontend
        ↓
User clicks "Save as Draft"
        ↓
Frontend: POST /api/drafts
        ↓
Next.js API Route: app/api/drafts/route.ts
        ↓
Laravel Backend: POST /api/posts (with status='draft')
        ↓
Laravel saves post with status='draft'
        ↓
If media files uploaded:
  - Upload files to storage
  - Create post_media records
        ↓
Return post_id and success response
        ↓
Frontend shows "Draft saved successfully"
```

### 2. Publishing a Post

```
User has draft post
        ↓
User clicks "Publish Post"
        ↓
Frontend: PUT /api/posts/{id}/publish
        ↓
Next.js API Route: app/api/posts/[id]/publish/route.ts
        ↓
Laravel Backend: PUT /api/posts/{id}/publish
        ↓
Laravel updates post:
  - status = 'posted'
  - published_at = NOW()
        ↓
(Optional: Later integrate social media publishing here)
        ↓
Return success response
        ↓
Frontend shows "Post published successfully"
```

### 3. Editing a Draft Post

```
User views post list
        ↓
User clicks "Edit" on a draft post
        ↓
Frontend: GET /api/posts/{id}
        ↓
Laravel Backend: GET /api/posts/{id}
        ↓
Laravel checks: status must be 'draft'
        ↓
Return post data with media
        ↓
Frontend loads post into composer
        ↓
User edits content
        ↓
User clicks "Save as Draft" or "Publish"
        ↓
Update flow (same as create)
```

### 4. Viewing Post List

```
User navigates to Posts page
        ↓
Frontend: GET /api/posts?user_id={id}
        ↓
Laravel Backend: GET /api/posts
        ↓
Laravel returns all posts (drafts + published) for user
        ↓
Frontend displays list:
  - Draft posts: Show "Edit" button
  - Published posts: No edit button (read-only)
        ↓
User can filter by status if needed
```

---

## Laravel API Endpoints

### Base URL

```
http://social-platform/api
```

All endpoints require authentication (Bearer token in Authorization header).

---

### 1. Create Post (Draft)

**Endpoint:** `POST /api/posts`

**Description:** Creates a new post with status 'draft'. Can include media files.

**Request Headers:**

```
Authorization: Bearer {token}
Content-Type: multipart/form-data (if media) or application/json
```

**Request Body (JSON):**

```json
{
  "user_id": 1,
  "content": "This is my post content",
  "content_html": "<p>This is my <strong>post</strong> content</p>",
  "status": "draft",
  "category": "General",
  "platforms": ["facebook", "twitter"]
}
```

**Request Body (Form Data - with media):**

```
user_id: 1
content: "This is my post content"
content_html: "<p>This is my post content</p>"
status: draft
category: General
platforms: ["facebook", "twitter"]
files[]: [File 1]
files[]: [File 2]
```

**Response (Success - 201):**

```json
{
  "success": true,
  "message": "Post created successfully",
  "data": {
    "id": 123,
    "user_id": 1,
    "content": "This is my post content",
    "content_html": "<p>This is my post content</p>",
    "status": "draft",
    "category": "General",
    "platforms": ["facebook", "twitter"],
    "published_at": null,
    "created_at": "2024-01-15T10:30:00.000000Z",
    "updated_at": "2024-01-15T10:30:00.000000Z",
    "media": [
      {
        "id": 1,
        "post_id": 123,
        "file_name": "image1.jpg",
        "file_url": "http://social-platform/storage/posts/123/image1.jpg",
        "file_type": "image/jpeg",
        "file_size": 245678,
        "display_order": 0
      },
      {
        "id": 2,
        "post_id": 123,
        "file_name": "image2.png",
        "file_url": "http://social-platform/storage/posts/123/image2.png",
        "file_type": "image/png",
        "file_size": 456789,
        "display_order": 1
      }
    ]
  }
}
```

**Response (Error - 400):**

```json
{
  "success": false,
  "error": "Validation failed",
  "errors": {
    "content": ["The content field is required."],
    "user_id": ["The user id field is required."]
  }
}
```

---

### 2. Get All Posts (List)

**Endpoint:** `GET /api/posts`

**Description:** Retrieves all posts for the authenticated user. Can filter by status.

**Query Parameters:**

- `status` (optional): Filter by status (`draft` or `posted`)
- `category` (optional): Filter by category
- `page` (optional): Page number for pagination (default: 1)
- `per_page` (optional): Items per page (default: 20, max: 100)
- `sort` (optional): Sort field (`created_at`, `updated_at`, `published_at`) (default: `created_at`)
- `order` (optional): Sort order (`asc` or `desc`) (default: `desc`)

**Request Headers:**

```
Authorization: Bearer {token}
```

**Example Request:**

```
GET /api/posts?status=draft&page=1&per_page=20&sort=updated_at&order=desc
```

**Response (Success - 200):**

```json
{
  "success": true,
  "data": [
    {
      "id": 123,
      "user_id": 1,
      "content": "This is a draft post",
      "content_html": "<p>This is a draft post</p>",
      "status": "draft",
      "category": "General",
      "platforms": ["facebook"],
      "published_at": null,
      "created_at": "2024-01-15T10:30:00.000000Z",
      "updated_at": "2024-01-15T11:45:00.000000Z",
      "media": [
        {
          "id": 1,
          "file_name": "image1.jpg",
          "file_url": "http://social-platform/storage/posts/123/image1.jpg",
          "file_type": "image/jpeg",
          "file_size": 245678,
          "display_order": 0
        }
      ],
      "media_count": 1
    },
    {
      "id": 122,
      "user_id": 1,
      "content": "This is a published post",
      "content_html": "<p>This is a published post</p>",
      "status": "posted",
      "category": "Marketing",
      "platforms": ["facebook", "twitter"],
      "published_at": "2024-01-14T15:20:00.000000Z",
      "created_at": "2024-01-14T14:00:00.000000Z",
      "updated_at": "2024-01-14T15:20:00.000000Z",
      "media": [],
      "media_count": 0
    }
  ],
  "meta": {
    "current_page": 1,
    "per_page": 20,
    "total": 45,
    "last_page": 3,
    "from": 1,
    "to": 20
  }
}
```

---

### 3. Get Single Post

**Endpoint:** `GET /api/posts/{id}`

**Description:** Retrieves a single post by ID with all associated media.

**Request Headers:**

```
Authorization: Bearer {token}
```

**Response (Success - 200):**

```json
{
  "success": true,
  "data": {
    "id": 123,
    "user_id": 1,
    "content": "This is my post content",
    "content_html": "<p>This is my post content</p>",
    "status": "draft",
    "category": "General",
    "platforms": ["facebook", "twitter"],
    "published_at": null,
    "created_at": "2024-01-15T10:30:00.000000Z",
    "updated_at": "2024-01-15T10:30:00.000000Z",
    "media": [
      {
        "id": 1,
        "post_id": 123,
        "file_name": "image1.jpg",
        "file_path": "posts/123/image1.jpg",
        "file_url": "http://social-platform/storage/posts/123/image1.jpg",
        "file_type": "image/jpeg",
        "file_size": 245678,
        "display_order": 0,
        "created_at": "2024-01-15T10:30:00.000000Z"
      }
    ]
  }
}
```

**Response (Error - 404):**

```json
{
  "success": false,
  "error": "Post not found"
}
```

**Response (Error - 403):**

```json
{
  "success": false,
  "error": "You don't have permission to access this post"
}
```

---

### 4. Update Post (Draft Only)

**Endpoint:** `PUT /api/posts/{id}`

**Description:** Updates a post. **Only allowed if status is 'draft'**.

**Request Headers:**

```
Authorization: Bearer {token}
Content-Type: application/json (or multipart/form-data if updating media)
```

**Request Body (JSON):**

```json
{
  "content": "Updated post content",
  "content_html": "<p>Updated post content</p>",
  "category": "Marketing",
  "platforms": ["facebook", "twitter", "linkedin"]
}
```

**Request Body (Form Data - with new media):**

```
content: "Updated post content"
content_html: "<p>Updated post content</p>"
category: Marketing
platforms: ["facebook", "twitter"]
files[]: [New File 1]
files[]: [New File 2]
remove_media_ids: [1, 2]  // IDs of media to remove
```

**Response (Success - 200):**

```json
{
  "success": true,
  "message": "Post updated successfully",
  "data": {
    "id": 123,
    "user_id": 1,
    "content": "Updated post content",
    "content_html": "<p>Updated post content</p>",
    "status": "draft",
    "category": "Marketing",
    "platforms": ["facebook", "twitter", "linkedin"],
    "published_at": null,
    "created_at": "2024-01-15T10:30:00.000000Z",
    "updated_at": "2024-01-15T12:00:00.000000Z",
    "media": [...]
  }
}
```

**Response (Error - 400):**

```json
{
  "success": false,
  "error": "Cannot update post with status 'posted'. Only draft posts can be updated."
}
```

**Response (Error - 404):**

```json
{
  "success": false,
  "error": "Post not found"
}
```

---

### 5. Publish Post (Change Status to Posted)

**Endpoint:** `PUT /api/posts/{id}/publish`

**Description:** Changes post status from 'draft' to 'posted' and sets published_at timestamp.

**Request Headers:**

```
Authorization: Bearer {token}
```

**Request Body (optional):**

```json
{
  "platforms": ["facebook", "twitter"] // Optional: update platforms before publishing
}
```

**Response (Success - 200):**

```json
{
  "success": true,
  "message": "Post published successfully",
  "data": {
    "id": 123,
    "user_id": 1,
    "content": "This is my post content",
    "content_html": "<p>This is my post content</p>",
    "status": "posted",
    "category": "General",
    "platforms": ["facebook", "twitter"],
    "published_at": "2024-01-15T12:30:00.000000Z",
    "created_at": "2024-01-15T10:30:00.000000Z",
    "updated_at": "2024-01-15T12:30:00.000000Z",
    "media": [...]
  }
}
```

**Response (Error - 400):**

```json
{
  "success": false,
  "error": "Post is already published"
}
```

---

### 6. Delete Post

**Endpoint:** `DELETE /api/posts/{id}`

**Description:** Deletes a post and all associated media files. Works for both drafts and published posts.

**Request Headers:**

```
Authorization: Bearer {token}
```

**Response (Success - 200):**

```json
{
  "success": true,
  "message": "Post deleted successfully"
}
```

**Response (Error - 404):**

```json
{
  "success": false,
  "error": "Post not found"
}
```

---

### 7. Upload Media to Post

**Endpoint:** `POST /api/posts/{id}/media`

**Description:** Adds media files to an existing post. Only allowed for draft posts.

**Request Headers:**

```
Authorization: Bearer {token}
Content-Type: multipart/form-data
```

**Request Body (Form Data):**

```
files[]: [File 1]
files[]: [File 2]
display_order: 0  // Optional: starting order number
```

**Response (Success - 201):**

```json
{
  "success": true,
  "message": "Media uploaded successfully",
  "data": [
    {
      "id": 3,
      "post_id": 123,
      "file_name": "new_image.jpg",
      "file_url": "http://social-platform/storage/posts/123/new_image.jpg",
      "file_type": "image/jpeg",
      "file_size": 345678,
      "display_order": 0
    }
  ]
}
```

---

### 8. Delete Media from Post

**Endpoint:** `DELETE /api/posts/{id}/media/{media_id}`

**Description:** Removes a media file from a post. Only allowed for draft posts.

**Request Headers:**

```
Authorization: Bearer {token}
```

**Response (Success - 200):**

```json
{
  "success": true,
  "message": "Media deleted successfully"
}
```

---

## Frontend Integration

### Next.js API Routes (Proxy to Laravel)

The frontend will use Next.js API routes that proxy requests to Laravel backend.

#### 1. Create Draft: `app/api/posts/route.ts`

```typescript
import { NextRequest, NextResponse } from "next/server";
import { getServerSession } from "next-auth/next";
import { authOptions } from "@/lib/authOptions";
import { serverApi } from "@/lib/apiService";

export async function POST(request: NextRequest) {
  try {
    const session = await getServerSession(authOptions);

    if (!session || !session.user) {
      return NextResponse.json({ error: "Not authenticated" }, { status: 401 });
    }

    const formData = await request.formData();

    // Add user_id to form data
    formData.append("user_id", session.user.id.toString());
    formData.append("status", "draft");

    // Forward to Laravel backend
    const response = await serverApi.request("/api/posts", {
      method: "POST",
      body: formData,
      headers: {
        Authorization: `Bearer ${session.accessToken}`,
      },
    });

    if (response.error) {
      return NextResponse.json(
        { error: response.error },
        { status: response.status }
      );
    }

    return NextResponse.json({
      success: true,
      data: response.data,
    });
  } catch (error) {
    return NextResponse.json(
      { error: "Failed to create post" },
      { status: 500 }
    );
  }
}
```

#### 2. Get Posts List: `app/api/posts/route.ts` (GET)

```typescript
export async function GET(request: NextRequest) {
  try {
    const session = await getServerSession(authOptions);

    if (!session || !session.user) {
      return NextResponse.json({ error: "Not authenticated" }, { status: 401 });
    }

    const { searchParams } = new URL(request.url);
    const params: Record<string, string> = {};

    // Forward query parameters
    searchParams.forEach((value, key) => {
      params[key] = value;
    });

    const response = await serverApi.request("/api/posts", {
      method: "GET",
      params,
      headers: {
        Authorization: `Bearer ${session.accessToken}`,
      },
    });

    if (response.error) {
      return NextResponse.json(
        { error: response.error },
        { status: response.status }
      );
    }

    return NextResponse.json({
      success: true,
      data: response.data,
      meta: response.meta,
    });
  } catch (error) {
    return NextResponse.json(
      { error: "Failed to fetch posts" },
      { status: 500 }
    );
  }
}
```

#### 3. Update Post: `app/api/posts/[id]/route.ts` (PUT)

```typescript
export async function PUT(
  request: NextRequest,
  { params }: { params: { id: string } }
) {
  try {
    const session = await getServerSession(authOptions);

    if (!session || !session.user) {
      return NextResponse.json({ error: "Not authenticated" }, { status: 401 });
    }

    const formData = await request.formData();

    const response = await serverApi.request(`/api/posts/${params.id}`, {
      method: "PUT",
      body: formData,
      headers: {
        Authorization: `Bearer ${session.accessToken}`,
      },
    });

    if (response.error) {
      return NextResponse.json(
        { error: response.error },
        { status: response.status }
      );
    }

    return NextResponse.json({
      success: true,
      data: response.data,
    });
  } catch (error) {
    return NextResponse.json(
      { error: "Failed to update post" },
      { status: 500 }
    );
  }
}
```

#### 4. Publish Post: `app/api/posts/[id]/publish/route.ts`

```typescript
export async function PUT(
  request: NextRequest,
  { params }: { params: { id: string } }
) {
  try {
    const session = await getServerSession(authOptions);

    if (!session || !session.user) {
      return NextResponse.json({ error: "Not authenticated" }, { status: 401 });
    }

    const body = await request.json();

    const response = await serverApi.request(
      `/api/posts/${params.id}/publish`,
      {
        method: "PUT",
        body,
        headers: {
          Authorization: `Bearer ${session.accessToken}`,
        },
      }
    );

    if (response.error) {
      return NextResponse.json(
        { error: response.error },
        { status: response.status }
      );
    }

    return NextResponse.json({
      success: true,
      data: response.data,
    });
  } catch (error) {
    return NextResponse.json(
      { error: "Failed to publish post" },
      { status: 500 }
    );
  }
}
```

---

## API Request/Response Examples

### Example 1: Create Draft with Media

**Frontend Request:**

```typescript
const formData = new FormData();
formData.append("content", "Check out this amazing product!");
formData.append(
  "content_html",
  "<p>Check out this <strong>amazing</strong> product!</p>"
);
formData.append("category", "Marketing");
formData.append("platforms", JSON.stringify(["facebook", "twitter"]));
formData.append("files[]", file1);
formData.append("files[]", file2);

const response = await fetch("/api/posts", {
  method: "POST",
  body: formData,
});
```

**Laravel Receives:**

- `content`: "Check out this amazing product!"
- `content_html`: "<p>Check out this <strong>amazing</strong> product!</p>"
- `category`: "Marketing"
- `platforms`: '["facebook", "twitter"]'
- `files[]`: [File objects]

**Laravel Response:**

```json
{
  "success": true,
  "message": "Post created successfully",
  "data": {
    "id": 123,
    "status": "draft",
    "media": [
      {
        "id": 1,
        "file_url": "http://social-platform/storage/posts/123/image1.jpg"
      }
    ]
  }
}
```

### Example 2: Get Posts List with Filter

**Frontend Request:**

```typescript
const response = await fetch("/api/posts?status=draft&page=1&per_page=10");
```

**Laravel Response:**

```json
{
  "success": true,
  "data": [
    {
      "id": 123,
      "content": "Draft post 1",
      "status": "draft",
      "media_count": 2
    },
    {
      "id": 124,
      "content": "Draft post 2",
      "status": "draft",
      "media_count": 0
    }
  ],
  "meta": {
    "current_page": 1,
    "total": 15,
    "last_page": 2
  }
}
```

### Example 3: Publish Draft Post

**Frontend Request:**

```typescript
const response = await fetch(`/api/posts/${postId}/publish`, {
  method: "PUT",
  headers: { "Content-Type": "application/json" },
  body: JSON.stringify({
    platforms: ["facebook", "twitter"],
  }),
});
```

**Laravel Response:**

```json
{
  "success": true,
  "message": "Post published successfully",
  "data": {
    "id": 123,
    "status": "posted",
    "published_at": "2024-01-15T12:30:00.000000Z"
  }
}
```

---

## Error Handling

### Standard Error Response Format

```json
{
  "success": false,
  "error": "Error message here",
  "errors": {
    "field_name": ["Validation error message"]
  }
}
```

### HTTP Status Codes

- `200 OK`: Successful GET, PUT, DELETE
- `201 Created`: Successful POST (resource created)
- `400 Bad Request`: Validation errors, invalid request
- `401 Unauthorized`: Missing or invalid authentication
- `403 Forbidden`: Authenticated but not authorized (e.g., trying to edit someone else's post)
- `404 Not Found`: Resource not found
- `422 Unprocessable Entity`: Validation errors (Laravel standard)
- `500 Internal Server Error`: Server error

### Common Error Scenarios

1. **Trying to edit a published post:**

   ```json
   {
     "success": false,
     "error": "Cannot update post with status 'posted'. Only draft posts can be updated."
   }
   ```

2. **Post not found:**

   ```json
   {
     "success": false,
     "error": "Post not found"
   }
   ```

3. **Unauthorized access:**

   ```json
   {
     "success": false,
     "error": "You don't have permission to access this post"
   }
   ```

4. **Validation errors:**
   ```json
   {
     "success": false,
     "error": "Validation failed",
     "errors": {
       "content": ["The content field is required."],
       "user_id": ["The user id field is required."]
     }
   }
   ```

---

## Implementation Checklist

### Phase 1: Database Setup (Laravel)

- [ ] Create `posts` table migration
- [ ] Create `post_media` table migration
- [ ] Add foreign key constraints
- [ ] Add indexes for performance
- [ ] Run migrations

### Phase 2: Laravel Models & Relationships

- [ ] Create `Post` model
- [ ] Create `PostMedia` model
- [ ] Define relationships:
  - [ ] Post belongsTo User
  - [ ] Post hasMany PostMedia
  - [ ] PostMedia belongsTo Post
  - [ ] PostMedia belongsTo User
- [ ] Add fillable fields and casts

### Phase 3: Laravel Controllers

- [ ] Create `PostController` with:
  - [ ] `store()` - Create draft post
  - [ ] `index()` - List posts with filters
  - [ ] `show()` - Get single post
  - [ ] `update()` - Update draft post (with status check)
  - [ ] `publish()` - Change status to posted
  - [ ] `destroy()` - Delete post
- [ ] Create `PostMediaController` with:
  - [ ] `store()` - Upload media
  - [ ] `destroy()` - Delete media

### Phase 4: Laravel API Routes

- [ ] Define routes in `routes/api.php`:
  ```php
  Route::middleware('auth:sanctum')->group(function () {
    Route::apiResource('posts', PostController::class);
    Route::put('posts/{id}/publish', [PostController::class, 'publish']);
    Route::post('posts/{id}/media', [PostMediaController::class, 'store']);
    Route::delete('posts/{id}/media/{media_id}', [PostMediaController::class, 'destroy']);
  });
  ```

### Phase 5: File Storage Setup

- [ ] Configure storage disk (local or S3)
- [ ] Create storage directory structure
- [ ] Set up public URL generation
- [ ] Implement file upload handling
- [ ] Implement file deletion on post/media delete

### Phase 6: Frontend - Next.js API Routes

- [ ] Create `app/api/posts/route.ts` (POST, GET)
- [ ] Create `app/api/posts/[id]/route.ts` (GET, PUT, DELETE)
- [ ] Create `app/api/posts/[id]/publish/route.ts` (PUT)
- [ ] Create `app/api/posts/[id]/media/route.ts` (POST, DELETE)

### Phase 7: Frontend - Post List Page

- [ ] Create `app/posts/page.tsx`
- [ ] Implement post list display
- [ ] Add status filter (draft/posted)
- [ ] Add "Edit" button (only for drafts)
- [ ] Add "Publish" button (only for drafts)
- [ ] Add "Delete" button
- [ ] Display media thumbnails

### Phase 8: Frontend - Save Draft Functionality

- [ ] Update `app/compose/page.tsx`
- [ ] Add `handleSaveDraft()` function
- [ ] Connect "Save as Draft" button
- [ ] Handle media uploads
- [ ] Show success/error messages

### Phase 9: Frontend - Edit Draft Functionality

- [ ] Add "Edit" button handler in post list
- [ ] Load draft post data into composer
- [ ] Pre-fill form fields
- [ ] Load existing media
- [ ] Handle update submission

### Phase 10: Frontend - Publish Functionality

- [ ] Update `handlePublish()` in compose page
- [ ] Call publish API endpoint
- [ ] Update post status in list
- [ ] Show success message

### Phase 11: Testing

- [ ] Test creating draft with media
- [ ] Test creating draft without media
- [ ] Test updating draft
- [ ] Test publishing draft
- [ ] Test editing published post (should fail)
- [ ] Test deleting post
- [ ] Test media upload/delete
- [ ] Test post list filtering
- [ ] Test pagination
- [ ] Test authorization (user can only access own posts)

---

## Summary

This documentation provides:

1. ✅ **Unified table design** - Single `posts` table with `status` field
2. ✅ **Separate media storage** - `post_media` table with proper relationships
3. ✅ **Complete API specifications** - All endpoints with request/response examples
4. ✅ **Status-based editing** - Only drafts can be edited
5. ✅ **Workflow documentation** - Complete flow from creation to publishing
6. ✅ **Frontend integration guide** - Next.js API routes as proxy
7. ✅ **Error handling** - Standard error responses
8. ✅ **Implementation checklist** - Step-by-step guide

**Next Steps:**

1. Implement Laravel backend APIs according to this specification
2. Create Next.js API routes to proxy requests
3. Build frontend post list page
4. Integrate save draft and publish functionality
5. Test the complete flow

---

**Note:** Social media integration (Facebook, Twitter, etc.) is intentionally excluded from this phase. Focus on storage and management first, then add social media publishing later.
