# Posts & Drafts API Documentation

**For Next.js Frontend Integration**

This document provides complete API reference for the Posts and Drafts system. All endpoints require authentication using Laravel Sanctum Bearer tokens.

---

## 📋 Table of Contents

1. [Base Configuration](#base-configuration)
2. [Authentication](#authentication)
3. [Posts Endpoints](#posts-endpoints)
4. [Media Endpoints](#media-endpoints)
5. [Response Formats](#response-formats)
6. [Error Handling](#error-handling)
7. [Next.js Integration Examples](#nextjs-integration-examples)

---

## Base Configuration

### Base URL

```
http://social-platform/api
```

**Note:** Update this URL based on your Laravel backend configuration.

---

## Authentication

All endpoints (except public auth routes) require authentication using Laravel Sanctum Bearer tokens.

### Headers Required

```
Authorization: Bearer {your_token_here}
Content-Type: application/json
```

For file uploads:

```
Authorization: Bearer {your_token_here}
Content-Type: multipart/form-data
```

### Getting Authentication Token

Use the login endpoint to get your token:

```typescript
POST /api/login
Content-Type: application/json

{
  "email": "user@example.com",
  "password": "password123",
  "remember_me": false
}
```

**Response:**

```json
{
    "success": true,
    "message": "Login successful.",
    "data": {
        "token": "1|abc123def456...",
        "token_expires_at": "2024-12-20T10:30:00Z",
        "user": {
            "id": 1,
            "name": "John Doe",
            "email": "user@example.com"
        }
    }
}
```

---

## Posts Endpoints

### 1. Create Post (Draft)

Creates a new draft post. Can include media files.

**Endpoint:** `POST /api/posts`

**Headers:**

```
Authorization: Bearer {token}
Content-Type: multipart/form-data (if files) or application/json
```

**Request Body (JSON - without files):**

```json
{
    "content": "This is my post content",
    "content_html": "<p>This is my <strong>post</strong> content</p>",
    "category": "Marketing",
    "platforms": ["facebook", "twitter"]
}
```

**Request Body (Form Data - with files):**

```
content: "This is my post content"
content_html: "<p>This is my post content</p>"
category: Marketing
platforms: ["facebook", "twitter"]
files[]: [File 1]
files[]: [File 2]
```

**Field Descriptions:**

-   `content` (required): Plain text content of the post
-   `content_html` (optional): HTML formatted content
-   `category` (optional): Post category (default: "General")
-   `platforms` (optional): JSON array or string of platform names
-   `files[]` (optional): Array of files (images/videos), max 10MB per file

**Response (201 Created):**

```json
{
    "success": true,
    "message": "Post created successfully",
    "data": {
        "id": 123,
        "app_user_id": 1,
        "content": "This is my post content",
        "content_html": "<p>This is my post content</p>",
        "status": "draft",
        "category": "Marketing",
        "platforms": ["facebook", "twitter"],
        "published_at": null,
        "created_at": "2024-01-15T10:30:00.000000Z",
        "updated_at": "2024-01-15T10:30:00.000000Z",
        "media": [
            {
                "id": 1,
                "post_id": 123,
                "app_user_id": 1,
                "file_name": "image1.jpg",
                "file_url": "http://social-platform/storage/posts/123/image1.jpg",
                "file_type": "image/jpeg",
                "file_size": 245678,
                "mime_type": "image/jpeg",
                "display_order": 0,
                "created_at": "2024-01-15T10:30:00.000000Z"
            }
        ]
    }
}
```

---

### 2. Get All Posts (List)

Retrieves all posts for the authenticated user with optional filtering and pagination.

**Endpoint:** `GET /api/posts`

**Headers:**

```
Authorization: Bearer {token}
```

**Query Parameters:**

-   `status` (optional): Filter by status - `draft` or `posted`
-   `category` (optional): Filter by category name
-   `page` (optional): Page number (default: 1)
-   `per_page` (optional): Items per page (default: 20, max: 100)
-   `sort` (optional): Sort field - `created_at`, `updated_at`, or `published_at` (default: `created_at`)
-   `order` (optional): Sort order - `asc` or `desc` (default: `desc`)

**Example Request:**

```
GET /api/posts?status=draft&page=1&per_page=20&sort=updated_at&order=desc
```

**Response (200 OK):**

```json
{
    "success": true,
    "data": [
        {
            "id": 123,
            "app_user_id": 1,
            "content": "This is a draft post",
            "content_html": "<p>This is a draft post</p>",
            "status": "draft",
            "category": "Marketing",
            "platforms": ["facebook"],
            "published_at": null,
            "created_at": "2024-01-15T10:30:00.000000Z",
            "updated_at": "2024-01-15T11:45:00.000000Z",
            "media": [
                {
                    "id": 1,
                    "file_name": "image1.jpg",
                    "file_url": "http://social-platform/storage/posts/123/image1.jpg",
                    "file_type": "image/jpeg",
                    "file_size": 245678,
                    "display_order": 0
                }
            ],
            "media_count": 1
        },
        {
            "id": 122,
            "app_user_id": 1,
            "content": "This is a published post",
            "status": "posted",
            "category": "News",
            "platforms": ["facebook", "twitter"],
            "published_at": "2024-01-14T15:20:00.000000Z",
            "created_at": "2024-01-14T14:00:00.000000Z",
            "updated_at": "2024-01-14T15:20:00.000000Z",
            "media": [],
            "media_count": 0
        }
    ],
    "meta": {
        "current_page": 1,
        "per_page": 20,
        "total": 45,
        "last_page": 3,
        "from": 1,
        "to": 20
    }
}
```

---

### 3. Get Single Post

Retrieves a single post by ID with all associated media.

**Endpoint:** `GET /api/posts/{id}`

**Headers:**

```
Authorization: Bearer {token}
```

**URL Parameters:**

-   `id`: Post ID (integer)

**Response (200 OK):**

```json
{
    "success": true,
    "data": {
        "id": 123,
        "app_user_id": 1,
        "content": "This is my post content",
        "content_html": "<p>This is my post content</p>",
        "status": "draft",
        "category": "Marketing",
        "platforms": ["facebook", "twitter"],
        "published_at": null,
        "created_at": "2024-01-15T10:30:00.000000Z",
        "updated_at": "2024-01-15T10:30:00.000000Z",
        "media": [
            {
                "id": 1,
                "post_id": 123,
                "app_user_id": 1,
                "file_name": "image1.jpg",
                "file_path": "posts/123/image1.jpg",
                "file_url": "http://social-platform/storage/posts/123/image1.jpg",
                "file_type": "image/jpeg",
                "file_size": 245678,
                "mime_type": "image/jpeg",
                "display_order": 0,
                "created_at": "2024-01-15T10:30:00.000000Z"
            }
        ]
    }
}
```

**Error Responses:**

-   `403 Forbidden`: User doesn't own this post
-   `404 Not Found`: Post not found

---

### 4. Update Post (Draft Only)

Updates a draft post. **Only works if post status is 'draft'**.

**Endpoint:** `PUT /api/posts/{id}`

**Headers:**

```
Authorization: Bearer {token}
Content-Type: multipart/form-data (if files) or application/json
```

**URL Parameters:**

-   `id`: Post ID (integer)

**Request Body (JSON):**

```json
{
    "content": "Updated post content",
    "content_html": "<p>Updated post content</p>",
    "category": "News",
    "platforms": ["facebook", "twitter", "linkedin"]
}
```

**Request Body (Form Data - with media changes):**

```
content: "Updated post content"
content_html: "<p>Updated post content</p>"
category: News
platforms: ["facebook", "twitter"]
files[]: [New File 1]
files[]: [New File 2]
remove_media_ids: [1, 2]
```

**Field Descriptions:**

-   `content` (optional): Updated text content
-   `content_html` (optional): Updated HTML content
-   `category` (optional): Updated category
-   `platforms` (optional): Updated platforms array
-   `files[]` (optional): New files to add
-   `remove_media_ids` (optional): Array of media IDs to remove

**Response (200 OK):**

```json
{
  "success": true,
  "message": "Post updated successfully",
  "data": {
    "id": 123,
    "app_user_id": 1,
    "content": "Updated post content",
    "content_html": "<p>Updated post content</p>",
    "status": "draft",
    "category": "News",
    "platforms": ["facebook", "twitter", "linkedin"],
    "published_at": null,
    "created_at": "2024-01-15T10:30:00.000000Z",
    "updated_at": "2024-01-15T12:00:00.000000Z",
    "media": [...]
  }
}
```

**Error Responses:**

-   `400 Bad Request`: Post is not a draft (already published)
-   `403 Forbidden`: User doesn't own this post
-   `404 Not Found`: Post not found

---

### 5. Publish Post

Changes post status from 'draft' to 'posted' and sets published_at timestamp.

**Endpoint:** `PUT /api/posts/{id}/publish`

**Headers:**

```
Authorization: Bearer {token}
Content-Type: application/json
```

**URL Parameters:**

-   `id`: Post ID (integer)

**Request Body (optional):**

```json
{
    "platforms": ["facebook", "twitter"]
}
```

**Response (200 OK):**

```json
{
  "success": true,
  "message": "Post published successfully",
  "data": {
    "id": 123,
    "app_user_id": 1,
    "content": "This is my post content",
    "content_html": "<p>This is my post content</p>",
    "status": "posted",
    "category": "Marketing",
    "platforms": ["facebook", "twitter"],
    "published_at": "2024-01-15T12:30:00.000000Z",
    "created_at": "2024-01-15T10:30:00.000000Z",
    "updated_at": "2024-01-15T12:30:00.000000Z",
    "media": [...]
  }
}
```

**Error Responses:**

-   `400 Bad Request`: Post is already published or not a draft
-   `403 Forbidden`: User doesn't own this post
-   `404 Not Found`: Post not found

---

### 6. Delete Post

Deletes a post and all associated media files. Works for both drafts and published posts.

**Endpoint:** `DELETE /api/posts/{id}`

**Headers:**

```
Authorization: Bearer {token}
```

**URL Parameters:**

-   `id`: Post ID (integer)

**Response (200 OK):**

```json
{
    "success": true,
    "message": "Post deleted successfully"
}
```

**Error Responses:**

-   `403 Forbidden`: User doesn't own this post
-   `404 Not Found`: Post not found

---

## Media Endpoints

### 7. Upload Media to Post

Adds media files to an existing post. **Only allowed for draft posts**.

**Endpoint:** `POST /api/posts/{id}/media`

**Headers:**

```
Authorization: Bearer {token}
Content-Type: multipart/form-data
```

**URL Parameters:**

-   `id`: Post ID (integer)

**Request Body (Form Data):**

```
files[]: [File 1]
files[]: [File 2]
display_order: 0
```

**Field Descriptions:**

-   `files[]` (required): Array of files to upload (max 10MB per file)
-   `display_order` (optional): Starting display order number

**Response (201 Created):**

```json
{
    "success": true,
    "message": "Media uploaded successfully",
    "data": [
        {
            "id": 3,
            "post_id": 123,
            "app_user_id": 1,
            "file_name": "new_image.jpg",
            "file_url": "http://social-platform/storage/posts/123/new_image.jpg",
            "file_type": "image/jpeg",
            "file_size": 345678,
            "mime_type": "image/jpeg",
            "display_order": 0,
            "created_at": "2024-01-15T13:00:00.000000Z"
        }
    ]
}
```

**Error Responses:**

-   `400 Bad Request`: Post is not a draft
-   `403 Forbidden`: User doesn't own this post
-   `404 Not Found`: Post not found
-   `422 Validation Error`: Invalid file or validation failed

---

### 8. Delete Media from Post

Removes a media file from a post. **Only allowed for draft posts**.

**Endpoint:** `DELETE /api/posts/{id}/media/{media_id}`

**Headers:**

```
Authorization: Bearer {token}
```

**URL Parameters:**

-   `id`: Post ID (integer)
-   `media_id`: Media ID (integer)

**Response (200 OK):**

```json
{
    "success": true,
    "message": "Media deleted successfully"
}
```

**Error Responses:**

-   `400 Bad Request`: Post is not a draft
-   `403 Forbidden`: User doesn't own this post
-   `404 Not Found`: Post or media not found

---

## Response Formats

### Success Response Structure

All successful responses follow this structure:

```json
{
  "success": true,
  "message": "Operation successful message",
  "data": { ... }
}
```

For paginated lists:

```json
{
  "success": true,
  "data": [...],
  "meta": {
    "current_page": 1,
    "per_page": 20,
    "total": 45,
    "last_page": 3,
    "from": 1,
    "to": 20
  }
}
```

### Error Response Structure

```json
{
    "success": false,
    "error": "Error message here",
    "errors": {
        "field_name": ["Validation error message"]
    }
}
```

---

## Error Handling

### HTTP Status Codes

-   `200 OK`: Successful GET, PUT, DELETE
-   `201 Created`: Successful POST (resource created)
-   `400 Bad Request`: Validation errors, invalid request, business logic errors
-   `401 Unauthorized`: Missing or invalid authentication token
-   `403 Forbidden`: Authenticated but not authorized (e.g., trying to access someone else's post)
-   `404 Not Found`: Resource not found
-   `422 Unprocessable Entity`: Validation errors (Laravel standard)
-   `500 Internal Server Error`: Server error

### Common Error Scenarios

#### 1. Trying to edit a published post:

```json
{
    "success": false,
    "error": "Cannot update post with status 'posted'. Only draft posts can be updated."
}
```

#### 2. Post not found:

```json
{
    "success": false,
    "error": "Post not found"
}
```

#### 3. Unauthorized access:

```json
{
    "success": false,
    "error": "You don't have permission to access this post"
}
```

#### 4. Validation errors:

```json
{
    "success": false,
    "error": "Validation failed",
    "errors": {
        "content": ["The content field is required."],
        "files.0": ["The file must not be greater than 10240 kilobytes."]
    }
}
```

---

## Next.js Integration Examples

### Example 1: Create Draft with Media

```typescript
// app/api/posts/route.ts (Next.js API Route)
import { NextRequest, NextResponse } from "next/server";
import { getServerSession } from "next-auth/next";
import { authOptions } from "@/lib/authOptions";

export async function POST(request: NextRequest) {
    try {
        const session = await getServerSession(authOptions);

        if (!session || !session.user) {
            return NextResponse.json(
                { error: "Not authenticated" },
                { status: 401 }
            );
        }

        const formData = await request.formData();

        // Forward to Laravel backend
        const response = await fetch(
            `${process.env.LARAVEL_API_URL}/api/posts`,
            {
                method: "POST",
                headers: {
                    Authorization: `Bearer ${session.accessToken}`,
                },
                body: formData,
            }
        );

        const data = await response.json();

        if (!response.ok) {
            return NextResponse.json(data, { status: response.status });
        }

        return NextResponse.json(data);
    } catch (error) {
        return NextResponse.json(
            { error: "Failed to create post" },
            { status: 500 }
        );
    }
}
```

### Example 2: Get Posts List

```typescript
// app/api/posts/route.ts (GET handler)
export async function GET(request: NextRequest) {
    try {
        const session = await getServerSession(authOptions);

        if (!session || !session.user) {
            return NextResponse.json(
                { error: "Not authenticated" },
                { status: 401 }
            );
        }

        const { searchParams } = new URL(request.url);
        const params = new URLSearchParams();

        // Forward query parameters
        searchParams.forEach((value, key) => {
            params.append(key, value);
        });

        const response = await fetch(
            `${process.env.LARAVEL_API_URL}/api/posts?${params.toString()}`,
            {
                method: "GET",
                headers: {
                    Authorization: `Bearer ${session.accessToken}`,
                },
            }
        );

        const data = await response.json();

        if (!response.ok) {
            return NextResponse.json(data, { status: response.status });
        }

        return NextResponse.json(data);
    } catch (error) {
        return NextResponse.json(
            { error: "Failed to fetch posts" },
            { status: 500 }
        );
    }
}
```

### Example 3: Update Draft Post

```typescript
// app/api/posts/[id]/route.ts
export async function PUT(
    request: NextRequest,
    { params }: { params: { id: string } }
) {
    try {
        const session = await getServerSession(authOptions);

        if (!session || !session.user) {
            return NextResponse.json(
                { error: "Not authenticated" },
                { status: 401 }
            );
        }

        const formData = await request.formData();

        const response = await fetch(
            `${process.env.LARAVEL_API_URL}/api/posts/${params.id}`,
            {
                method: "PUT",
                headers: {
                    Authorization: `Bearer ${session.accessToken}`,
                },
                body: formData,
            }
        );

        const data = await response.json();

        if (!response.ok) {
            return NextResponse.json(data, { status: response.status });
        }

        return NextResponse.json(data);
    } catch (error) {
        return NextResponse.json(
            { error: "Failed to update post" },
            { status: 500 }
        );
    }
}
```

### Example 4: Publish Post

```typescript
// app/api/posts/[id]/publish/route.ts
export async function PUT(
    request: NextRequest,
    { params }: { params: { id: string } }
) {
    try {
        const session = await getServerSession(authOptions);

        if (!session || !session.user) {
            return NextResponse.json(
                { error: "Not authenticated" },
                { status: 401 }
            );
        }

        const body = await request.json();

        const response = await fetch(
            `${process.env.LARAVEL_API_URL}/api/posts/${params.id}/publish`,
            {
                method: "PUT",
                headers: {
                    Authorization: `Bearer ${session.accessToken}`,
                    "Content-Type": "application/json",
                },
                body: JSON.stringify(body),
            }
        );

        const data = await response.json();

        if (!response.ok) {
            return NextResponse.json(data, { status: response.status });
        }

        return NextResponse.json(data);
    } catch (error) {
        return NextResponse.json(
            { error: "Failed to publish post" },
            { status: 500 }
        );
    }
}
```

### Example 5: Frontend Component - Create Draft

```typescript
// components/PostComposer.tsx
"use client";

import { useState } from "react";

export default function PostComposer() {
    const [content, setContent] = useState("");
    const [files, setFiles] = useState<File[]>([]);
    const [loading, setLoading] = useState(false);

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        setLoading(true);

        try {
            const formData = new FormData();
            formData.append("content", content);
            formData.append("content_html", `<p>${content}</p>`);
            formData.append("category", "General");
            formData.append(
                "platforms",
                JSON.stringify(["facebook", "twitter"])
            );

            files.forEach((file) => {
                formData.append("files[]", file);
            });

            const response = await fetch("/api/posts", {
                method: "POST",
                body: formData,
            });

            const data = await response.json();

            if (!response.ok) {
                throw new Error(data.error || "Failed to create post");
            }

            alert("Draft saved successfully!");
            // Reset form or redirect
        } catch (error) {
            alert(error.message);
        } finally {
            setLoading(false);
        }
    };

    return (
        <form onSubmit={handleSubmit}>
            <textarea
                value={content}
                onChange={(e) => setContent(e.target.value)}
                placeholder="Write your post..."
                required
            />
            <input
                type="file"
                multiple
                onChange={(e) => setFiles(Array.from(e.target.files || []))}
            />
            <button type="submit" disabled={loading}>
                {loading ? "Saving..." : "Save as Draft"}
            </button>
        </form>
    );
}
```

### Example 6: Frontend Component - Posts List

```typescript
// app/posts/page.tsx
"use client";

import { useEffect, useState } from "react";

interface Post {
    id: number;
    content: string;
    status: "draft" | "posted";
    category: string;
    media: Array<{
        id: number;
        file_url: string;
        file_name: string;
    }>;
    created_at: string;
}

export default function PostsPage() {
    const [posts, setPosts] = useState<Post[]>([]);
    const [loading, setLoading] = useState(true);
    const [statusFilter, setStatusFilter] = useState<string>("");

    useEffect(() => {
        fetchPosts();
    }, [statusFilter]);

    const fetchPosts = async () => {
        try {
            const params = new URLSearchParams();
            if (statusFilter) params.append("status", statusFilter);
            params.append("sort", "updated_at");
            params.append("order", "desc");

            const response = await fetch(`/api/posts?${params.toString()}`);
            const data = await response.json();

            if (data.success) {
                setPosts(data.data);
            }
        } catch (error) {
            console.error("Failed to fetch posts:", error);
        } finally {
            setLoading(false);
        }
    };

    const handlePublish = async (postId: number) => {
        try {
            const response = await fetch(`/api/posts/${postId}/publish`, {
                method: "PUT",
                headers: { "Content-Type": "application/json" },
                body: JSON.stringify({ platforms: ["facebook", "twitter"] }),
            });

            const data = await response.json();

            if (data.success) {
                alert("Post published successfully!");
                fetchPosts();
            }
        } catch (error) {
            alert("Failed to publish post");
        }
    };

    if (loading) return <div>Loading...</div>;

    return (
        <div>
            <select
                value={statusFilter}
                onChange={(e) => setStatusFilter(e.target.value)}
            >
                <option value="">All Posts</option>
                <option value="draft">Drafts</option>
                <option value="posted">Published</option>
            </select>

            {posts.map((post) => (
                <div key={post.id}>
                    <p>{post.content}</p>
                    <p>Status: {post.status}</p>
                    <p>Category: {post.category}</p>
                    {post.status === "draft" && (
                        <>
                            <button onClick={() => handlePublish(post.id)}>
                                Publish
                            </button>
                            <button>Edit</button>
                        </>
                    )}
                    <button>Delete</button>
                </div>
            ))}
        </div>
    );
}
```

---

## Important Notes

### Status Rules

| Action       | Draft Post | Posted Post            |
| ------------ | ---------- | ---------------------- |
| View         | ✅         | ✅                     |
| Edit         | ✅         | ❌                     |
| Delete       | ✅         | ✅                     |
| Publish      | ✅         | ❌ (already published) |
| Add Media    | ✅         | ❌                     |
| Remove Media | ✅         | ❌                     |

### Field Notes

1. **`app_user_id`**: This field is automatically set from the authenticated user. You don't need to send it in requests.

2. **`platforms`**: Can be sent as:

    - JSON string: `"[\"facebook\", \"twitter\"]"`
    - JSON array: `["facebook", "twitter"]`
    - Will be normalized to array format

3. **File Uploads**:

    - Max file size: 10MB per file
    - Supported formats: Images and videos (based on MIME type)
    - Files are stored in `storage/app/public/posts/{post_id}/`

4. **Pagination**:

    - Default: 20 items per page
    - Maximum: 100 items per page
    - Use `page` and `per_page` query parameters

5. **Sorting**:
    - Available fields: `created_at`, `updated_at`, `published_at`
    - Default: `created_at` descending

---

## Quick Reference

### Endpoints Summary

| Method | Endpoint                           | Description            |
| ------ | ---------------------------------- | ---------------------- |
| POST   | `/api/posts`                       | Create draft post      |
| GET    | `/api/posts`                       | List all posts         |
| GET    | `/api/posts/{id}`                  | Get single post        |
| PUT    | `/api/posts/{id}`                  | Update draft post      |
| PUT    | `/api/posts/{id}/publish`          | Publish draft          |
| DELETE | `/api/posts/{id}`                  | Delete post            |
| POST   | `/api/posts/{id}/media`            | Upload media to post   |
| DELETE | `/api/posts/{id}/media/{media_id}` | Delete media from post |

---

**Last Updated:** Based on implementation with `app_user_id` field structure.

**For questions or issues, refer to the Laravel backend code or contact the backend team.**
