# Postman API Testing Guide

## Fixing the 404 Error

### Issue 1: API Routes Not Registered
✅ **Fixed**: Added API routes to `bootstrap/app.php`

### Issue 2: Wrong URL Format
❌ **Wrong**: `http://social-platform/api/register`
✅ **Correct**: `http://localhost:8000/api/register` (or your server URL)

### Issue 3: Wrong Request Format
❌ **Wrong**: Using query parameters (`?email=...&name=...`)
✅ **Correct**: Use JSON body for POST requests

---

## How to Test Register Endpoint in Postman

### Step 1: Configure Request

1. **Method**: Select `POST`
2. **URL**: `http://localhost:8000/api/register`
   - Replace `localhost:8000` with your actual server URL
   - If using Laravel's built-in server: `http://localhost:8000`
   - If using a different port: `http://localhost:PORT`

### Step 2: Set Headers

Go to **Headers** tab and add:
```
Content-Type: application/json
Accept: application/json
```

### Step 3: Set Request Body

1. Go to **Body** tab
2. Select **raw**
3. Select **JSON** from dropdown
4. Enter JSON body:

```json
{
    "name": "Ganesh",
    "email": "ganesh@yopmail.com",
    "password": "123456"
}
```

### Step 4: Send Request

Click **Send** button.

---

## Expected Response

### Success Response (201 Created)
```json
{
    "success": true,
    "message": "User registered successfully.",
    "data": {
        "user": {
            "id": 1,
            "name": "Ganesh",
            "email": "ganesh@yopmail.com",
            "timezone": "UTC",
            "status": "active"
        }
    }
}
```

### Validation Error Response (422 Unprocessable Entity)
```json
{
    "message": "The email has already been taken. (and 1 more error)",
    "errors": {
        "email": [
            "The email has already been taken."
        ],
        "password": [
            "The password must be at least 6 characters."
        ]
    }
}
```

---

## Testing Login Endpoint

### Request Configuration

1. **Method**: `POST`
2. **URL**: `http://localhost:8000/api/login`
3. **Headers**:
   ```
   Content-Type: application/json
   Accept: application/json
   ```
4. **Body** (raw JSON):
```json
{
    "email": "ganesh@yopmail.com",
    "password": "123456"
}
```

### Expected Response

```json
{
    "success": true,
    "message": "Login successful.",
    "data": {
        "token": "1|abcdef1234567890...",
        "user": {
            "id": 1,
            "name": "Ganesh",
            "email": "ganesh@yopmail.com",
            "timezone": "UTC",
            "status": "active"
        }
    }
}
```

---

## Testing Protected Routes

### Get User Profile

1. **Method**: `GET`
2. **URL**: `http://localhost:8000/api/user`
3. **Headers**:
   ```
   Authorization: Bearer {your_token_here}
   Accept: application/json
   ```
   - Replace `{your_token_here}` with the token from login response

### Example:
```
Authorization: Bearer 1|abcdef1234567890...
```

---

## Common Issues and Solutions

### 1. 404 Not Found
**Cause**: API routes not registered or wrong URL
**Solution**: 
- Check `bootstrap/app.php` has `api: __DIR__.'/../routes/api.php'`
- Verify URL is `http://localhost:8000/api/register` (not `http://social-platform/api/register`)
- Make sure Laravel server is running: `php artisan serve`

### 2. 500 Internal Server Error
**Cause**: Database migration not run
**Solution**: Run `php artisan migrate`

### 3. 422 Validation Error
**Cause**: Invalid or missing data
**Solution**: Check request body matches required format

### 4. 401 Unauthorized (for protected routes)
**Cause**: Missing or invalid token
**Solution**: 
- Login first to get token
- Add token to Authorization header: `Bearer {token}`

---

## Postman Collection Setup

### Environment Variables

Create a Postman environment with:
- `base_url`: `http://localhost:8000`
- `api_token`: (will be set after login)

### Pre-request Script (for Login)

```javascript
// Set base URL
pm.environment.set("base_url", "http://localhost:8000");
```

### Tests Script (for Login)

```javascript
// Save token to environment
if (pm.response.code === 200) {
    const jsonData = pm.response.json();
    if (jsonData.data && jsonData.data.token) {
        pm.environment.set("api_token", jsonData.data.token);
    }
}
```

### Using Environment Variables

- **URL**: `{{base_url}}/api/register`
- **Authorization Header**: `Bearer {{api_token}}`

---

## Quick Test Checklist

- [ ] Laravel server running (`php artisan serve`)
- [ ] Migration run (`php artisan migrate`)
- [ ] Correct URL format (`http://localhost:8000/api/register`)
- [ ] POST method selected
- [ ] Headers set (`Content-Type: application/json`)
- [ ] Body is JSON (not query parameters)
- [ ] Valid JSON format in body

---

## Example Postman Screenshot Setup

### Register Request:
```
POST http://localhost:8000/api/register

Headers:
Content-Type: application/json
Accept: application/json

Body (raw JSON):
{
    "name": "Ganesh",
    "email": "ganesh@yopmail.com",
    "password": "123456"
}
```

### Login Request:
```
POST http://localhost:8000/api/login

Headers:
Content-Type: application/json
Accept: application/json

Body (raw JSON):
{
    "email": "ganesh@yopmail.com",
    "password": "123456"
}
```

---

## Testing with cURL

### Register
```bash
curl -X POST http://localhost:8000/api/register \
  -H "Content-Type: application/json" \
  -H "Accept: application/json" \
  -d '{
    "name": "Ganesh",
    "email": "ganesh@yopmail.com",
    "password": "123456"
  }'
```

### Login
```bash
curl -X POST http://localhost:8000/api/login \
  -H "Content-Type: application/json" \
  -H "Accept: application/json" \
  -d '{
    "email": "ganesh@yopmail.com",
    "password": "123456"
  }'
```

